<?php

/**
 * Custom icons for the plugin.
 *
 * @package Delicious_Recipes
 * @subpackage Delicious_Recipes/Classes
 * @since 1.0.0
 */

if ( ! class_exists( 'Delicious_Recipes_SVG' ) ) {
	/**
	 * SVG ICONS CLASS
	 * Retrieve the SVG code for the specified icon. Based on a solution in Twenty Nineteen.
	 */
	class Delicious_Recipes_SVG {
		/**
		 * Returns svg icons.
		 *
		 * @return array
		 */
		public static function get_recipe_keys_icons() {
			return self::$recipe_keys_icons;
		}

		/**
		 * GET SVG CODE
		 * Get the SVG code for the specified icon
		 *
		 * @param string $icon Icon name.
		 * @param string $group Icon group.
		 * @param string $color Color.
		 */
		public static function get_svg( $icon, $group = 'recipe-keys', $color = '#1A1A1B' ) {
			if ( 'recipe-keys' === $group ) {
				$arr = self::$recipe_keys_icons;
			} elseif ( 'recipe-cats' === $group ) {
				$arr = self::$recipe_category_icons;
			} elseif ( 'dashboard' === $group ) {
				$arr = apply_filters( 'delicious_recipes_user_dashboard_svgs', self::$dashboard_svg_icons );
			} else {
				$arr = array();
			}
			if ( array_key_exists( $icon, $arr ) ) {
				$repl = '<svg class="svg-icon" aria-hidden="true" role="img" focusable="false" ';
				$svg  = preg_replace( '/^<svg /', $repl, trim( $arr[ $icon ] ) ); // Add extra attributes to SVG code.
				$svg  = str_replace( '#1A1A1B', $color, $svg );   // Replace the color.
				// $svg  = str_replace( '#', '%23', $svg );          // Urlencode hashes.
				$svg = preg_replace( "/([\n\t]+)/", ' ', $svg ); // Remove newlines & tabs.
				$svg = preg_replace( '/>\s*</', '><', $svg );    // Remove whitespace between SVG tags.
				return $svg;
			}
			return '';
		}

		/**
		 * ICON STORAGE
		 * Store the code for all SVGs in an array.
		 *
		 * @var array
		 * @since 1.5.1 Made private, use get_recipe_keys_icons();
		 */
		protected static $recipe_keys_icons = array(
			'vegetarian-meals'  => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs><style>.vegeterian-meals-circle{fill:#8fbc04;}.vegeterian-meals-path{fill:#fff;}</style></defs><g transform="translate(-1219 -2556)"><circle class="vegeterian-meals-circle" cx="19" cy="19" r="19" transform="translate(1219 2556)"/><path class="vegeterian-meals-path" d="M-158.562,677.785c.064,1.227-.807,2.049-1.97,2.671-.917.491-1.944.821-2.4,1.924a6.7,6.7,0,0,0-.267,4.213c.126.531.312,1.048.457,1.576.069.251.108.511.2.957a3.814,3.814,0,0,1-2.294-1.941,21.259,21.259,0,0,1-.792-2.854c-.28-1.2-.514-1.465-1.712-1.7a4.767,4.767,0,0,1-3.946-5.026.938.938,0,0,0-.266-.674,5.845,5.845,0,0,1-2.161-5.753,5.368,5.368,0,0,1,4.085-4.451,10.571,10.571,0,0,0,3.371-1.484,5.412,5.412,0,0,1,7.328,1.285.882.882,0,0,0,.518.284,3.229,3.229,0,0,1,2.057,4.868.854.854,0,0,0,.018.664,3.9,3.9,0,0,1-.492,4.116A8.971,8.971,0,0,1-158.562,677.785Zm-5.546,8.753c0-.231,0-.462,0-.693a16.288,16.288,0,0,1,1.158-7.414,6.982,6.982,0,0,0,.353-5.056.783.783,0,0,1,.152-.6,4.4,4.4,0,0,1,2.676-1.659c.511-.1,1.033-.135,1.618-.209-1.948-.911-3.391-.046-4.788,1.223a10.388,10.388,0,0,1,.827-5.4c-1.122,1.468-2.019,3.026-1.743,4.954.142.995.464,1.963.678,2.949a11.187,11.187,0,0,1,.133,1.155c-1.5-1.773-2.841-2.488-4.411-2.193a6.434,6.434,0,0,1,3.032,1.849,2.607,2.607,0,0,1,.557,3.237C-165.079,681.226-164.828,683.9-164.108,686.538Zm-4.729-18.917a5.109,5.109,0,0,0-1.757,2.615,6.887,6.887,0,0,0,1.042,5.537,1.7,1.7,0,0,1,.235,1.44,4.088,4.088,0,0,0,3.217,4.972c-1.966-1.591-3.369-3.356-2.1-6.01C-170.45,674.16-170.689,670.645-168.837,667.62Z" transform="translate(1402.478 1898.587)"/></g></svg>',

			'gluten-free'       => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>gluten-free</defs> <rect width="38" height="38" rx="19" fill="#F7BD31"/> <path d="M19.3906 6C20.0594 6.54483 20.5985 7.23178 20.9687 8.01093C21.3389 8.79007 21.531 9.64188 21.531 10.5045C21.531 11.3672 21.3389 12.219 20.9687 12.9981C20.5985 13.7773 20.0594 14.4642 19.3906 15.009C18.7192 14.4658 18.1777 13.7795 17.8057 13.0001C17.4338 12.2208 17.2407 11.3681 17.2407 10.5045C17.2407 9.64093 17.4338 8.78828 17.8057 8.00891C18.1777 7.22954 18.7192 6.54315 19.3906 6Z" fill="white"/> <path d="M19.6236 31.063C19.6167 31.0357 19.6117 31.008 19.6085 30.98C19.5506 29.9596 19.6714 28.9369 19.9655 27.958C20.1602 27.2838 20.5036 26.6619 20.9705 26.138C21.5254 25.5453 22.2303 25.1137 23.0105 24.889C23.4476 24.7591 23.8998 24.6868 24.3555 24.674C24.7813 24.6575 25.2076 24.6911 25.6255 24.774C25.6615 24.781 25.6976 24.792 25.7486 24.804C25.7546 24.974 25.7705 25.145 25.7645 25.315C25.7555 25.542 25.7376 25.769 25.7056 25.995C25.5792 27.035 25.1237 28.0073 24.4055 28.77C24.0527 29.1317 23.6579 29.45 23.2295 29.718C22.2523 30.3425 21.167 30.779 20.0295 31.005C19.9085 31.03 19.7865 31.047 19.6645 31.066C19.6508 31.0664 19.6371 31.0653 19.6236 31.063Z" fill="white"/> <path d="M25.7416 18.987C25.775 19.2804 25.775 19.5767 25.7416 19.87C25.7083 20.272 25.6309 20.6691 25.5105 21.0541C25.2273 21.9871 24.6655 22.8113 23.9006 23.4161C23.0211 24.1093 22.0184 24.6301 20.9455 24.951C20.5528 25.0742 20.1518 25.1691 19.7456 25.2351C19.7056 25.2421 19.6656 25.243 19.6116 25.25C19.6066 25.155 19.5985 25.071 19.5965 24.986C19.564 24.0774 19.671 23.1693 19.9136 22.293C20.0796 21.681 20.3568 21.1047 20.7315 20.5931C21.1001 20.1147 21.5669 19.7208 22.1005 19.438C22.7002 19.1134 23.3616 18.9189 24.0415 18.867C24.5817 18.817 25.1264 18.8476 25.6575 18.9581C25.6895 18.9701 25.7136 18.978 25.7416 18.987Z" fill="white"/> <path d="M19.1506 19.443C18.9056 19.394 18.6736 19.354 18.4446 19.3C17.3367 19.0478 16.2855 18.5917 15.3446 17.955C14.8598 17.6358 14.4271 17.2438 14.0616 16.793C13.5861 16.1786 13.262 15.4609 13.1156 14.698C13.0219 14.2477 12.9849 13.7875 13.0056 13.328C13.0056 13.276 13.0146 13.228 13.0206 13.161C13.1276 13.141 13.2326 13.116 13.3386 13.1C14.3773 12.9248 15.4446 13.0641 16.4036 13.5C17.4189 13.9557 18.2142 14.7927 18.6176 15.83C18.8507 16.4134 19.0076 17.0245 19.0846 17.648C19.1455 18.1181 19.1725 18.592 19.1656 19.066C19.1696 19.186 19.1586 19.301 19.1506 19.443Z" fill="white"/> <path d="M13.0366 24.796C13.2016 24.769 13.3696 24.734 13.5366 24.716C13.7439 24.6895 13.9526 24.6744 14.1616 24.671C14.8133 24.6493 15.4627 24.7595 16.0708 24.995C16.679 25.2304 17.2333 25.5861 17.7006 26.041C18.109 26.4596 18.4272 26.9575 18.6356 27.504C18.8608 28.0741 19.0127 28.6706 19.0876 29.279C19.1498 29.7604 19.1769 30.2457 19.1685 30.731C19.1685 30.842 19.1555 30.952 19.1475 31.078C19.0085 31.055 18.8796 31.038 18.7526 31.012C17.5695 30.7798 16.4429 30.3195 15.4355 29.657C14.9397 29.3413 14.4952 28.9514 14.1175 28.501C13.6122 27.8716 13.2688 27.1279 13.1175 26.335C13.0267 25.8884 12.9898 25.4324 13.0076 24.977C13.0081 24.9473 13.0101 24.9176 13.0135 24.888C13.0205 24.858 13.0276 24.833 13.0366 24.796Z" fill="white"/> <path d="M19.6165 19.434C19.6105 19.342 19.6055 19.261 19.6035 19.181C19.5662 18.2388 19.681 17.2967 19.9435 16.391C20.1147 15.7889 20.3976 15.2244 20.7776 14.727C21.2375 14.1523 21.8369 13.7048 22.5186 13.427C23.1205 13.1745 23.7658 13.0414 24.4185 13.035C24.8187 13.0251 25.2187 13.0586 25.6115 13.135L25.6985 13.155L25.7505 13.172C25.7565 13.327 25.7716 13.483 25.7686 13.638C25.7686 13.829 25.7506 14.019 25.7296 14.207C25.63 15.2728 25.1883 16.2779 24.4705 17.072C24.0879 17.4778 23.6527 17.8306 23.1765 18.121C22.1482 18.7653 21.0043 19.2031 19.8086 19.41L19.7526 19.417L19.6165 19.434Z" fill="white"/> <path d="M19.1446 25.25C19.0686 25.24 18.9946 25.234 18.9236 25.22C17.6658 24.9946 16.4676 24.5141 15.4026 23.808C14.9127 23.4954 14.4742 23.1088 14.1026 22.662C13.6077 22.0391 13.2702 21.306 13.1186 20.525C13.0241 20.0711 12.9869 19.6072 13.0076 19.144C13.0076 19.114 13.0076 19.085 13.0146 19.055C13.0216 19.025 13.0236 19.012 13.0326 18.978C13.1696 18.954 13.3076 18.923 13.4466 18.905C14.1964 18.8014 14.9596 18.8532 15.6886 19.057C16.3674 19.2488 16.9944 19.5906 17.5236 20.057C17.9497 20.437 18.2905 20.9029 18.5236 21.424C18.7966 22.0233 18.9802 22.6594 19.0686 23.312C19.1234 23.6551 19.1541 24.0016 19.1606 24.349C19.1606 24.629 19.1606 24.909 19.1606 25.189C19.1567 25.2097 19.1514 25.2301 19.1446 25.25Z" fill="white"/> <path d="M33.7071 6.70709L27.7071 12.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M11.7071 28.7071L6.70709 33.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> </svg> ',

			'paleo'             => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs><style>.paleo-circle{fill:#b4a560;}.paleo-path{fill:#fff;}</style></defs><g transform="translate(-1219 -3416)"><circle class="paleo-circle" cx="19" cy="19" r="19" transform="translate(1219 3416)"/><g transform="translate(1225.605 3425)"><path class="paleo-path" d="M-224.483-118.87a12.3,12.3,0,0,1,5.527-2.357,8.325,8.325,0,0,1,4.882.733,4.871,4.871,0,0,1,2.43,2.929,7.735,7.735,0,0,1,.285,4.091,5.656,5.656,0,0,1-3.883,4.35,14.1,14.1,0,0,1-5.511.651c-.99-.023-1.983.077-2.975.123-.1,0-.191.033-.287.034a13.174,13.174,0,0,0-3.932.78,7.1,7.1,0,0,1-8.841-4.56,8.315,8.315,0,0,1,.276-5.072c.35-1.049.834-2.052,1.19-3.1.4-1.166.758-2.347,1.052-3.542a5.539,5.539,0,0,1,1.59-3.1,2.83,2.83,0,0,1,4.231.33,4.377,4.377,0,0,1,.63,1.184,18.318,18.318,0,0,0,2.7,5.454C-224.892-119.609-224.707-119.252-224.483-118.87Zm-2.361,8.788c.59-.062,1.076-.074,1.546-.169,1.239-.252,2.461-.608,3.709-.8a18.047,18.047,0,0,0,6.241-1.854,12.416,12.416,0,0,0,2.231-1.647c1.642-1.495.767-4.106-.685-4.936a7.211,7.211,0,0,0-3.632-.986,10.593,10.593,0,0,0-7.322,2.956c-1.1,1.007-2.225,1.989-3.347,2.973a4.062,4.062,0,0,0-1.365,2.052,1.683,1.683,0,0,0,1.213,2.174A11.907,11.907,0,0,0-226.844-110.082Zm-6.448-13.441-.068-.011c-.044.18-.082.362-.132.54q-.814,2.917-1.63,5.833c-.283,1.018-.611,2.027-.819,3.06a3.418,3.418,0,0,0,.45,2.505,1.521,1.521,0,0,0,2.2.511,1.484,1.484,0,0,0,.236-2.337,3.06,3.06,0,0,1-1-2.64c.107-1.536.272-3.069.428-4.6C-233.528-121.618-233.4-122.57-233.292-123.524Zm2.532-3.792c-.5,0-.907.263-.923.589-.014.284.41.548.894.556s.932-.253.949-.566S-230.231-127.317-230.759-127.315Z" transform="translate(237.069 127.699)"/><path class="paleo-path" d="M-87.739-47.066A4.194,4.194,0,0,1-84.36-44.74a1.805,1.805,0,0,1-.4,2.052,5.846,5.846,0,0,1-5.962,1.678,1.7,1.7,0,0,1-1.233-1.533A3.987,3.987,0,0,1-87.739-47.066Zm-2.551,5.658c-1.039-1.643-.551-3.154.2-4.663a1.287,1.287,0,0,0-.336.255,4.008,4.008,0,0,0-1.041,3.285C-91.4-41.777-91.175-41.57-90.29-41.408Z" transform="translate(105.979 54.842)"/></g></g></svg>',

			'freezer-meals'     => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs><style>.freezer-meals-circle{fill:#06b2f7;}.freezer-meals-path{fill:#fff;}</style></defs><g transform="translate(-1391 -2616)"><circle class="freezer-meals-circle" cx="19" cy="19" r="19" transform="translate(1391 2616)"/><g transform="translate(1393.672 2624.837)"><path class="freezer-meals-path" d="M22.788.163H10.794A1.522,1.522,0,0,0,9.328,1.735V7.186H24.254V1.735A1.521,1.521,0,0,0,22.788.163ZM12.82,4.732a.511.511,0,0,1-1.017,0V2.672a.511.511,0,0,1,1.017,0Z" transform="translate(0 0)"/><path class="freezer-meals-path" d="M9.329,36.158A1.413,1.413,0,0,0,10.8,37.511H22.79a1.413,1.413,0,0,0,1.465-1.353V24.593H9.329ZM11.8,27.71a.517.517,0,1,1,1.017,0v2.728a.517.517,0,1,1-1.017,0Z" transform="translate(-0.001 -16.388)"/></g></g></svg>',

			'low-carb'          => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="37" viewBox="0 0 38 37"><defs><style>.low-carb-circle{fill:#a6846e;}.low-carb-path{fill:#fff;}</style></defs><g transform="translate(-1219 -2677)"><ellipse class="low-carb-circle" cx="19" cy="18.5" rx="19" ry="18.5" transform="translate(1219 2677)"/><g transform="translate(1834.406 2484.021)"><path class="low-carb-path" d="M-598.782,202.979a29.11,29.11,0,0,1,8.668,1.457,10.235,10.235,0,0,1,5.031,4.1,3.644,3.644,0,0,1,.52,1.769c.067,1.255-.076,2.523,0,3.777a4.349,4.349,0,0,1-2.155,3.938,8.076,8.076,0,0,1-4.536,1.515,32.081,32.081,0,0,1-4.259-.218,4.569,4.569,0,0,1-2.94-1.56c-.811-.878-1.646-1.737-2.409-2.656a5.206,5.206,0,0,0-3.793-1.837,17.946,17.946,0,0,1-2.187-.25,3.066,3.066,0,0,1-1.348-.6,1.752,1.752,0,0,1-.523-1.126c-.075-1.424-.009-2.856-.083-4.281a2.629,2.629,0,0,1,1.857-2.815,14.141,14.141,0,0,1,3.139-.85C-602.139,203.123-600.456,203.09-598.782,202.979Zm13.26,6.987a3.92,3.92,0,0,0-1.16-2.42,10.247,10.247,0,0,0-6.247-3.344,38.383,38.383,0,0,0-9.911-.418,9.005,9.005,0,0,0-3.915,1.06,11,11,0,0,0-1.047.741l.089.161c.2-.073.406-.149.611-.218a16.534,16.534,0,0,1,2.445-.8,30.992,30.992,0,0,1,12.314.286,9.946,9.946,0,0,1,6.185,4.126C-586,209.377-585.809,209.6-585.523,209.966Zm-.066.677a.809.809,0,0,0-.2.108c-2.037,2.3-4.723,2.808-7.607,2.725a5.907,5.907,0,0,1-4.587-2.135c-.336-.4-.733-.761-1.027-1.19a5.242,5.242,0,0,0-3.788-2.2,11.466,11.466,0,0,1-5.223-1.6,1.583,1.583,0,0,0,1.33,1.335,25.736,25.736,0,0,0,3.2.662,5.72,5.72,0,0,1,4.268,2.311,10.3,10.3,0,0,0,2.473,2.383,7.259,7.259,0,0,0,4.667.949,9.3,9.3,0,0,0,5.239-1.534A2.523,2.523,0,0,0-585.589,210.643Zm-5.87.8c.838.021,1.452-.448,1.47-1.122a1.832,1.832,0,0,0-1.875-1.579c-.842-.021-1.463.451-1.474,1.122A1.828,1.828,0,0,0-591.459,211.443Z"/></g></g></svg>',

			'slow-cooker'       => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="37" viewBox="0 0 38 37"><defs><style>.slow-cooker-circle{fill:#f08a0e;}.slow-cooker-path{fill:#fff;}</style></defs><g transform="translate(-1411 -3482)"><ellipse class="slow-cooker-circle" cx="19" cy="18.5" rx="19" ry="18.5" transform="translate(1411 3482)"/><g transform="translate(381.516 3015.043)"><path class="slow-cooker-path" d="M1039.484,544.335v-1.251a3.589,3.589,0,0,1,.713-.232,9.877,9.877,0,0,1,1.089-.01l.037-.679h15.091l.035.679a9.022,9.022,0,0,1,1.049.01,4.193,4.193,0,0,1,.754.232v1.251a1.93,1.93,0,0,1-.585.3,10.529,10.529,0,0,1-1.235.014v.5c0,2.181-.013,4.362.01,6.542,0,.439-.144.564-.548.528-.286-.025-.577,0-.874,0-.017.264-.031.48-.045.7h-1.738l-.049-.666h-8.647l-.043.659h-1.765v-.693c-.315,0-.605-.02-.892,0-.4.036-.553-.089-.548-.528.023-2.181.011-4.362.011-6.542v-.425c-.067-.036-.09-.059-.114-.06a9.258,9.258,0,0,1-1.122-.03A2.027,2.027,0,0,1,1039.484,544.335Zm10.248,5.053a.864.864,0,1,0-1.729,0,.864.864,0,0,0,1.729,0Z" transform="translate(0 -58.85)"/><path class="slow-cooker-path" d="M1060.414,482.38v-1.648c.743-.138,1.5-.3,2.257-.414s1.513-.194,2.272-.266c.267-.025.359-.122.344-.387a9.912,9.912,0,0,1,0-1.041.6.6,0,0,1,.683-.662q2.084-.009,4.168,0a.6.6,0,0,1,.676.669c.017.442,0,.885,0,1.317l4.875.741v1.691Zm5.276-2.447h4.724c0-.426.021-.855-.015-1.279a.432.432,0,0,0-.312-.281q-2.022-.029-4.045,0a.47.47,0,0,0-.336.308A12.43,12.43,0,0,0,1065.69,479.933Z" transform="translate(-19.184 0)"/></g></g></svg>',

			'dairy'             => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs><style>.dairy-circle{fill:#62bdc2;}.dairy-path{fill:#fff;}</style></defs><g transform="translate(-1219 -2736)"><circle class="dairy-circle" cx="19" cy="19" r="19" transform="translate(1219 2736)"/><g transform="translate(1814.985 2514.889)"><path class="dairy-path" d="M-579.2,243.665h3.987c-.2,1.972.9,3.49,1.555,5.165a10.283,10.283,0,0,1,.79,3.37c.1,3.492.04,6.988.035,10.483,0,1.363-.27,1.626-1.6,1.626-1.929,0-3.858,0-5.787,0-.987,0-1.335-.307-1.339-1.29-.015-3.8-.063-7.606.04-11.405a8.492,8.492,0,0,1,.869-3.016C-579.97,247.015-579,245.54-579.2,243.665Z" transform="translate(0 -12.909)"/><path class="dairy-path" d="M-567.31,229.907c-.475,0-.952-.035-1.423.008-.677.062-.919-.172-.933-.883-.014-.753.3-.9.947-.877q1.548.049,3.1,0c.737-.024.687.435.7.935.017.528-.076.871-.718.83-.556-.036-1.116-.007-1.675-.007Z" transform="translate(-9.925)"/></g></g></svg>',

			'kids'              => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs><style>.kids-circle{fill:#62bdc2;}.kids-path{fill:#fff;}</style></defs><g transform="translate(-1411 -3547)"><circle class="kids-circle" cx="19" cy="19" r="19" transform="translate(1411 3547)"/><g transform="translate(26.719 3513.354)"><path class="kids-path" d="M1396.573,246.464l-.286.552c-.226.438-.445.881-.68,1.315a1.112,1.112,0,0,1-1.918.13,1.069,1.069,0,0,1-.08-1.112c.37-.756.762-1.5,1.145-2.252q.391-.765.785-1.529c.082-.159.159-.322.253-.474a1.212,1.212,0,0,1,1.055-.634q2.385-.01,4.77,0a1.188,1.188,0,0,1,1.081.689c.461.872.906,1.753,1.357,2.63.21.408.425.813.627,1.225a.215.215,0,0,0,.228.139c1.84,0,3.68,0,5.52,0a1,1,0,0,1,.979.6c.5.991,1.021,1.974,1.53,2.962a.836.836,0,0,1-.037.9.761.761,0,0,1-.792.342.838.838,0,0,1-.645-.48q-.32-.617-.643-1.232a.166.166,0,0,0-.091-.08c0,.055,0,.109,0,.164q0,3.63,0,7.26c0,.2,0,.2-.194.2-.56,0-1.12,0-1.68,0-.133,0-.169-.042-.168-.172,0-1.4,0-2.81,0-4.215v-.2h-.232c0,.064-.011.131-.011.2q0,2.092,0,4.185c0,.205,0,.206-.208.206-.545,0-1.09,0-1.635,0-.151,0-.2-.043-.2-.2q.007-4.095,0-8.19c0-.214,0-.217-.213-.236-.8-.071-1.594-.131-2.389-.212a1.206,1.206,0,0,1-1-.689q-.427-.84-.857-1.678a.25.25,0,0,0-.09-.1c0,.064-.01.128-.01.192q0,5.362-.006,10.725a.958.958,0,0,0,0,.12c.015.119-.036.157-.153.156-.44,0-.88,0-1.32,0-.28,0-.56,0-.84,0-.111,0-.147-.036-.146-.146,0-.58,0-1.16,0-1.74q0-2.5,0-5.01c0-.111,0-.187-.155-.188s-.181.051-.181.19q.006,3.308,0,6.615c0,.05,0,.1,0,.15.005.088-.029.128-.121.127q-1.1,0-2.205,0c-.124,0-.118-.074-.118-.158q0-1.35,0-2.7,0-4.087,0-8.175v-.161Z" transform="translate(0 -194.651)"/><path class="kids-path" d="M1513.86,44.093a2.307,2.307,0,1,1-2.295-2.31A2.311,2.311,0,0,1,1513.86,44.093Z" transform="translate(-112.283 0)"/><path class="kids-path" d="M1834.589,252.716a1.72,1.72,0,0,1,0-3.44,1.72,1.72,0,0,1,0,3.44Z" transform="translate(-426.211 -201.268)"/></g></g></svg>',

			'mixing'            => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs><style>.mixing-circle{fill:#ef7e1e;}.mixing-path{fill:#fff;}</style></defs><g transform="translate(-1219 -2797)"><circle class="mixing-circle" cx="19" cy="19" r="19" transform="translate(1219 2797)"/><path class="mixing-path" d="M404.174,1464.381c-.218-.014-.437-.026-.654-.044a11.83,11.83,0,0,1-10.746-9.634,3.255,3.255,0,0,1-.089-1.067,1.981,1.981,0,0,1,1.993-1.734q2.308,0,4.616,0a.19.19,0,0,0,.2-.121,5.586,5.586,0,0,1,1.6-1.876,13.564,13.564,0,0,1,4.612-2.347,5.554,5.554,0,0,1,2.066-.234.213.213,0,0,1,.059.005c.25.1.4-.023.573-.2q2.833-2.849,5.68-5.684a.513.513,0,0,1,.837,0q.427.427.854.854a.5.5,0,0,1,0,.786q-2.926,2.926-5.855,5.847a.246.246,0,0,0-.084.251,5.152,5.152,0,0,1-.154,2.141c-.046.185-.1.368-.155.569h.146q2.218,0,4.436,0a1.985,1.985,0,0,1,1.844,1.216,2.278,2.278,0,0,1,.1,1.291,11.842,11.842,0,0,1-9.356,9.733,10.789,10.789,0,0,1-1.921.221c-.059,0-.118.012-.177.018Zm-5.19-11.162h-4.252c-.044,0-.088,0-.131,0a.655.655,0,0,0-.607.765c.089.5.209,1,.312,1.495.018.088.063.108.147.113.457.029.916.047,1.369.1a9.435,9.435,0,0,1,3.778,1.3c.536.328,1.035.714,1.553,1.072a10.988,10.988,0,0,0,3.554,1.706,6.778,6.778,0,0,0,2.572.162,7.207,7.207,0,0,0,2.447-.808c1.353-.72,2.7-1.449,4.1-2.083a.267.267,0,0,0,.111-.128,10.3,10.3,0,0,0,.844-2.864.67.67,0,0,0-.709-.837q-2.464,0-4.927,0a.157.157,0,0,0-.17.108,14.3,14.3,0,0,1-1.444,2.485,6.07,6.07,0,0,1-2.057,1.877,4.733,4.733,0,0,1-3.423.481,4.105,4.105,0,0,1-3.12-4.452C398.945,1453.554,398.964,1453.394,398.984,1453.219Zm2.444,3.3.01.034c.159-.019.32-.026.476-.06a5.194,5.194,0,0,0,1.967-.969,10.411,10.411,0,0,0,1.906-1.921c.1-.122.195-.247.309-.392h-.146c-.367,0-.734,0-1.1,0a.262.262,0,0,0-.207.089q-1.566,1.571-3.136,3.138C401.479,1456.466,401.454,1456.495,401.428,1456.522Zm1.7.706c.07,0,.157,0,.244,0,.1-.006.206-.017.308-.034a4.18,4.18,0,0,0,2.487-1.38,11.342,11.342,0,0,0,1.658-2.529.359.359,0,0,0,.017-.071c-.117,0-.224.005-.331,0a.177.177,0,0,0-.167.084c-.334.436-.664.876-1.013,1.3a9.66,9.66,0,0,1-2.14,2C403.841,1456.825,403.473,1457.023,403.124,1457.229Zm-.082-5.358.014.031c.462,0,.923,0,1.385,0a.252.252,0,0,0,.2-.082q1.379-1.384,2.761-2.764c.027-.026.051-.055.077-.083A18.871,18.871,0,0,0,403.042,1451.871Zm2.651-3.207c-.031.008-.053.013-.075.02a12.88,12.88,0,0,0-3.686,1.9,5.017,5.017,0,0,0-1.23,1.226c-.016.024-.027.051-.047.087.272,0,.527,0,.782,0a.178.178,0,0,0,.11-.055,18.36,18.36,0,0,1,2.716-2.24C404.727,1449.281,405.207,1448.982,405.693,1448.664Zm-5.036,7.076,2.526-2.526c-.441,0-.907,0-1.373,0a.172.172,0,0,0-.111.07,5.838,5.838,0,0,0-.79,1.345A2.8,2.8,0,0,0,400.657,1455.74Zm7.574-5.975-2.139,2.136c.29,0,.592,0,.893,0a.147.147,0,0,0,.1-.072c.313-.531.627-1.062.933-1.6C408.1,1450.071,408.168,1449.9,408.23,1449.765Zm-8.284,4.342.028,0,.473-.886h-.4Zm8.349-2.227.016.031c.037-.013.1-.017.107-.042.047-.11.079-.225.117-.339l-.025-.01Z" transform="translate(833.333 1362.763)"/></g></svg>',

			'non-vegetarian'    => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs><style>.non-vegeterian-circle{fill:#b36567;}.non-vegeterian-path{fill:#fff;stroke:#fff;stroke-width:0.25px;}</style></defs><g transform="translate(-1219 -2797)"><circle class="non-vegeterian-circle" cx="19" cy="19" r="19" transform="translate(1219 2797)"/><path class="non-vegeterian-path" d="M589.568-224.927a11.941,11.941,0,0,1,6.773,2.409,3.683,3.683,0,0,0,2.886.751,28.973,28.973,0,0,1,4.021-.1,2.1,2.1,0,0,0,1.8-1.4,6.929,6.929,0,0,1,.489-1.426,1.055,1.055,0,0,1,1.271-.594c.528.129.621.58.667,1.034a4.555,4.555,0,0,1,.023.759,1.422,1.422,0,0,0,.673,1.336c.762.551.916.98.638,1.516a1.343,1.343,0,0,1-1.7.6,3.468,3.468,0,0,1-.891-.319,1.421,1.421,0,0,0-1.441.006,1.257,1.257,0,0,0-.635.755,19.658,19.658,0,0,1-1.512,2.683.945.945,0,0,0-.176,1c.8,2.114.318,3.962-1.273,5.1a3.88,3.88,0,0,1-3.38.619c-3.052-.833-6.067-.329-9.109.049a30.19,30.19,0,0,1-4.922.17c-2.182-.088-3.385-1.26-4.049-3.536a8.623,8.623,0,0,1,7.273-11.1C587.85-224.739,588.71-224.825,589.568-224.927Zm14.494,4.669a2.236,2.236,0,0,1,2.609-.243,2.265,2.265,0,0,0,1.282.169,1.011,1.011,0,0,0,.63-.6.767.767,0,0,0-.244-.7,2.308,2.308,0,0,1-1.131-2.471.537.537,0,0,0-.042-.31c-.119-.207-.242-.531-.409-.562a.941.941,0,0,0-.78.216,2.308,2.308,0,0,0-.56,1.169,1.936,1.936,0,0,1-1.589,1.86,2.944,2.944,0,0,0-.381.182Zm-2.085,4.232a7.473,7.473,0,0,1-4.461,2.017c-1.8.174-3.056-.854-4.258-2.223a4.06,4.06,0,0,0,4.229,2.874C599.684-213.438,601.5-214.486,601.978-216.027Zm-21.564-1.879c1.705-2.4,3.247-4.721,5.994-5.786C583.845-223.564,580.993-220.854,580.413-217.906Z" transform="translate(643.629 3033.324)"/></g></svg>',

			'cold'              => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs><style>.cold-circle{fill:#73d4ef;}.cold-path{fill:#fff;}</style></defs><g transform="translate(-1219 -3679)"><circle class="cold-circle" cx="19" cy="19" r="19" transform="translate(1219 3679)"/><path class="cold-path" d="M-1247.965,728.665c-.71.441-1.368.866-2.045,1.261a1.326,1.326,0,0,1-.653.185.82.82,0,0,1-.751-.633.862.862,0,0,1,.381-.964c.606-.388,1.22-.762,1.83-1.143.089-.056.176-.116.316-.208l-1.428-.34c-.057-.013-.114-.024-.171-.038a.887.887,0,0,1-.735-1.07.886.886,0,0,1,1.136-.635c1.022.234,2.042.479,3.066.709a.57.57,0,0,0,.383-.027c1.417-.868,2.826-1.749,4.261-2.642-.071-.055-.123-.1-.182-.14-1.328-.828-2.654-1.659-3.99-2.474a.719.719,0,0,0-.482-.079c-1.025.223-2.044.472-3.066.708a.887.887,0,0,1-1.128-.648.88.88,0,0,1,.719-1.055c.508-.127,1.019-.245,1.529-.367.009,0,.015-.012.043-.034-.07-.05-.133-.1-.2-.143-.6-.376-1.205-.749-1.806-1.126a.885.885,0,0,1-.368-1.268.893.893,0,0,1,1.3-.213c.587.363,1.171.73,1.757,1.094.078.049.16.094.277.162-.131-.572-.255-1.1-.371-1.629a.858.858,0,0,1,.651-1.066.861.861,0,0,1,1.051.68q.364,1.531.709,3.066a.612.612,0,0,0,.3.425c1.267.778,2.526,1.569,3.789,2.355.057.035.116.066.2.116.01-.094.023-.165.024-.235,0-1.508.005-3.016-.007-4.523a.636.636,0,0,0-.174-.4c-.726-.745-1.464-1.48-2.2-2.213a.886.886,0,0,1-.268-.953.867.867,0,0,1,1.442-.346c.392.374.759.774,1.138,1.162l.073-.029v-.333c0-.691,0-1.381,0-2.072a.873.873,0,0,1,.888-.916.861.861,0,0,1,.861.908c.005.807,0,1.614,0,2.483.1-.08.158-.126.213-.181.331-.329.658-.663.992-.989a.88.88,0,0,1,1.276-.027.864.864,0,0,1-.033,1.254c-.759.768-1.529,1.526-2.284,2.3a.557.557,0,0,0-.156.347c-.01,1.547-.006,3.093-.006,4.64,0,.036.008.072.015.139.1-.057.185-.1.264-.148,1.254-.78,2.506-1.565,3.765-2.339a.611.611,0,0,0,.3-.427q.347-1.549.721-3.093a.85.85,0,0,1,1.068-.646.871.871,0,0,1,.62,1.092c-.1.446-.206.889-.309,1.334-.015.064-.024.13-.044.243.107-.057.18-.09.247-.133.6-.374,1.2-.753,1.807-1.125a.883.883,0,0,1,1.273.245.879.879,0,0,1-.356,1.248c-.592.373-1.188.74-1.782,1.11-.081.05-.159.105-.289.192.572.133,1.091.248,1.606.374a.878.878,0,0,1,.744.962.879.879,0,0,1-1.158.741c-1.022-.235-2.042-.481-3.066-.708a.6.6,0,0,0-.408.048c-1.375.843-2.743,1.7-4.112,2.553-.032.02-.06.047-.109.085.08.056.146.106.216.15,1.32.822,2.638,1.648,3.965,2.459a.68.68,0,0,0,.456.071c1.015-.222,2.025-.465,3.038-.7a.888.888,0,0,1,1.16.63.884.884,0,0,1-.761,1.078c-.5.12-1,.232-1.561.363.12.085.191.14.266.187.577.36,1.156.718,1.732,1.079a.9.9,0,0,1,.413,1.3.9.9,0,0,1-1.329.193q-.855-.53-1.709-1.062c-.08-.049-.162-.094-.3-.172.132.568.255,1.075.368,1.584a.879.879,0,0,1-.37,1,.818.818,0,0,1-.98-.04,1.254,1.254,0,0,1-.352-.578c-.261-1.026-.488-2.06-.736-3.089a.48.48,0,0,0-.172-.278c-1.332-.84-2.67-1.669-4.007-2.5a.347.347,0,0,0-.09-.014c-.007.092-.019.182-.019.272,0,1.479-.005,2.957.008,4.436a.68.68,0,0,0,.181.425c.74.76,1.5,1.5,2.242,2.256a.84.84,0,0,1,.175,1.017.816.816,0,0,1-.889.467,1.2,1.2,0,0,1-.55-.29c-.377-.349-.723-.73-1.146-1.163-.011.186-.021.284-.022.382,0,.7.006,1.4,0,2.1a.86.86,0,0,1-.853.917.865.865,0,0,1-.9-.934c-.007-.691,0-1.381,0-2.072a.713.713,0,0,0-.067-.35c-.36.362-.707.739-1.086,1.081a1.38,1.38,0,0,1-.6.326.817.817,0,0,1-.889-.467.87.87,0,0,1,.182-1.02c.58-.59,1.138-1.205,1.761-1.745a1.9,1.9,0,0,0,.712-1.685c-.066-1.232-.017-2.47-.018-3.705,0-.086-.007-.171-.013-.294-.1.052-.172.086-.241.129-1.263.786-2.522,1.576-3.789,2.355a.615.615,0,0,0-.3.426c-.227,1-.462,2.006-.7,3.008a.875.875,0,0,1-1.064.739.877.877,0,0,1-.637-1.127C-1248.216,729.737-1248.094,729.228-1247.965,728.665Z" transform="translate(2479.045 2975.375)"/></g></svg>',

			'fork-knife'        => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs><style>.fork-knife-circle{fill:#232323;}.fork-knife-path{fill:#fff;}</style></defs><g transform="translate(-1411 -3679)"><circle class="fork-knife-circle" cx="19" cy="19" r="19" transform="translate(1411 3679)"/><path class="fork-knife-path" d="M1042.118-206.656c.874-1.909,1.394-3.9,2.566-5.367,1.188-1.489,3.02-2.464,4.57-3.664l.388.481-5.252,4.982.527.546,5.214-5.119.466.433-5.136,5.207.562.529,4.639-5.574.838.782-5.108,5.209.37.337,5.01-5.253.555.471a4.38,4.38,0,0,1-.416.908,26.527,26.527,0,0,1-2.856,3.519,14.5,14.5,0,0,1-3.193,1.851c-.335.174-.818.087-1.126.286-.731.473-1.4,1.044-2.089,1.576a22.127,22.127,0,0,0,1.6,2.008c1.974,1.981,4.03,3.882,5.961,5.9.286.3.074,1.076.093,1.631-.594-.014-1.405.217-1.745-.083-1.593-1.409-3.073-2.949-4.56-4.473-.938-.961-1.818-1.978-2.888-3.148-1.881,1.9-3.565,3.6-5.252,5.3-.711.716-1.419,1.436-2.144,2.138-.64.619-1.325,1.494-2.247.556s-.059-1.617.574-2.252c2.374-2.379,4.756-4.75,7.237-7.225l-10.061-9.912.313-.617a9.987,9.987,0,0,1,2.825.773c2.5,1.581,4.911,3.3,7.307,5.041A29.2,29.2,0,0,1,1042.118-206.656Z" transform="translate(388.789 3902.688)"/></g></svg>',

			'spicy'             => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs><style>.spicy-circle{fill:#ed4040;}.spicy-path{fill:#fff;}</style></defs><g transform="translate(-1219 -3745)"><circle class="spicy-circle" cx="19" cy="19" r="19" transform="translate(1219 3745)"/><g transform="translate(1229.586 3754.041)"><path class="spicy-path" d="M4.037,2.148c-1.169,1.072-.26,3.508-.26,3.508-1.332,1.527-.91,4.158-.91,4.158L4.46,5.494c-.195-2.339.682-3.021.682-3.021.162-.357-.682-.715-1.1-.325" transform="translate(-2.553 -1.979)"/><path class="spicy-path" d="M5.472,3.3c-.975.877-.065,3.184-.065,3.184C4.075,8.01,4.53,9.407,4.5,10.252l6.465-3.833c-.715-1.884-4.71-.877-4.71-.877a1.728,1.728,0,0,1,0-2.177,1.31,1.31,0,0,1-.78-.065" transform="translate(-3.662 -2.871)"/><path class="spicy-path" d="M15.307,22.62A9.892,9.892,0,0,1,11.8,14.434,3.792,3.792,0,0,0,9.1,10.5c-.715-.13-2.924-.227-2.177.422.715.65-2.631.13-2.631,1.527,0,1.007-.845-.195-1.852,2.306-1.007,2.6-.032,4.386,1.429,6.01,2.859,3.119,9.551,8.8,16.47,5.23,3.573-1.787-1.982-.455-5.035-3.379" transform="translate(-2.029 -7.67)"/></g></g></svg>',

			'fish'              => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs><style>.fish-circle{fill:#829496;}.fish-path{fill:#fff;}</style></defs><g transform="translate(-1411 -3745)"><circle class="fish-circle" cx="19" cy="19" r="19" transform="translate(1411 3745)"/><path class="fish-path" d="M156.016,668.307l-.854.757c.4.083.731.151,1.065.223a11.2,11.2,0,0,1,4.04,1.549c.517.351.5.315,1.015,0,1.411-.842,2.842-1.651,4.359-2.528-.219.449-.4.8-.562,1.159-.2.449-.383.908-.58,1.36a3.744,3.744,0,0,0-.011,3.133c.359.815.714,1.632,1.1,2.524l-4.859-2.827a6.937,6.937,0,0,1-2.312,1.237,28.261,28.261,0,0,1-13.427.8,13.31,13.31,0,0,1-3.913-1.12,6.9,6.9,0,0,1-1.688-1.15,1.38,1.38,0,0,1,.033-2.107,6.981,6.981,0,0,1,3.061-1.626,23.778,23.778,0,0,1,4.852-.9,1.336,1.336,0,0,0,.749-.322,2.215,2.215,0,0,1,1.531-.645,30.853,30.853,0,0,1,6.284.395C155.924,668.236,155.943,668.258,156.016,668.307Zm-4.326,7.062a15.908,15.908,0,0,0,8.1-1.832A47.645,47.645,0,0,1,151.69,675.369Zm-6.161-.921c1.092-.846,1.155-3.255.074-4.074A5.493,5.493,0,0,1,145.529,674.448Zm-2.365-4.085a.874.874,0,1,0,.9.869A.876.876,0,0,0,143.164,670.363Z" transform="translate(1278.08 3092.184)"/></g></svg>',

			'raw'               => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs><style>.raw-circle{fill:#ee6b35;}.raw-path{fill:#fff;}</style></defs><g transform="translate(-1219 -3811)"><circle class="raw-circle" cx="19" cy="19" r="19" transform="translate(1219 3811)"/><path class="raw-path" d="M217.345,254.158a6.283,6.283,0,0,0,.693-3.177,7.337,7.337,0,0,0-.4-1.239,2.911,2.911,0,0,1-.059-.534c.186.059.426.064.55.186a3.56,3.56,0,0,1,.959,2.193,6.558,6.558,0,0,1-.408,2.877.265.265,0,1,0,.4.284,5.714,5.714,0,0,1,2.642-1.408,11.146,11.146,0,0,1,2.072.023.811.811,0,0,1,.433.2.4.4,0,0,1,.145.323c-.019.078-.194.117-.3.172-.013.007-.032,0-.048,0a4.745,4.745,0,0,0-4.105,1.788.281.281,0,0,0,.039.081c1.031,1.069,1.077,1.355.367,2.65a48.486,48.486,0,0,1-4.9,6.932,69.135,69.135,0,0,1-6.732,7.406,31.733,31.733,0,0,1-2.864,2.269c-.352.259-.816.542-1.264.227-.423-.3-.333-.818-.229-1.224a23.569,23.569,0,0,1,.952-3.171,72.643,72.643,0,0,1,8.447-15.145c.334-.459.734-.871,1.108-1.3a1.732,1.732,0,0,1,2.065-.547C217.041,254.079,217.181,254.111,217.345,254.158Zm6.732-.341c-1.6-1.053-4.288.159-5.745,2.048.716.445.716.444,1.326-.125.166-.155.329-.314.5-.458C221.3,254.355,222.461,253.5,224.078,253.817Zm-6.269-4.316a3.305,3.305,0,0,1,.414,2.448,23.621,23.621,0,0,1-.718,2.418c-.26.74-.277.766.453,1.194C219.237,253.457,219.179,250.361,217.809,249.5Zm-3.171,10.219.088-.121-1.917-2.061-.443.677Zm.749,5.445.453-.564-2.151-.865-.06.111Zm-7.025-.14,1.311.759.062-.086-1.13-1.185Z" transform="translate(1024.752 3566.793)"/></g></svg>',

			'liquid'            => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs><style>.liquid-circle{fill:#597400;}.liquid-path{fill:#fff;}</style></defs><g transform="translate(-1411 -3811)"><circle class="liquid-circle" cx="19" cy="19" r="19" transform="translate(1411 3811)"/><path class="liquid-path" d="M669.573,613.68c.5,1.3.892,2.531,1.435,3.693.779,1.668,1.662,3.288,2.511,4.923.485.935,1.079,1.824,1.471,2.795a5.693,5.693,0,0,1-1.57,6.377,5.806,5.806,0,0,1-8.731-7.479,42.554,42.554,0,0,0,4.629-9.494Zm-.918,4.112-.07-.024c-.047.1-.092.2-.14.293-.791,1.628-1.566,3.263-2.379,4.88-.417.829-.938,1.606-1.345,2.439a3.51,3.51,0,0,0,.018,3.444,1.6,1.6,0,0,0,1.819.864,1.8,1.8,0,0,0,.88-2.861,3.213,3.213,0,0,1-.613-3.1c.392-1.313.794-2.622,1.2-3.931C668.228,619.124,668.443,618.458,668.655,617.792Zm2.07,14.11c2.023-.014,3.938-2.114,4.015-4.4A6.384,6.384,0,0,1,670.725,631.9Z" transform="translate(760.215 3206.32)"/></g></svg>',

			'dairy-free'        => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs>dairy-free</defs><g id="Group_4626" data-name="Group 4626" transform="translate(-521 -148)"><g id="Group_4430" data-name="Group 4430" transform="translate(-698 -2588)"><circle id="Ellipse_97" data-name="Ellipse 97" cx="19" cy="19" r="19" transform="translate(1219 2736)" fill="#2fcde1"/><g id="Group_4424" data-name="Group 4424" transform="translate(1814.985 2514.889)"><path id="Path_30559" data-name="Path 30559" d="M-579.2,243.665h3.987c-.2,1.972.9,3.49,1.555,5.165a10.283,10.283,0,0,1,.79,3.37c.1,3.492.04,6.988.035,10.483,0,1.363-.27,1.626-1.6,1.626-1.929,0-3.858,0-5.787,0-.987,0-1.335-.307-1.339-1.29-.015-3.8-.063-7.606.04-11.405a8.492,8.492,0,0,1,.869-3.016C-579.97,247.015-579,245.54-579.2,243.665Z" transform="translate(0 -12.909)" fill="#fff"/><path id="Path_30560" data-name="Path 30560" d="M-567.31,229.907c-.475,0-.952-.035-1.423.008-.677.062-.919-.172-.933-.883-.014-.753.3-.9.947-.877q1.548.049,3.1,0c.737-.024.687.435.7.935.017.528-.076.871-.718.83-.556-.036-1.116-.007-1.675-.007Z" transform="translate(-9.925)" fill="#fff"/></g></g><line id="Line_1" data-name="Line 1" x2="38" transform="translate(527.065 153.065) rotate(45)" fill="none" stroke="#fff" stroke-width="2"/></g></svg>',

			'nut-free'          => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs>nut-free</defs><g id="Group_4625" data-name="Group 4625" transform="translate(-521 -221)"><g id="Group_4430" data-name="Group 4430" transform="translate(-698 -2515)"><circle id="Ellipse_97" data-name="Ellipse 97" cx="19" cy="19" r="19" transform="translate(1219 2736)" fill="#ee9060"/><g id="Group_4595" data-name="Group 4595" transform="translate(1201.842 2717.144)"><path id="Path_30585" data-name="Path 30585" d="M43.37,52.69c-.107,3.887-3.271,9.635-7.162,9.635s-7.052-5.748-7.158-9.635Z" transform="translate(-0.892 -13.78)" fill="#fff"/><path id="Path_30586" data-name="Path 30586" d="M44.8,37.451H26.42a9.257,9.257,0,0,1,9.189-9.071,9.056,9.056,0,0,1,6.5,2.748A9.286,9.286,0,0,1,44.8,37.451Z" transform="translate(-0.262)" fill="#fff"/></g></g><line id="Line_1" data-name="Line 1" x2="9.435" y2="9.435" transform="translate(544.5 243.5)" fill="none" stroke="#fff" stroke-width="2"/><line id="Line_2" data-name="Line 2" x2="7.435" y2="7.435" transform="translate(527.065 226.065)" fill="none" stroke="#fff" stroke-width="2"/></g></svg>',

			'pescetarian'       => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs>pescetarian</defs><g id="Group_4624" data-name="Group 4624" transform="translate(-953 -528)"><circle id="Ellipse_97" data-name="Ellipse 97" cx="19" cy="19" r="19" transform="translate(953 528)" fill="#747278"/><g id="Group_4610" data-name="Group 4610" transform="translate(954.5 522.412)"><path id="Path_30630" data-name="Path 30630" d="M28.524,20.266a26.827,26.827,0,0,0-4.952,2.294A11.6,11.6,0,0,0,19.923,19.8a2.508,2.508,0,0,1,.417-3.023,6.4,6.4,0,0,0-6.255,2.163c-4.3.938-7.585,5.291-7.585,6.151,0,.678,3.831,5.916,8.679,6.281A3.777,3.777,0,0,0,18.906,32.7a1.453,1.453,0,0,1-.287-1.72,10.687,10.687,0,0,0,5.109-3.284A26.877,26.877,0,0,0,28.55,29.91a26.1,26.1,0,0,1-1.2-4.822A17.184,17.184,0,0,1,28.524,20.266ZM11.609,25.531A1.434,1.434,0,1,1,13.042,24.1,1.429,1.429,0,0,1,11.609,25.531Z" transform="translate(0 0)" fill="#fff"/></g></g></svg>',

			'quick-meals'       => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs>quick-meals</defs><g id="Group_4623" data-name="Group 4623" transform="translate(-953 -456)"><circle id="Ellipse_97" data-name="Ellipse 97" cx="19" cy="19" r="19" transform="translate(953 456)" fill="#e6b170"/><g id="Group_4612" data-name="Group 4612" transform="translate(948.249 443.5)"><g id="Group_4611" data-name="Group 4611" transform="translate(13.751 22.5)"><path id="Path_30631" data-name="Path 30631" d="M33.751,24.869l-.2-.2a.556.556,0,0,0-.787,0l-6.086,5.536H13.851a9.722,9.722,0,0,0,3.67,9.081l-1.593,1.593a.556.556,0,0,0,0,.787l.2.2a.556.556,0,0,0,.787,0l1.79-1.79a9.725,9.725,0,0,0,9.566,0l1.79,1.79a.556.556,0,0,0,.787,0l.2-.2a.556.556,0,0,0,0-.787l-1.593-1.593a9.714,9.714,0,0,0,3.77-7.691,9.842,9.842,0,0,0-.1-1.391h-3.75l4.383-4.553A.557.557,0,0,0,33.751,24.869Z" transform="translate(-13.751 -23.95)" fill="#fff"/><circle id="Ellipse_103" data-name="Ellipse 103" cx="0.695" cy="0.695" r="0.695" transform="translate(5.562 4.172)" fill="#fff"/><circle id="Ellipse_104" data-name="Ellipse 104" cx="0.695" cy="0.695" r="0.695" transform="translate(7.648 2.781)" fill="#fff"/><circle id="Ellipse_105" data-name="Ellipse 105" cx="1.391" cy="1.391" r="1.391" transform="translate(3.476)" fill="#fff"/></g></g></g></svg>',

			'whole-30'          => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs>whole-30</defs><g id="Group_4622" data-name="Group 4622" transform="translate(-953 -383)"><circle id="Ellipse_97" data-name="Ellipse 97" cx="19" cy="19" r="19" transform="translate(953 383)" fill="#f62934"/><g id="Group_4613" data-name="Group 4613" transform="translate(959 389)"><path id="Path_30632" data-name="Path 30632" d="M23.961,10H2a11.107,11.107,0,0,0,6.369,9.992L7.38,21.53a1.112,1.112,0,0,0,.988,1.647h9.224a1.112,1.112,0,0,0,.988-1.647l-.988-1.537A11.107,11.107,0,0,0,23.961,10Z" transform="translate(0 0.784)" fill="#fff"/><circle id="Ellipse_106" data-name="Ellipse 106" cx="1.5" cy="1.5" r="1.5" transform="translate(5 7)" fill="#fff"/><circle id="Ellipse_107" data-name="Ellipse 107" cx="1" cy="1" r="1" transform="translate(11 7)" fill="#fff"/><circle id="Ellipse_108" data-name="Ellipse 108" cx="1" cy="1" r="1" transform="translate(8 3)" fill="#fff"/><circle id="Ellipse_109" data-name="Ellipse 109" cx="1" cy="1" r="1" transform="translate(15 2)" fill="#fff"/><circle id="Ellipse_110" data-name="Ellipse 110" cx="1" cy="1" r="1" transform="translate(17 8)" fill="#fff"/></g></g></svg>',

			'vegan'             => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs>vegan</defs><g id="Group_4621" data-name="Group 4621" transform="translate(-953 -311)"><circle id="Ellipse_97" data-name="Ellipse 97" cx="19" cy="19" r="19" transform="translate(953 311)" fill="#0dab20"/><path id="Path_30625" data-name="Path 30625" d="M17.264,12.731S15.7,8.959,12.97,8.6c0,0,.408.447,1.047,1.263a37.325,37.325,0,0,1,7.157,14.979s1.982-16.407,10.9-20.481c0,0-7.873,8.7-9.635,26.758H19.907s-.55-10.681-5.4-16.848c0,0-3.744.33-5.065-2.2S8.455,6.9,6.8,4.362C19.063,3.334,17.264,12.731,17.264,12.731Z" transform="translate(952.197 317.781)" fill="#fff"/></g></svg>',

			'keto'              => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs>keto</defs><g id="Group_4620" data-name="Group 4620" transform="translate(-953 -238)"><circle id="Ellipse_97" data-name="Ellipse 97" cx="19" cy="19" r="19" transform="translate(953 238)" fill="#769558"/><g id="Group_4608" data-name="Group 4608" transform="matrix(0.966, 0.259, -0.259, 0.966, 966.366, 243.78)"><path id="Path_30622" data-name="Path 30622" d="M16.138,15.8a3.73,3.73,0,0,0-3.493,3.929,3.73,3.73,0,0,0,3.493,3.929,3.73,3.73,0,0,0,3.492-3.929A3.731,3.731,0,0,0,16.138,15.8Zm-1.751,3.183a.553.553,0,0,1-.541.433A.343.343,0,0,1,13.512,19a2.792,2.792,0,0,1,2.626-2.328,2.373,2.373,0,0,1,.436.049A2.856,2.856,0,0,0,14.387,18.985Z" transform="translate(-3.04 -3.799)" fill="#fff"/><path id="Path_30623" data-name="Path 30623" d="M6.434,11.631a7.079,7.079,0,0,0,1.234-4,5.686,5.686,0,0,1,1.48-3.826c-.062-.176-.141-.346-.191-.526L8.933,3.2A4.393,4.393,0,0,0,.477,5.588l0,.006a8.4,8.4,0,0,1-.118,4.953A7.028,7.028,0,0,0,5.7,19.666a8.31,8.31,0,0,1,.732-8.035Z" transform="translate(0)" fill="#fff"/><path id="Path_30624" data-name="Path 30624" d="M20.958,13.425a8.513,8.513,0,0,1-1.513-4.69V8.65a4.183,4.183,0,1,0-8.355,0v.007A8.528,8.528,0,0,1,9.5,13.391a6.989,6.989,0,1,0,11.463.034Zm-5.77,9.206a5.291,5.291,0,0,1-4.383-8.258A10.119,10.119,0,0,0,12.569,8.65a2.678,2.678,0,0,1,5.356,0v.085a10.1,10.1,0,0,0,1.7,5.663,5.236,5.236,0,0,1,.876,2.939A5.31,5.31,0,0,1,15.189,22.631Z" transform="translate(-1.975 -1.024)" fill="#fff"/></g></g></svg>',

			'high-protein'      => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs>high-protein</defs><g id="Group_4619" data-name="Group 4619" transform="translate(-953 -166)"><circle id="Ellipse_97" data-name="Ellipse 97" cx="19" cy="19" r="19" transform="translate(953 166)" fill="#990216"/><g id="Group_4607" data-name="Group 4607" transform="translate(964 174)"><path id="Path_30620" data-name="Path 30620" d="M17.08,21.6l17.31-7.427L22.417,10.128Zm8.13-7.927a1.6,1.6,0,0,1-1.432.887,1.439,1.439,0,0,1-.429-.064,1.2,1.2,0,0,1-.75-.662,1.222,1.222,0,0,1,.021-1.014,1.6,1.6,0,0,1,1.432-.887,1.457,1.457,0,0,1,.429.064,1.2,1.2,0,0,1,.75.662A1.22,1.22,0,0,1,25.21,13.671Zm-3.953,3.412a.951.951,0,0,1,.847-.525.869.869,0,0,1,.256.038.728.728,0,0,1,.454.4.736.736,0,0,1-.013.614.95.95,0,0,1-.847.526A.863.863,0,0,1,21.7,18.1a.726.726,0,0,1-.455-.4A.737.737,0,0,1,21.257,17.083Z" transform="translate(-16.874 -10.128)" fill="#fff"/><g id="Group_4606" data-name="Group 4606" transform="translate(0 4.193)"><path id="Path_30621" data-name="Path 30621" d="M16.31,39.859l0,2.583v.48l.44-.189L34.078,35.3l.191-.082V25.86l-17.953,7.7,0,4.267a1.344,1.344,0,0,1,.241-.024.71.71,0,0,1,.5.17.572.572,0,0,1,.146.421A1.941,1.941,0,0,1,16.31,39.859Zm13.115-7.25a1.188,1.188,0,0,1-.867-.313,1.038,1.038,0,0,1-.283-.769,2.388,2.388,0,0,1,.571-1.454,3.2,3.2,0,0,1,1.224-.935,2.059,2.059,0,0,1,.766-.155,1.189,1.189,0,0,1,.866.313,1.041,1.041,0,0,1,.283.769,2.388,2.388,0,0,1-.571,1.454,3.2,3.2,0,0,1-1.224.935A2.059,2.059,0,0,1,29.425,32.61Zm-2.134,3.163a1.453,1.453,0,0,1,.909-1.206,1.114,1.114,0,0,1,.4-.077.646.646,0,0,1,.472.172.564.564,0,0,1,.152.418,1.454,1.454,0,0,1-.909,1.207,1.13,1.13,0,0,1-.4.077.645.645,0,0,1-.472-.172A.561.561,0,0,1,27.291,35.773Zm-7.82,3.11a1.98,1.98,0,0,1,.473-1.2,2.637,2.637,0,0,1,1.01-.769,1.724,1.724,0,0,1,.637-.128,1,1,0,0,1,.728.263.871.871,0,0,1,.237.645,1.979,1.979,0,0,1-.473,1.2,2.627,2.627,0,0,1-1.01.769,1.713,1.713,0,0,1-.637.128,1,1,0,0,1-.727-.263A.872.872,0,0,1,19.471,38.883Zm-.531-4.4a1.453,1.453,0,0,1,.909-1.207,1.126,1.126,0,0,1,.4-.077.645.645,0,0,1,.472.172.563.563,0,0,1,.153.419,1.454,1.454,0,0,1-.909,1.207,1.117,1.117,0,0,1-.4.077.643.643,0,0,1-.472-.172A.56.56,0,0,1,18.94,34.479Z" transform="translate(-16.308 -25.86)" fill="#fff"/></g></g></g></svg>',

			'organic'           => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs>organic</defs><g id="Group_4618" data-name="Group 4618" transform="translate(-953 -93)"><circle id="Ellipse_97" data-name="Ellipse 97" cx="19" cy="19" r="19" transform="translate(953 93)" fill="#0f6b2b"/><g id="Group_4596" data-name="Group 4596" transform="translate(952.53 98.416)"><path id="Path_30587" data-name="Path 30587" d="M25.861,14.085a14.434,14.434,0,0,1,1.632,3.3l2.023.171A15.648,15.648,0,0,1,31.861,12.5a3.977,3.977,0,0,0,4.023-.391c1.7-1.566,1.54-6.311,1.1-9.051a.519.519,0,0,0-.943-.245c-2.368,3.229-6.529,3.669-6.46,6.776a2.156,2.156,0,0,0,.575,1.566,9.474,9.474,0,0,1,4.046-3.2.079.079,0,0,0,.046-.024,14.845,14.845,0,0,0-2.322,2.471,15.828,15.828,0,0,0-3.4,6.189A15.5,15.5,0,0,0,27.1,13.719,14.915,14.915,0,0,0,24.251,10c.023,0,.023,0,.046.024a8.355,8.355,0,0,1,3.38,2.862,1.778,1.778,0,0,0,.529-1.321C28.344,8.9,24.78,8.41,22.849,5.573a.457.457,0,0,0-.828.2c-.437,2.324-.713,6.385.713,7.779A3.33,3.33,0,0,0,25.861,14.085Z" transform="translate(-8.459)" fill="#fff"/><path id="Path_30588" data-name="Path 30588" d="M26.207,68.05a21.1,21.1,0,0,1-2.174.587,2.458,2.458,0,0,1-1.781,1.125,3.1,3.1,0,0,1-.694.049,18.669,18.669,0,0,1-2.151-.171.374.374,0,0,1-.162-.024.41.41,0,0,1-.37-.391.4.4,0,0,1,.44-.44c.069,0,.116.024.185.024a11.448,11.448,0,0,0,2.637.122c1.157-.2,1.48-.93,1.434-1.566a.979.979,0,0,0-.972-.978,20.57,20.57,0,0,1-2.683-.122A17.243,17.243,0,0,0,17,66.191,10.876,10.876,0,0,0,12.837,68c-.763.538-1.619,1.15-2.128,1.541a.619.619,0,0,0-.116.856l2.174,3.107a.537.537,0,0,0,.787.122l.833-.66a.791.791,0,0,1,.694-.122,17.342,17.342,0,0,0,4.141.978c2.29.171,6.732-2.617,8.027-3.523A1.245,1.245,0,0,0,26.207,68.05Z" transform="translate(0 -49.219)" fill="#fff"/></g></g></svg>',

			'corn-free'         => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs>corn-free</defs><g id="Group_4617" data-name="Group 4617" transform="translate(-751 -456)"><circle id="Ellipse_97" data-name="Ellipse 97" cx="19" cy="19" r="19" transform="translate(751 456)" fill="#fecb00"/><g id="Group_4605" data-name="Group 4605" transform="translate(759 465)"><path id="Path_30608" data-name="Path 30608" d="M20.3,7.292a.209.209,0,0,0,.3,0l.9-.9a.754.754,0,0,0,0-1.066l-.514-.514a.754.754,0,0,0-1.065,0l-.9.9a.209.209,0,0,0-.062.148.207.207,0,0,0,.062.147Z" transform="translate(-5.849 -1.418)" fill="#fff"/><path id="Path_30609" data-name="Path 30609" d="M17.557,10.033a.209.209,0,0,0,.3,0l.9-.9a.754.754,0,0,0,0-1.066l-.514-.514a.753.753,0,0,0-1.065,0l-.9.9a.209.209,0,0,0,0,.3Z" transform="translate(-5.003 -2.263)" fill="#fff"/><path id="Path_30610" data-name="Path 30610" d="M24.494,6.009l1.283,1.283a.209.209,0,0,0,.3,0l.9-.9a.753.753,0,0,0,0-1.066l-.513-.514a.752.752,0,0,0-1.065,0l-.9.9A.209.209,0,0,0,24.494,6.009Z" transform="translate(-7.54 -1.418)" fill="#fff"/><path id="Path_30611" data-name="Path 30611" d="M23.718,7.556a.753.753,0,0,0-1.065,0l-.9.9a.209.209,0,0,0,0,.3l1.284,1.283a.209.209,0,0,0,.3,0l.9-.9a.753.753,0,0,0,0-1.065Z" transform="translate(-6.694 -2.264)" fill="#fff"/><path id="Path_30612" data-name="Path 30612" d="M23.037,4.553a.21.21,0,0,0,.3,0l.9-.9a.753.753,0,0,0,0-1.065l-.514-.513a.751.751,0,0,0-1.064,0l-.9.9a.209.209,0,0,0,0,.3Z" transform="translate(-6.694 -0.572)" fill="#fff"/><path id="Path_30613" data-name="Path 30613" d="M12.756,13.037a.754.754,0,0,0-1.065,0l-.9.9a.209.209,0,0,0,0,.3l1.283,1.284a.21.21,0,0,0,.3,0l.9-.9a.753.753,0,0,0,0-1.065Z" transform="translate(-3.312 -3.955)" fill="#fff"/><path id="Path_30614" data-name="Path 30614" d="M20.978,10.3a.753.753,0,0,0-1.065,0l-.9.9a.209.209,0,0,0,0,.3L20.3,12.775a.209.209,0,0,0,.148.062.207.207,0,0,0,.147-.062l.9-.9a.756.756,0,0,0,0-1.066Z" transform="translate(-5.849 -3.11)" fill="#fff"/><path id="Path_30615" data-name="Path 30615" d="M10.017,15.776a.753.753,0,0,0-1.065,0l-.9.9a.21.21,0,0,0,0,.3l1.284,1.283a.209.209,0,0,0,.3,0l.9-.9a.754.754,0,0,0,0-1.066Z" transform="translate(-2.466 -4.801)" fill="#fff"/><path id="Path_30616" data-name="Path 30616" d="M7.276,18.518a.754.754,0,0,0-1.066,0l-.9.9a.209.209,0,0,0,0,.3L6.595,21a.21.21,0,0,0,.3,0l.9-.9a.753.753,0,0,0,0-1.065Z" transform="translate(-1.62 -5.647)" fill="#fff"/><path id="Path_30617" data-name="Path 30617" d="M13.532,11.492l1.283,1.283a.211.211,0,0,0,.3,0l.9-.9a.752.752,0,0,0,0-1.065L15.5,10.3a.754.754,0,0,0-1.065,0l-.9.9A.209.209,0,0,0,13.532,11.492Z" transform="translate(-4.157 -3.11)" fill="#fff"/><path id="Path_30618" data-name="Path 30618" d="M18.236,13.036a.753.753,0,0,0-1.066,0l-.9.9a.209.209,0,0,0,0,.3l1.283,1.284a.207.207,0,0,0,.148.062.209.209,0,0,0,.148-.062l.9-.9a.754.754,0,0,0,0-1.066Z" transform="translate(-5.003 -3.955)" fill="#fff"/><path id="Path_30619" data-name="Path 30619" d="M22.068,9.251a.406.406,0,0,0-.388-.283h-.034a23.784,23.784,0,0,0-3.55.643,24.494,24.494,0,0,0,1.66-2.892c1.371-2.846,1.547-4.992.5-6.039A2.612,2.612,0,0,0,18.327,0c-2.262,0-5.886,1.8-9.2,4.448A19.846,19.846,0,0,0,9.656,1.5.406.406,0,0,0,8.9,1.261,52.507,52.507,0,0,1,3.029,9.2c-2.6,2.816-3.016,5.291-1.264,7.538L.2,18.3a.684.684,0,0,0,0,.968l1.419,1.419a.684.684,0,0,0,.968,0l1.764-1.764a4.927,4.927,0,0,0,2.4.634c1.745,0,3.577-.908,5.6-2.777a63.139,63.139,0,0,1,9.535-7.056A.4.4,0,0,0,22.068,9.251ZM10.658,5.875a.209.209,0,0,0,.3,0l.9-.9a.753.753,0,0,0,0-1.065l-.263-.263q.19-.129.38-.252l.583.584a.209.209,0,0,0,.3,0l.9-.9a.753.753,0,0,0,.159-.828c.158-.083.313-.163.469-.241l.072.073a.209.209,0,0,0,.148.062.212.212,0,0,0,.148-.062l.457-.457a11.307,11.307,0,0,1,1.92-.659l1.116,1.116a.209.209,0,0,0,.3,0l.976-.975a1.3,1.3,0,0,1,.174.144,1.185,1.185,0,0,1,.141.172l-.975.974a.21.21,0,0,0,0,.3l1.117,1.117a11.216,11.216,0,0,1-.643,1.9l-.474.473a.21.21,0,0,0,0,.3l.08.08c-.076.154-.156.308-.239.465a.752.752,0,0,0-.838.156l-.9.9a.209.209,0,0,0,0,.3l.588.588c-.082.127-.167.253-.252.38l-.267-.268a.755.755,0,0,0-1.065,0l-.9.9a.208.208,0,0,0,0,.3l.229.229q-.311.121-.626.254a.748.748,0,0,0-.6.216l-.09.091a17.808,17.808,0,0,0-1.9,1.033.751.751,0,0,0-.216-.609l-.513-.514a.754.754,0,0,0-1.065,0l-.9.9a.209.209,0,0,0,0,.3l1.09,1.09c-.124.1-.245.21-.366.318a.722.722,0,0,0-.14-.194l-.514-.514a.754.754,0,0,0-1.065,0l-.9.9a.209.209,0,0,0,0,.3L8.513,15.1q-.179.158-.355.3a.692.692,0,0,0-.1-.119l-.513-.514a.754.754,0,0,0-1.066,0l-.9.9a.209.209,0,0,0,0,.3l.672.673c-.152.072-.3.135-.442.189l-.158-.162a.754.754,0,0,0-1.062,0l-.316-.317a.752.752,0,0,0,0-1.061l-.514-.514a.752.752,0,0,0-.533-.221.571.571,0,0,0-.059,0A9.062,9.062,0,0,1,5.3,11.674,11.3,11.3,0,0,0,6.915,9.7a.205.205,0,0,0,.25-.034l.9-.9a.754.754,0,0,0,0-1.065L8.02,7.656q.1-.221.19-.441l.554.553a.209.209,0,0,0,.3,0l.9-.9a.756.756,0,0,0,0-1.066l-.546-.545c.116-.095.234-.188.351-.28Z" transform="translate(0 0)" fill="#fff"/></g><line id="Line_1" data-name="Line 1" x2="9.435" y2="9.435" transform="translate(774.5 478.5)" fill="none" stroke="#fff" stroke-width="2"/><line id="Line_2" data-name="Line 2" x2="9.435" y2="9.435" transform="translate(757.065 461.065)" fill="none" stroke="#fff" stroke-width="2"/></g></svg>',

			'soy-free'          => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs>soy-free</defs><g id="Group_4616" data-name="Group 4616" transform="translate(-751 -383)"><circle id="Ellipse_97" data-name="Ellipse 97" cx="19" cy="19" r="19" transform="translate(751 383)" fill="#ee9060"/><g id="Group_4603" data-name="Group 4603" transform="translate(759 392.105)"><path id="Path_30606" data-name="Path 30606" d="M17.906,22.856a14.373,14.373,0,0,1-1.244-1.639q-.742-1.105-1.729-2.774a4.449,4.449,0,0,0-1.182-1.188l-.034-.022a3.173,3.173,0,0,0-1.975-.734,3.746,3.746,0,0,0-2.009.662q-2.393,1.964-.644,7.479,2.094,4.734,5.874,5.887a5.95,5.95,0,0,1,1.76-2.513,12.8,12.8,0,0,1,1.4-1.163l.05-.031c1.154-.578,2.055-1.076,2.727-1.5A19.711,19.711,0,0,1,17.906,22.856Zm-7.354-2.883-.006.019a6.519,6.519,0,0,0-.218,1.154,7.148,7.148,0,0,0-.009,1.154.436.436,0,0,1-.4.47H9.883a.435.435,0,0,1-.435-.407,8.248,8.248,0,0,1,.009-1.3v0a7.007,7.007,0,0,1,.246-1.3.447.447,0,0,1,.426-.33.431.431,0,0,1,.124.019A.439.439,0,0,1,10.552,19.973Zm2.124,7.361a.452.452,0,0,1-.127.019.441.441,0,0,1-.42-.311,5.455,5.455,0,0,1-.174-.777v-.009a2.986,2.986,0,0,1,0-.765.436.436,0,0,1,.432-.376c.022,0,.04,0,.062,0a.44.44,0,0,1,.373.494,2.009,2.009,0,0,0,0,.547,4.285,4.285,0,0,0,.143.631A.438.438,0,0,1,12.676,27.334Zm1.4.924a.386.386,0,0,1-.118.016.434.434,0,0,1-.42-.32,2.848,2.848,0,0,1-.075-.3.436.436,0,0,1,.339-.516.444.444,0,0,1,.09-.009.435.435,0,0,1,.426.351c.016.068.034.149.062.246A.436.436,0,0,1,14.072,28.258Zm3.187-3.175a.439.439,0,0,1-.578.224A5.6,5.6,0,0,1,14.1,22.887a7.2,7.2,0,0,1-1.135-3.638.437.437,0,0,1,.423-.448H13.4a.437.437,0,0,1,.438.426,6.382,6.382,0,0,0,1,3.2l.009.012a4.758,4.758,0,0,0,2.186,2.065A.437.437,0,0,1,17.259,25.083Z" transform="translate(-8.35 -16.5)" fill="#fff"/><path id="Path_30607" data-name="Path 30607" d="M46.317,23.34a3.021,3.021,0,0,0-.7-.09,2.046,2.046,0,0,0-1.287.429L44.3,23.7a9.255,9.255,0,0,0-3,3.3.435.435,0,0,1-.23.193,9.327,9.327,0,0,1-2.348,2.637,4.7,4.7,0,0,1-.485.395c0,.006,0,.012-.006.019a.424.424,0,0,1-.236.159,26.731,26.731,0,0,1-3.24,1.841,12.621,12.621,0,0,0-1.263,1.06l-.016.016a5.112,5.112,0,0,0-1.586,2.31,2.039,2.039,0,0,0-.075.557,4.992,4.992,0,0,0,1.213,2.8,7.554,7.554,0,0,0,3.856,2.7l.037.012a5.033,5.033,0,0,0,1.732.3q3.1,0,7.218-3.5h0q4.595-4.585,4.549-9.043Q50.3,24.623,46.317,23.34ZM34.7,38.369a.5.5,0,0,1-.134.019.44.44,0,0,1-.417-.3c-.075-.239-.137-.47-.183-.7-.056-.289-.131-.706-.224-1.256a.439.439,0,0,1,.361-.5.29.29,0,0,1,.072-.006.437.437,0,0,1,.432.367c.09.538.162.945.218,1.222a6.209,6.209,0,0,0,.159.6A.437.437,0,0,1,34.7,38.369Zm5.143,1.561a.432.432,0,0,1-.606.124,3.11,3.11,0,0,1-.42-.32,1.742,1.742,0,0,1-.317-.4L38.48,39.3a.43.43,0,0,1,.168-.572.455.455,0,0,1,.224-.059.431.431,0,0,1,.376.215l.016.028a.859.859,0,0,0,.137.171l.016.012a2.2,2.2,0,0,0,.3.23A.433.433,0,0,1,39.839,39.93Zm3.318-1.58a5.026,5.026,0,0,1-.47.236l-.022.006a1.729,1.729,0,0,1-.448.115c-.006,0-.009,0-.016,0-.037,0-.081.006-.124.006a1.947,1.947,0,0,1-.333-.031l-.006,0a1.977,1.977,0,0,1-.46-.137l-.019-.009a.436.436,0,0,1-.205-.569.441.441,0,0,1,.4-.261.454.454,0,0,1,.177.037l.019.009a1.1,1.1,0,0,0,.236.068h0a1.013,1.013,0,0,0,.177.022H42.1l.022,0a1.081,1.081,0,0,0,.249-.072,2.767,2.767,0,0,0,.364-.183.433.433,0,0,1,.594.171A.434.434,0,0,1,43.157,38.351Zm.435-10.806a6.026,6.026,0,0,1-1.723,3.281,14.2,14.2,0,0,1-1.86,1.617l-.006,0a10.138,10.138,0,0,1-2.04.958.427.427,0,0,1-.152.028.438.438,0,0,1-.152-.849,9.55,9.55,0,0,0,1.844-.855,13.284,13.284,0,0,0,1.745-1.518l0,0a5.152,5.152,0,0,0,1.477-2.814.439.439,0,0,1,.432-.361.3.3,0,0,1,.075.006A.438.438,0,0,1,43.593,27.544Z" transform="translate(-24.515 -21.151)" fill="#fff"/></g><line id="Line_1" data-name="Line 1" x2="9.435" y2="9.435" transform="translate(774.5 405.5)" fill="none" stroke="#fff" stroke-width="2"/><line id="Line_2" data-name="Line 2" x2="7.435" y2="7.435" transform="translate(757.065 388.065)" fill="none" stroke="#fff" stroke-width="2"/></g></svg>',

			'sugar-free'        => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs>sugar-free</defs><g id="Group_4615" data-name="Group 4615" transform="translate(-751 -311)"><g id="Group_4430" data-name="Group 4430" transform="translate(-468 -2425)"><circle id="Ellipse_97" data-name="Ellipse 97" cx="19" cy="19" r="19" transform="translate(1219 2736)" fill="#ab5a35"/></g><line id="Line_1" data-name="Line 1" x2="7.435" y2="7.435" transform="translate(776.5 335.5)" fill="none" stroke="#fff" stroke-width="2"/><line id="Line_2" data-name="Line 2" x2="7.435" y2="7.435" transform="translate(757.065 316.065)" fill="none" stroke="#fff" stroke-width="2"/><path id="Path_30592" data-name="Path 30592" d="M42.089,56.093l-7.876-1.079L29.14,50.84l7.614.982Z" transform="translate(736.96 270.449)" fill="#fff"/><path id="Path_30593" data-name="Path 30593" d="M44.087,59.7,40.16,69.888,30.92,68.6l.09-.225,3.822-9.944Z" transform="translate(736.514 268.546)" fill="#fff"/><path id="Path_30594" data-name="Path 30594" d="M30.929,56.018,27.07,66.15,21,61.182,24.934,51.05Z" transform="translate(739 270.396)" fill="#fff"/></g></svg>',

			'egg-free'          => '<svg xmlns="http://www.w3.org/2000/svg" width="38" height="38" viewBox="0 0 38 38"><defs>egg-free</defs><g id="Group_4614" data-name="Group 4614" transform="translate(-751 -238)"><g id="Group_4597" data-name="Group 4597" transform="translate(-468 -2498)"><circle id="Ellipse_97" data-name="Ellipse 97" cx="19" cy="19" r="19" transform="translate(1219 2736)" fill="#dc9f65"/><path id="Path_30591" data-name="Path 30591" d="M28.17,10.729c-3.967,0-7.181,4.462-7.181,9.966s3.214,9.967,7.181,9.967S35.349,26.2,35.349,20.7,32.135,10.729,28.17,10.729Zm3.211,10.6c.91-1.385-.593-8.228-.593-8.228,1.977,2.137,2.6,7.234,2.6,7.234Z" transform="translate(1210.011 2734.271)" fill="#fff"/></g><line id="Line_3" data-name="Line 3" x2="38" transform="translate(757.065 243.065) rotate(45)" fill="none" stroke="#fff" stroke-width="2"/></g></svg>',

			'kosher'            => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>kosher</defs> <rect width="38" height="38" rx="19" fill="#2DB68D"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M11.5085 19.2464C11.5085 14.6944 15.2049 11 19.7569 11C24.3089 11 28.0053 14.6944 28.0053 19.2464C28.0053 23.7984 24.3089 27.4928 19.7569 27.4928C15.2049 27.4928 11.5085 23.7984 11.5085 19.2464ZM20.4266 25.1678C20.4826 25.1059 20.5286 25.0399 20.5685 24.9699L21.804 22.8309H24.6608C24.8687 22.8309 25.1525 22.7529 25.3125 22.481C25.4704 22.2071 25.4004 21.9232 25.2945 21.7373L23.8631 19.2624L25.1406 17.0534C25.1805 16.9814 25.2165 16.9054 25.2425 16.8255C25.4044 16.3177 25.1286 15.8359 24.6048 15.7199C24.5228 15.702 24.4389 15.694 24.3569 15.694H21.804L20.4166 13.289C20.1167 12.7692 19.451 12.7692 19.1512 13.289L17.7598 15.696H15.2089C15.1269 15.696 15.0429 15.704 14.961 15.7219C14.4372 15.8379 14.1613 16.3197 14.3233 16.8275C14.3492 16.9074 14.3852 16.9834 14.4252 17.0554L15.6987 19.2624L14.2713 21.7353C14.1653 21.9213 14.0954 22.2051 14.2533 22.479C14.4132 22.7509 14.6971 22.8289 14.905 22.8289L17.7618 22.8309L18.9992 24.9699C19.0392 25.0399 19.0852 25.1059 19.1412 25.1678C19.501 25.5697 20.0668 25.5697 20.4266 25.1678ZM19.7829 24.2322L18.9732 22.8309H20.5905L19.7829 24.2322ZM15.4568 21.7813L16.3044 20.3139C16.5703 20.7718 16.8961 21.3355 17.154 21.7813H15.4568ZM18.3675 21.7813L18.2256 21.5334C17.8917 20.9577 17.3499 20.0201 17.0181 19.4443L16.9121 19.2624L17.0181 19.0805C17.3499 18.5047 17.8917 17.5671 18.2256 16.9914L18.3675 16.7455L18.5774 16.7435H21.1963L21.3422 16.9934C21.6761 17.5691 22.2158 18.5047 22.5477 19.0825L22.6496 19.2624L22.5477 19.4423C22.2158 20.0201 21.6761 20.9557 21.3422 21.5314L21.1963 21.7813H18.3675ZM23.2574 20.3139L24.107 21.7813H22.4097C22.6676 21.3355 22.9935 20.7718 23.2574 20.3139ZM16.3044 18.2129L15.4588 16.7455L17.154 16.7435C16.8941 17.1913 16.5703 17.7551 16.3044 18.2129ZM22.4097 16.7435H24.105L23.2574 18.2129C22.9935 17.7551 22.6676 17.1913 22.4097 16.7435ZM18.9732 15.694L19.7829 14.2946L20.5905 15.694H18.9732Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M8.62573 26.8911C8.64972 23.0708 8.87762 19.1525 8.73369 15.58C8.73369 15.3081 8.87163 15.0742 9.06954 14.9823L9.07154 16.3317C9.07154 16.5576 9.14751 16.7415 9.23747 16.7415H9.23947C9.32943 16.7415 9.40539 16.5576 9.40539 16.3317V14.9643C9.57732 15.0243 9.71326 15.1882 9.76524 15.3981V15.8859L9.76324 15.8239C9.45337 19.4184 9.80922 23.1087 9.90118 26.8151C9.90118 27.169 9.55133 27.4948 9.26146 27.4948H9.25946C8.96958 27.4948 8.62573 27.245 8.62573 26.8911Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M29.8426 27.0031C30.3664 21.6434 30.1525 17.6731 29.9585 12.8792C29.9585 12.6713 30.1305 12.4994 30.3384 12.4994H30.6203C30.8282 12.4994 31.0001 12.6713 31.0001 12.8792V27.115C31.0001 27.3229 30.8282 27.4948 30.6203 27.4948H30.3384C30.1305 27.4948 29.8426 27.211 29.8426 27.0031Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M29.2808 19.0565C28.781 17.981 28.883 13.6688 30.1624 12.5553L30.3043 20.4959C30.3523 19.9661 29.5447 19.6203 29.2808 19.0565Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M8 16.0818C8 15.3142 8.006 13.8208 8.2359 13.1331C8.27588 13.0152 8.32186 12.9212 8.37584 12.8452L8.37784 16.3317C8.37784 16.5576 8.4538 16.7415 8.54376 16.7415C8.63372 16.7415 8.71169 16.5576 8.71169 16.3317V12.6473C8.71169 12.6333 8.70969 12.6193 8.70969 12.6053C8.82964 12.5713 8.95359 12.5733 9.07353 12.5793C9.07153 12.6013 9.07153 12.6253 9.07153 12.6473V16.3317C9.07153 16.5576 9.1475 16.7415 9.23746 16.7415H9.23946C9.32942 16.7415 9.40539 16.5576 9.40539 16.3317V12.6473C9.40539 12.6253 9.40339 12.6013 9.40139 12.5793C9.52134 12.5753 9.64728 12.5713 9.76523 12.5993C9.76523 12.6153 9.76523 12.6313 9.76523 12.6473V16.3317C9.76523 16.5576 9.8412 16.7415 9.93116 16.7415H9.93316C10.0231 16.7415 10.0991 16.5576 10.0991 16.3317V12.8152C10.1711 12.9072 10.235 13.0331 10.289 13.2051C10.4929 13.8648 10.4909 15.3681 10.4909 16.0818C10.4909 17.5972 9.93316 18.0789 9.24546 18.0789C8.55776 18.0789 8 17.5972 8 16.0818Z" fill="white"/> </svg> ',

			'low-fat'           => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>low-fat</defs> <path d="M38 19C38 29.4934 29.4934 38 19 38C8.50659 38 0 29.4934 0 19C0 8.50659 8.50659 0 19 0C29.4934 0 38 8.50659 38 19Z" fill="#FF6E4E"/> <path d="M21.2864 9.7565C20.5232 9.7565 18.9842 11.7673 18.0152 13.1543C17.1761 11.7863 15.5326 9.29733 14.6839 9.29733C13.9682 9.29733 12.6889 11.0453 11.7389 12.5115C10.9441 13.7402 9.60772 16.036 9.60772 17.6225C9.60772 20.2793 11.6597 22.4612 14.2596 22.6765C14.7821 26.0838 17.7334 28.7027 21.2832 28.7027C25.2036 28.7027 28.3892 25.5138 28.3892 21.5967C28.3892 19.3642 26.4734 16.074 25.3302 14.3102C24.8394 13.5502 22.3282 9.75967 21.2801 9.75967L21.2864 9.7565ZM10.8744 17.6193C10.8744 15.732 13.6642 11.7103 14.6807 10.7065C15.1937 11.2322 16.3021 12.654 17.2394 14.3038C16.1311 16.017 14.2976 19.1647 14.1899 21.3908C12.3216 21.147 10.8744 19.5478 10.8744 17.6162V17.6193ZM21.2864 27.4328C18.0659 27.4328 15.4471 24.814 15.4471 21.5935C15.4471 18.5408 20.0609 12.2107 21.2864 11.1213C22.5119 12.2107 27.1257 18.5408 27.1257 21.5935C27.1257 24.814 24.5069 27.4328 21.2864 27.4328ZM21.7709 24.6335C24.4214 23.959 24.2852 21.755 24.2631 21.5048C24.2282 21.1565 24.4816 20.8462 24.8299 20.8113C25.1814 20.7733 25.4854 21.0235 25.5234 21.3718C25.6564 22.5878 25.1086 25.0927 22.0844 25.8622C22.0306 25.8748 21.9799 25.8812 21.9292 25.8812C21.6474 25.8812 21.3877 25.6912 21.3149 25.403C21.2294 25.0642 21.4321 24.719 21.7741 24.6335H21.7709ZM32.4046 5.529C28.8136 1.96333 24.0572 0 19.0032 0H18.9526C8.47406 0.0253333 -0.0252769 8.57217 5.64851e-05 19.0475C0.0253898 29.507 8.54367 38 19 38H19.0476C29.5229 37.9747 38.0254 29.4278 38.0001 18.9525C37.9874 13.8763 35.9987 9.1105 32.4014 5.53217L32.4046 5.529ZM19.0476 36.7302H19.0032C14.3609 36.7302 10.1271 34.9315 6.96356 31.9992L12.8219 26.1155C13.0689 25.8685 13.0689 25.4663 12.8219 25.2193C12.5749 24.9723 12.1727 24.9723 11.9257 25.2193L6.06739 31.103C3.10656 27.9427 1.28572 23.7025 1.27306 19.0412C1.24139 9.26567 9.17706 1.292 18.9557 1.26667H19.0032C23.5094 1.26667 27.7622 2.94183 31.0492 5.99133L25.9731 11.0928C25.7261 11.3398 25.7261 11.742 25.9731 11.989C26.0966 12.1125 26.2581 12.1727 26.4196 12.1727C26.5811 12.1727 26.7457 12.1093 26.8692 11.9858L31.9454 6.88433C35.0234 10.1682 36.7207 14.4273 36.7334 18.9525C36.7587 28.7312 28.8231 36.7048 19.0444 36.7302H19.0476Z" fill="white"/> </svg>',

			'low-sugar'         => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>low-sugar</defs> <rect x="0.500122" y="0.5" width="37" height="37" rx="18.5" fill="white"/> <rect x="0.500122" y="0.5" width="37" height="37" rx="18.5" stroke="#E49B4C"/> <mask id="mask0_215_3967" style="mask-type:alpha" maskUnits="userSpaceOnUse" x="0" y="0" width="39" height="38"> <rect x="0.00012207" width="38" height="38" rx="19" fill="#2DB68D"/> </mask> <g mask="url(#mask0_215_3967)"> <rect x="-0.999878" y="19" width="39" height="19" fill="#E49B4C"/> </g> <g clip-path="url(#clip0_215_3967)"> <path d="M16.1877 19L21.8127 19L21.8127 26.5L26.3502 26.5L19.0002 33.85L11.6502 26.5L16.1877 26.5L16.1877 19Z" fill="white"/> </g> <path d="M14.216 8V4.36364H14.7647V7.5277H16.4124V8H14.216ZM20.0554 6.18182C20.0554 6.57008 19.9844 6.90388 19.8424 7.18324C19.7003 7.46141 19.5056 7.67566 19.2582 7.82599C19.012 7.97514 18.732 8.04972 18.4184 8.04972C18.1035 8.04972 17.8224 7.97514 17.575 7.82599C17.3288 7.67566 17.1346 7.46082 16.9926 7.18146C16.8505 6.90211 16.7795 6.56889 16.7795 6.18182C16.7795 5.79356 16.8505 5.46035 16.9926 5.18217C17.1346 4.90282 17.3288 4.68857 17.575 4.53942C17.8224 4.38909 18.1035 4.31392 18.4184 4.31392C18.732 4.31392 19.012 4.38909 19.2582 4.53942C19.5056 4.68857 19.7003 4.90282 19.8424 5.18217C19.9844 5.46035 20.0554 5.79356 20.0554 6.18182ZM19.5121 6.18182C19.5121 5.88589 19.4642 5.63672 19.3683 5.4343C19.2736 5.23071 19.1434 5.07682 18.9777 4.97266C18.8131 4.86731 18.6267 4.81463 18.4184 4.81463C18.2088 4.81463 18.0218 4.86731 17.8573 4.97266C17.6927 5.07682 17.5625 5.23071 17.4667 5.4343C17.372 5.63672 17.3246 5.88589 17.3246 6.18182C17.3246 6.47775 17.372 6.72751 17.4667 6.93111C17.5625 7.13352 17.6927 7.28741 17.8573 7.39276C18.0218 7.49692 18.2088 7.54901 18.4184 7.54901C18.6267 7.54901 18.8131 7.49692 18.9777 7.39276C19.1434 7.28741 19.2736 7.13352 19.3683 6.93111C19.4642 6.72751 19.5121 6.47775 19.5121 6.18182ZM21.4075 8L20.399 4.36364H20.9761L21.6845 7.17969H21.7183L22.4551 4.36364H23.0268L23.7637 7.18146H23.7974L24.5041 4.36364H25.083L24.0727 8H23.5205L22.7552 5.27628H22.7268L21.9615 8H21.4075Z" fill="#E49B4C"/> <path d="M10.3586 12.3732C10.3387 12.1727 10.2534 12.0169 10.1026 11.9059C9.95177 11.7949 9.7471 11.7393 9.48858 11.7393C9.31292 11.7393 9.1646 11.7642 9.04362 11.8139C8.92265 11.862 8.82985 11.9291 8.76521 12.0153C8.70224 12.1014 8.67075 12.1992 8.67075 12.3086C8.66744 12.3997 8.6865 12.4793 8.72793 12.5472C8.77101 12.6152 8.82985 12.674 8.90442 12.7237C8.97899 12.7718 9.06517 12.814 9.16294 12.8505C9.26072 12.8853 9.36512 12.9151 9.47615 12.94L9.93354 13.0494C10.1556 13.0991 10.3594 13.1654 10.545 13.2482C10.7307 13.3311 10.8914 13.433 11.0273 13.554C11.1632 13.675 11.2684 13.8175 11.343 13.9815C11.4192 14.1456 11.4582 14.3337 11.4598 14.5458C11.4582 14.8574 11.3786 15.1275 11.2212 15.3562C11.0654 15.5832 10.84 15.7597 10.545 15.8857C10.2517 16.0099 9.89791 16.0721 9.48361 16.0721C9.07262 16.0721 8.71467 16.0091 8.40975 15.8832C8.10648 15.7572 7.8695 15.5708 7.69881 15.3239C7.52977 15.0753 7.44111 14.7679 7.43283 14.4016H8.47438C8.48598 14.5723 8.53486 14.7148 8.62104 14.8292C8.70887 14.9419 8.8257 15.0272 8.97154 15.0852C9.11903 15.1416 9.28557 15.1697 9.47118 15.1697C9.65347 15.1697 9.81173 15.1432 9.94597 15.0902C10.0819 15.0372 10.1871 14.9634 10.2617 14.869C10.3362 14.7745 10.3735 14.666 10.3735 14.5433C10.3735 14.429 10.3396 14.3329 10.2716 14.255C10.2053 14.1771 10.1075 14.1108 9.97828 14.0561C9.85068 14.0014 9.69407 13.9517 9.50847 13.907L8.95413 13.7678C8.52492 13.6634 8.18602 13.5001 7.93744 13.2781C7.68886 13.056 7.5654 12.7569 7.56706 12.3807C7.5654 12.0724 7.64744 11.8031 7.81315 11.5728C7.98053 11.3424 8.21005 11.1626 8.50172 11.0334C8.79339 10.9041 9.12483 10.8395 9.49604 10.8395C9.87388 10.8395 10.2037 10.9041 10.4854 11.0334C10.7688 11.1626 10.9892 11.3424 11.1466 11.5728C11.304 11.8031 11.3852 12.07 11.3902 12.3732H10.3586ZM15.3283 10.9091H16.4047V14.2152C16.4047 14.5864 16.316 14.9112 16.1387 15.1896C15.963 15.468 15.7169 15.6851 15.4004 15.8409C15.0839 15.995 14.7152 16.0721 14.2942 16.0721C13.8717 16.0721 13.5021 15.995 13.1856 15.8409C12.8691 15.6851 12.623 15.468 12.4473 15.1896C12.2716 14.9112 12.1838 14.5864 12.1838 14.2152V10.9091H13.2602V14.1232C13.2602 14.3171 13.3024 14.4895 13.3869 14.6403C13.4731 14.7911 13.5941 14.9096 13.7499 14.9957C13.9056 15.0819 14.0871 15.125 14.2942 15.125C14.5031 15.125 14.6845 15.0819 14.8386 14.9957C14.9944 14.9096 15.1146 14.7911 15.1991 14.6403C15.2852 14.4895 15.3283 14.3171 15.3283 14.1232V10.9091ZM20.666 12.5547C20.6312 12.4337 20.5823 12.3268 20.5193 12.234C20.4563 12.1396 20.3793 12.06 20.2881 11.9954C20.1986 11.9291 20.0959 11.8786 19.9799 11.8438C19.8655 11.8089 19.7388 11.7915 19.5996 11.7915C19.3394 11.7915 19.1107 11.8562 18.9135 11.9854C18.7179 12.1147 18.5655 12.3028 18.4561 12.5497C18.3467 12.795 18.292 13.0949 18.292 13.4496C18.292 13.8042 18.3459 14.1058 18.4536 14.3544C18.5613 14.603 18.7138 14.7927 18.911 14.9237C19.1082 15.0529 19.341 15.1175 19.6095 15.1175C19.8531 15.1175 20.0611 15.0745 20.2334 14.9883C20.4074 14.9004 20.54 14.777 20.6312 14.6179C20.724 14.4588 20.7704 14.2707 20.7704 14.0536L20.9891 14.0859H19.6766V13.2756H21.8069V13.9169C21.8069 14.3643 21.7125 14.7488 21.5236 15.0703C21.3346 15.3902 21.0745 15.6371 20.743 15.8111C20.4116 15.9834 20.0321 16.0696 19.6045 16.0696C19.1273 16.0696 18.708 15.9644 18.3467 15.7539C17.9854 15.5418 17.7037 15.241 17.5015 14.8516C17.301 14.4605 17.2008 13.9964 17.2008 13.4595C17.2008 13.0469 17.2604 12.679 17.3797 12.3558C17.5007 12.031 17.6697 11.7559 17.8868 11.5305C18.1039 11.3052 18.3567 11.1336 18.645 11.016C18.9334 10.8983 19.2457 10.8395 19.5822 10.8395C19.8705 10.8395 20.139 10.8817 20.3876 10.9663C20.6361 11.0491 20.8565 11.1668 21.0488 11.3192C21.2427 11.4717 21.4009 11.6532 21.5236 11.8636C21.6462 12.0724 21.7249 12.3028 21.7597 12.5547H20.666ZM23.3357 16H22.1823L23.9398 10.9091H25.3268L27.0818 16H25.9284L24.6532 12.0724H24.6134L23.3357 16ZM23.2636 13.9989H25.988V14.8391H23.2636V13.9989ZM27.6945 16V10.9091H29.7031C30.0875 10.9091 30.4157 10.9779 30.6874 11.1154C30.9609 11.2513 31.1689 11.4444 31.3114 11.6946C31.4556 11.9432 31.5276 12.2357 31.5276 12.5721C31.5276 12.9102 31.4547 13.201 31.3089 13.4446C31.1631 13.6866 30.9518 13.8722 30.675 14.0014C30.3999 14.1307 30.0668 14.1953 29.6757 14.1953H28.3309V13.3303H29.5017C29.7072 13.3303 29.8779 13.3021 30.0138 13.2457C30.1497 13.1894 30.2508 13.1049 30.3171 12.9922C30.385 12.8795 30.419 12.7395 30.419 12.5721C30.419 12.4031 30.385 12.2605 30.3171 12.1445C30.2508 12.0285 30.1489 11.9407 30.0113 11.881C29.8754 11.8197 29.7039 11.7891 29.4967 11.7891H28.7709V16H27.6945ZM30.4438 13.6832L31.7091 16H30.5209L29.283 13.6832H30.4438Z" fill="black"/> <defs> <clipPath id="clip0_215_3967"> <rect width="15" height="15" fill="white" transform="translate(26.5001 34) rotate(180)"/> </clipPath> </defs> </svg> ',

			'low-sodium'        => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>low-sodium</defs> <rect x="0.00012207" width="38" height="38" rx="19" fill="#58C3FF"/> <path d="M32.4047 5.53533C28.82 1.96333 24.0605 0 19.0002 0H18.9558C13.8797 0.0126667 9.11385 1.99817 5.53235 5.5955C1.95402 9.19283 -0.0124838 13.9682 0.000182882 19.0443C0.0223495 29.507 8.54068 38 19.0002 38H19.0445C24.1207 37.9873 28.8865 36.0018 32.468 32.4045C36.0495 28.8072 38.0128 24.0318 38.0033 18.9557C37.9939 13.8795 36.0052 9.11367 32.4078 5.53217L32.4047 5.53533ZM31.5687 31.5115C28.2278 34.8682 23.7787 36.7238 19.0413 36.7333H19.0002C14.3547 36.7333 10.1208 34.9347 6.95418 31.9992L10.789 28.1453C11.036 27.8983 11.036 27.4962 10.789 27.2492C10.542 27.0022 10.1398 27.0022 9.89285 27.2492L6.06118 31.103C3.10352 27.9427 1.28268 23.6993 1.27002 19.038C1.26052 14.3007 3.09402 9.84517 6.43485 6.4885C9.77568 3.13183 14.2248 1.27617 18.959 1.26667H19.0002C23.5095 1.26667 27.7623 2.94183 31.0493 5.99767L25.3683 11.7008C25.1213 11.9478 25.1213 12.35 25.3683 12.597C25.4918 12.7205 25.6533 12.7807 25.8148 12.7807C25.9763 12.7807 26.141 12.7173 26.2645 12.5938L31.9455 6.89067C35.0235 10.1777 36.7208 14.4368 36.7303 18.962C36.7398 23.6993 34.9063 28.1548 31.5655 31.5115H31.5687ZM22.4803 12.5115C21.6633 11.4063 20.403 10.7698 19.0255 10.7698C17.648 10.7698 16.3908 11.4032 15.5707 12.5115L8.52802 22.0242C8.13218 22.5593 7.99602 23.0565 8.11635 23.5442C8.30635 24.3137 9.10118 24.7855 9.76935 25.1085C9.83902 25.1433 9.91502 25.175 9.98785 25.2035C13.031 26.4163 16.071 27.0243 19.092 27.0243C22.113 27.0243 25.0548 26.429 27.9808 25.2352C28.1265 25.175 28.2627 25.1085 28.3925 25.0293C29.0512 24.6335 29.7415 24.13 29.8903 23.389C29.979 22.9393 29.8555 22.4802 29.5167 22.0242L22.474 12.5115H22.4803ZM27.7433 23.9432C27.6705 23.9875 27.5914 24.0255 27.5058 24.0603C21.942 26.3308 16.2072 26.3182 10.4628 24.0255C10.4153 24.0065 10.371 23.9875 10.3267 23.9653C9.48752 23.56 9.36718 23.3067 9.35135 23.237C9.32918 23.1452 9.40202 22.9773 9.55085 22.7747L16.5935 13.262C17.1698 12.483 18.0597 12.0333 19.0287 12.0333C19.9977 12.0333 20.8875 12.4798 21.4638 13.262L28.5065 22.7747C28.5857 22.8823 28.6775 23.0343 28.6553 23.1388C28.6332 23.2465 28.4875 23.4967 27.7465 23.94L27.7433 23.9432ZM13.7783 22.3472C13.7783 22.7842 13.4237 23.1388 12.9867 23.1388C12.5497 23.1388 12.195 22.7842 12.195 22.3472C12.195 21.9102 12.5497 21.5555 12.9867 21.5555C13.4237 21.5555 13.7783 21.9102 13.7783 22.3472ZM15.6783 19.1805C15.6783 19.6175 15.3237 19.9722 14.8867 19.9722C14.4497 19.9722 14.095 19.6175 14.095 19.1805C14.095 18.7435 14.4497 18.3888 14.8867 18.3888C15.3237 18.3888 15.6783 18.7435 15.6783 19.1805ZM18.2117 23.6138C18.2117 24.0508 17.857 24.4055 17.42 24.4055C16.983 24.4055 16.6283 24.0508 16.6283 23.6138C16.6283 23.1768 16.983 22.8222 17.42 22.8222C17.857 22.8222 18.2117 23.1768 18.2117 23.6138Z" fill="white"/> </svg> ',

			'diabetic-friendly' => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>diabetic-friendly</defs> <rect x="0.00338745" width="38" height="38" rx="19" fill="#8EBB5C"/> <path d="M19.2754 8C18.6594 8.82133 18.0727 9.584 17.5154 10.288C13.2914 15.7147 11.0034 18.7067 11.0034 21.728C11.0034 26.304 14.7287 30 19.2754 30C23.8221 30 27.5474 26.2747 27.5474 21.728C27.5474 18.736 25.2301 15.744 21.0354 10.3173C20.4781 9.584 19.8914 8.82133 19.2754 8ZM24.3501 20.6427C24.3794 20.672 24.3794 20.7013 24.3794 20.76V23.0187C24.3794 23.048 24.3501 23.1067 24.3501 23.136C24.3501 23.1653 24.2914 23.1653 24.2327 23.1653H20.5367V26.8613C20.5367 26.9493 20.4781 27.008 20.3901 27.008H18.1314C18.0434 27.008 17.9847 26.9493 17.9847 26.8613V23.1653H14.2887C14.2594 23.1653 14.2007 23.136 14.1714 23.136C14.1421 23.136 14.1421 23.0773 14.1421 23.0187V20.76C14.1421 20.672 14.2007 20.6133 14.2887 20.6133H17.9847V16.9173C17.9847 16.888 18.0141 16.8293 18.0141 16.8C18.0141 16.7707 18.0727 16.7707 18.1314 16.7707H20.3901C20.4781 16.7707 20.5367 16.8293 20.5367 16.9173V20.6133H24.2327C24.2914 20.6133 24.3207 20.6133 24.3501 20.6427Z" fill="white"/> </svg> ',

			'grain-free'        => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>grain-free</defs> <rect x="0.00665283" width="38" height="38" rx="19" fill="#FF7B5F"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M19.4476 8.18095C19.3357 8.06104 19.1947 8 19.0304 8C18.8662 8 18.7252 8.06104 18.6126 8.18095C16.8648 10.0479 16.8648 12.3029 18.6126 14.1699C18.7245 14.2898 18.8662 14.3508 19.0304 14.3508C19.1947 14.3508 19.3357 14.2898 19.4476 14.1699C21.1961 12.3029 21.1961 10.0479 19.4476 8.18095ZM18.4731 16.2846C18.4731 15.9765 18.7223 15.7272 19.0304 15.7272C19.3379 15.7272 19.5878 15.9765 19.5878 16.2846V30.4426C19.5878 30.75 19.3386 31 19.0304 31C18.7223 31 18.4731 30.7507 18.4731 30.4426V16.2846ZM12.0328 22.7684C11.9849 22.6115 12.0031 22.4581 12.0852 22.3164C12.1673 22.174 12.2908 22.0824 12.4507 22.0453C14.9419 21.4654 16.8946 22.5926 17.6373 25.0402C17.6853 25.1971 17.6678 25.3497 17.5857 25.4922C17.5036 25.6346 17.3801 25.7262 17.2202 25.7632C14.729 26.3432 12.7763 25.216 12.0336 22.7692L12.0328 22.7684ZM12.0328 18.302C11.9849 18.145 12.0031 17.9917 12.0852 17.85C12.1673 17.7075 12.2908 17.616 12.4507 17.5789C14.9419 16.999 16.8946 18.1261 17.6373 20.5737C17.6853 20.7307 17.6678 20.8833 17.5857 21.0258C17.5036 21.1682 17.3801 21.2598 17.2202 21.2968C14.729 21.8767 12.7763 20.7496 12.0336 18.302H12.0328ZM12.0328 13.8356C11.9849 13.6786 12.0031 13.5253 12.0852 13.3836C12.1673 13.2411 12.2908 13.1495 12.4507 13.1125C14.9419 12.5326 16.8946 13.6597 17.6373 16.1073C17.6853 16.2643 17.6678 16.4169 17.5857 16.5593C17.5036 16.7018 17.3801 16.7933 17.2202 16.8304C14.729 17.4103 12.7763 16.2832 12.0336 13.8363L12.0328 13.8356ZM26.0273 22.7677C26.0753 22.6107 26.0571 22.4574 25.975 22.3157C25.8929 22.1732 25.7693 22.0817 25.6095 22.0446C23.1183 21.4647 21.1656 22.5918 20.4221 25.0394C20.3749 25.1964 20.3923 25.349 20.4744 25.4915C20.5566 25.6339 20.6801 25.7255 20.84 25.7625C23.3312 26.3424 25.2839 25.2153 26.0266 22.7684L26.0273 22.7677ZM26.0273 18.3013C26.0753 18.1443 26.0571 17.991 25.975 17.8493C25.8929 17.7068 25.7693 17.6152 25.6095 17.5782C23.1183 16.9983 21.1656 18.1254 20.4221 20.573C20.3749 20.73 20.3923 20.8826 20.4744 21.025C20.5566 21.1675 20.6801 21.259 20.84 21.2961C23.3312 21.876 25.2839 20.7489 26.0266 18.3013H26.0273ZM26.0273 13.8348C26.0753 13.6779 26.0571 13.5245 25.975 13.3828C25.8929 13.2404 25.7693 13.1488 25.6095 13.1118C23.1183 12.5318 21.1656 13.659 20.4221 16.1066C20.3749 16.2635 20.3923 16.4162 20.4744 16.5586C20.5566 16.701 20.6801 16.7926 20.84 16.8297C23.3312 17.4096 25.2839 16.2824 26.0266 13.8356L26.0273 13.8348Z" fill="white"/> <path d="M33.7137 6.70709L27.7137 12.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M12.7137 27.7071L6.71374 33.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> </svg> ',

			'shellfish-free'    => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>shellfish-free</defs> <rect x="0.00991821" width="38" height="38" rx="19" fill="#F15701"/> <path d="M32.717 5.70709L26.717 11.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M11.717 26.7071L5.71701 32.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M21.883 20.9046C24.0589 19.8806 25.4572 17.7015 25.4572 15.2697C25.4572 15.0297 25.262 14.8377 25.022 14.8377C24.7821 14.8377 24.5869 15.0297 24.5869 15.2697C24.5869 17.2344 23.5277 19.0103 21.8414 19.951C21.739 19.2663 21.3678 18.6679 20.8431 18.2647C22.187 17.6823 23.1597 16.396 23.2941 14.8729C23.6685 14.7161 23.9981 14.3417 24.2413 13.7818C24.5229 13.1354 24.6797 12.2875 24.6797 11.3915C24.6797 9.92278 24.2605 8.63326 23.6109 8.10209C23.4861 7.99969 23.3165 7.97729 23.1693 8.03809C23.0222 8.10209 22.9198 8.23968 22.907 8.40287C22.8014 9.75319 22.3022 10.5787 21.9886 10.5787C21.9342 10.5787 21.8702 10.5499 21.803 10.4956C21.6782 10.3932 21.5086 10.3708 21.3614 10.4316C21.2143 10.4956 21.1119 10.6331 21.0991 10.7963C21.0831 10.9915 21.0767 11.1899 21.0767 11.3915C21.0767 12.2875 21.2335 13.1354 21.515 13.7818C21.7518 14.3257 22.0686 14.6937 22.4302 14.8569C22.2606 16.4696 20.8911 17.7335 19.2304 17.7335C17.5601 17.7335 16.1841 16.4536 16.0273 14.8249C16.3601 14.6457 16.6513 14.2905 16.8753 13.7818C17.1569 13.1354 17.3137 12.2875 17.3137 11.3915C17.3137 11.1931 17.3073 10.9915 17.2913 10.7963C17.2785 10.6363 17.1793 10.4956 17.0289 10.4316C16.8817 10.3676 16.7089 10.3932 16.5873 10.4956C16.5201 10.5499 16.4561 10.5787 16.4017 10.5787C16.2545 10.5787 16.0625 10.3868 15.9026 10.0732C15.685 9.654 15.5378 9.05883 15.4834 8.39967C15.4706 8.23968 15.3714 8.09889 15.221 8.03489C15.0738 7.97089 14.901 7.99649 14.7794 8.09889C14.1299 8.63006 13.7107 9.91958 13.7107 11.3883C13.7107 12.2843 13.8675 13.1322 14.1491 13.7786C14.4082 14.3737 14.7634 14.7609 15.1698 14.8985C15.3106 16.4216 16.2929 17.7015 17.6465 18.2743C17.1217 18.6807 16.7569 19.2823 16.6609 19.9702C15.925 19.567 15.2946 19.0007 14.8146 18.2999C14.2003 17.4072 13.8739 16.3576 13.8739 15.2665C13.8739 15.0265 13.6819 14.8345 13.4419 14.8345C13.2019 14.8345 13.0099 15.0265 13.0099 15.2665C13.0099 16.5304 13.3907 17.7495 14.1043 18.7895C14.7442 19.7206 15.6178 20.4534 16.6417 20.9238V26.4051C16.6417 27.1282 16.9361 27.7842 17.4161 28.2578L15.4226 30.2512C15.2978 30.376 15.2626 30.5648 15.3266 30.728C15.3938 30.888 15.5506 31 15.7234 31H22.827C23.003 31 23.1597 30.888 23.2269 30.728C23.2941 30.568 23.2557 30.3792 23.1341 30.2544L21.1279 28.2386C21.595 27.7682 21.8894 27.1186 21.8894 26.4083V20.9046H21.883ZM22.3022 13.433C22.075 12.9146 21.9438 12.1915 21.9374 11.4427C21.9534 11.4427 21.9694 11.4427 21.9886 11.4427C22.6862 11.4427 23.2653 10.7611 23.5661 9.6604C23.7293 10.182 23.8125 10.8219 23.8125 11.3915C23.8125 12.1595 23.6781 12.905 23.4477 13.4362C23.2589 13.8714 23.0542 14.0538 22.9198 14.0889C22.9038 14.0889 22.891 14.0858 22.875 14.0858C22.859 14.0858 22.843 14.0858 22.8302 14.0889C22.6958 14.0538 22.4942 13.8682 22.3022 13.433ZM15.5602 14.0858C15.5378 14.0858 15.5186 14.089 15.4962 14.0921C15.3618 14.0826 15.141 13.9066 14.9362 13.433C14.7026 12.9018 14.5714 12.1563 14.5714 11.3883C14.5714 10.7099 14.6706 10.118 14.8178 9.6604C15.1186 10.7611 15.6978 11.4427 16.3953 11.4427C16.4113 11.4427 16.4273 11.4427 16.4465 11.4427C16.4401 12.1915 16.3089 12.9146 16.0817 13.433C15.893 13.8618 15.6946 14.0474 15.5602 14.0858ZM17.4993 20.6358V20.3318C17.4993 19.3751 18.2704 18.5847 19.2272 18.5847H19.272C20.2287 18.5847 21.0191 19.3751 21.0191 20.3318V20.6198V20.623V22.0629C21.0191 23.0196 20.2287 23.8004 19.272 23.8004H19.2272C18.2704 23.8004 17.4993 23.0196 17.4993 22.0629V20.6358ZM17.4993 23.9956C17.9472 24.4084 18.5616 24.6644 19.2272 24.6644H19.272C19.9375 24.6644 20.5391 24.4116 21.0191 23.9956V26.4051C21.0191 26.5203 20.9999 26.6354 20.9775 26.7442C20.8271 27.5442 20.1135 28.1522 19.256 28.1522C18.6736 28.1522 18.1616 27.8738 17.8448 27.445C17.6273 27.1538 17.4993 26.7954 17.4993 26.4051V23.9956ZM21.7806 30.136H16.7633L18.1264 28.7665C18.4624 28.9329 18.8496 29.0321 19.2656 29.0321C19.6847 29.0321 20.0751 28.9329 20.4143 28.7633L21.7806 30.136Z" fill="white"/> </svg> ',

			'legume-free'       => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>legume-free</defs> <rect x="0.00991821" width="38" height="38" rx="19" fill="#F59C7B"/> <path d="M32.717 5.70709L26.717 11.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M11.717 26.7071L5.71701 32.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M25.1574 29.2373C27.2276 28.6786 28.9208 26.8928 29.3693 24.7805C29.7957 22.6856 28.9857 20.3653 27.3233 18.9874C26.9755 18.706 26.5858 18.4397 26.0797 18.1397C25.9111 18.0397 25.7622 17.9421 25.5912 17.8618C24.7455 17.3817 23.937 16.9259 23.4006 16.2014C22.9341 15.5652 22.7059 14.7767 22.4825 13.9486C22.4243 13.7613 22.3858 13.5764 22.3276 13.3891C22.2694 13.2019 22.2136 12.9949 22.1531 12.8274C21.4117 10.7758 19.8993 9.3152 17.9719 8.82826C16.0643 8.34363 13.9347 8.89541 12.559 10.1972C11.5682 11.1233 10.8962 12.4075 10.6383 13.9208C10.622 14.0592 10.6081 14.1779 10.5919 14.3163C10.5361 14.7909 10.5176 15.2898 10.541 15.7737C10.653 17.5509 11.2815 19.3688 12.4741 21.3331C13.8168 23.5555 15.555 25.4836 17.5966 27.0464C18.8483 27.9953 19.9569 28.6266 21.0736 29.0184C22.4799 29.5043 23.8955 29.5703 25.1574 29.2373ZM11.5607 14.0893C11.7954 12.7737 12.3651 11.6779 13.1989 10.8938C14.349 9.80605 16.1459 9.35554 17.7418 9.76346C19.3575 10.1737 20.636 11.4064 21.2342 13.1405C21.2494 13.1823 21.2622 13.244 21.2773 13.2858C20.2446 13.8862 19.5609 14.9285 19.4216 16.115C19.1777 18.1912 20.6611 20.0693 22.7374 20.3132C23.9238 20.4525 25.1182 20.0114 25.9415 19.1459C26.2417 19.3214 26.4801 19.5098 26.7186 19.6982C28.1076 20.8436 28.7989 22.8091 28.4318 24.5702C28.0623 26.351 26.6436 27.8483 24.9059 28.3057C23.829 28.6002 22.6158 28.5179 21.3805 28.1122C20.358 27.7716 19.3239 27.1891 18.1664 26.2914C16.2166 24.7996 14.5681 22.9622 13.2731 20.8455C12.1654 19.0115 11.5602 17.3367 11.4692 15.7223C11.4528 15.1792 11.4782 14.6208 11.5607 14.0893ZM25.0958 18.6657C25.0272 18.7378 24.9609 18.7902 24.8749 18.8402C24.8528 18.8576 24.8528 18.8576 24.8307 18.8751C24.7644 18.9274 24.6783 18.9775 24.6144 19.0101C24.5946 19.0077 24.5923 19.0275 24.5725 19.0252C24.5085 19.0578 24.4225 19.1078 24.3585 19.1404C24.3387 19.1381 24.3364 19.1578 24.3166 19.1555C24.2329 19.1858 24.1492 19.216 24.0654 19.2463C24.0457 19.244 24.0236 19.2614 24.0038 19.2591C23.92 19.2894 23.8386 19.2998 23.7549 19.3301L23.7351 19.3278C23.6537 19.3383 23.5723 19.3487 23.4909 19.3592C23.4711 19.3569 23.4513 19.3546 23.449 19.3744C23.3676 19.3848 23.2664 19.393 23.1873 19.3837C23.1675 19.3814 23.1477 19.3791 23.128 19.3768C23.0291 19.3651 22.9477 19.3756 22.8488 19.364C21.3041 19.2027 20.1872 17.7885 20.3707 16.2264C20.3823 16.1276 20.3916 16.0485 20.423 15.9519C20.4253 15.9321 20.4276 15.9124 20.4299 15.8926C20.459 15.8158 20.4706 15.7169 20.4997 15.6402C20.502 15.6204 20.5043 15.6006 20.5241 15.603C20.5532 15.5262 20.5822 15.4494 20.6113 15.3726L20.6136 15.3529C20.6427 15.2761 20.6915 15.2016 20.7403 15.1272C20.7427 15.1074 20.7648 15.09 20.7671 15.0702C20.8159 14.9957 20.8647 14.9213 20.9136 14.8468C20.9159 14.8271 20.9357 14.8294 20.938 14.8096C20.9845 14.7549 21.0333 14.6805 21.0996 14.6281C21.102 14.6084 21.1217 14.6107 21.1241 14.5909C21.1903 14.5385 21.2392 14.4641 21.3055 14.4117C21.3276 14.3943 21.3276 14.3943 21.3497 14.3768C21.4159 14.3245 21.4822 14.2721 21.5683 14.2221C21.8069 15.0921 22.063 15.9841 22.6342 16.7528C23.2928 17.632 24.2129 18.1611 25.0958 18.6657Z" fill="white"/> </svg> ',

			'caffeine-free'     => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>caffeine-free</defs> <rect x="0.0098877" width="38" height="38" rx="19" fill="#701B09" fill-opacity="0.79"/> <path d="M32.717 5.70709L26.717 11.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M11.717 26.7071L5.71699 32.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M25.9432 25.8905C29.0924 22.7081 28.5696 17.0429 24.768 13.2414C22.9386 11.4118 20.6265 10.2763 18.2572 10.0442C15.839 9.80645 13.6493 10.5339 12.0925 12.0911C12.0854 12.0982 12.0791 12.1057 12.072 12.1128C12.0706 12.1144 12.069 12.1159 12.0676 12.1175C8.91848 15.3 9.44126 20.9653 13.2428 24.7668C15.3754 26.8994 18.0937 28.0002 20.6322 28C22.6311 27.9998 24.5186 27.3168 25.9183 25.9169C25.9254 25.9099 25.9317 25.9023 25.9388 25.8952C25.9402 25.8936 25.9418 25.8921 25.9432 25.8905ZM17.3741 11.3342C17.6212 11.3342 17.8722 11.3466 18.1269 11.3716C20.1898 11.5738 22.2133 12.5727 23.825 14.1844C26.7525 17.1119 27.4823 21.2627 25.7393 24.015C25.4108 23.0449 24.8649 21.9782 23.6685 20.782C21.9269 19.0404 20.5231 18.6928 19.1649 18.3564C17.9301 18.0505 16.7638 17.7616 15.2848 16.2826C13.9584 14.956 13.5896 13.8808 13.3069 12.7821C14.3871 11.8365 15.7997 11.3342 17.3741 11.3342ZM14.1858 23.8238C11.2592 20.8971 10.5289 16.7478 12.2701 13.9953C12.5982 14.9746 13.1352 16.019 14.3418 17.2256C16.0831 18.967 17.5499 19.3304 18.845 19.6513C20.0798 19.957 21.2462 20.2459 22.7255 21.7251C24.0528 23.0525 24.4215 24.1278 24.7041 25.2267C21.9987 27.5935 17.3687 27.0063 14.1858 23.8238Z" fill="white"/> </svg> ',

			'alcohol-free'      => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>alcohol-free</defs> <rect x="0.0098877" width="38" height="38" rx="19" fill="#FF6999"/> <path d="M32.717 5.70709L26.717 11.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M11.717 26.7071L5.71699 32.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M15.8332 26.0833L17.2538 25.448C17.5557 25.313 17.7665 24.8838 17.7665 24.4042V20.6218C17.5022 20.453 17.2769 20.2256 17.1112 19.9599C15.111 19.1875 13.8482 17.1841 14.0266 15.0347C14.0269 15.0313 14.0272 15.0278 14.0276 15.0243L14.5395 10.3547C14.5616 10.1528 14.7322 10 14.9352 10H22.8821C23.0852 10 23.2557 10.1528 23.2778 10.3547L23.7897 15.0243C23.7901 15.0278 23.7904 15.0313 23.7907 15.0347C23.9691 17.1841 22.7064 19.1875 20.7061 19.9599C20.5404 20.2256 20.3151 20.453 20.0509 20.6218V24.4042C20.0509 24.8838 20.2617 25.313 20.5635 25.448L21.9844 26.0835C22.4401 26.2804 22.54 26.6517 22.4869 26.9103C22.4196 27.2383 22.1171 27.4675 21.7513 27.4675H16.0661C15.7003 27.4675 15.3978 27.2383 15.3304 26.9103C15.2773 26.6516 15.3773 26.2802 15.8332 26.0833ZM15.292 10.7961L15.1371 12.2089H22.6802L22.5253 10.7961H15.292ZM21.3489 26.6714L20.2385 26.1748C19.6501 25.9116 19.2547 25.2001 19.2547 24.4042V20.3883C19.2547 20.2346 19.3432 20.0947 19.482 20.0288C19.7374 19.9074 19.9496 19.7028 20.0794 19.453C20.1261 19.363 20.2053 19.2943 20.3008 19.2609C22.0395 18.6506 23.1479 16.9425 22.9977 15.1057L22.7675 13.005H15.0499L14.8196 15.1057C14.6694 16.9425 15.7778 18.6506 17.5165 19.2609C17.6121 19.2944 17.6913 19.3631 17.7379 19.453C17.8678 19.7028 18.0799 19.9074 18.3354 20.0288C18.4742 20.0947 18.5626 20.2346 18.5626 20.3883V24.4042C18.5626 25.2001 18.1672 25.9116 17.5788 26.1747L16.4685 26.6714H21.3489Z" fill="white"/> </svg> ',

			'high-fiber'        => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>high-fiber</defs> <rect x="0.0098877" width="38" height="38" rx="19" fill="#F59C7B"/> <path d="M18.6215 11.1809C18.1891 11.6921 17.9692 12.256 17.9824 12.8237C17.9956 13.3531 18.2154 13.8593 18.6215 14.2988C19.028 13.8593 19.2478 13.3531 19.2603 12.8237C19.2742 12.256 19.0541 11.6921 18.6215 11.1809ZM18.6215 14.8291C18.5677 14.8291 18.5134 14.8093 18.4711 14.7695C17.8818 14.2147 17.5614 13.5457 17.5443 12.8342C17.5259 12.0814 17.8441 11.3439 18.4636 10.7019C18.505 10.659 18.562 10.6349 18.6215 10.6349C18.6807 10.6349 18.7378 10.659 18.7795 10.7019C19.3993 11.3442 19.7172 12.0816 19.6989 12.8342C19.6818 13.5457 19.3614 14.215 18.7716 14.7695C18.7293 14.8093 18.6755 14.8291 18.6215 14.8291Z" fill="white"/> <path d="M18.6215 31C18.5002 31 18.4021 30.9019 18.4021 30.7808V14.6098C18.4021 14.4888 18.5002 14.3905 18.6215 14.3905C18.7425 14.3905 18.8406 14.4888 18.8406 14.6098V30.7808C18.8406 30.9019 18.7425 31 18.6215 31Z" fill="white"/> <path d="M18.6215 9.48194C18.5002 9.48194 18.4021 9.38359 18.4021 9.26254V6.21938C18.4021 6.09835 18.5002 6 18.6215 6C18.7425 6 18.8406 6.09835 18.8406 6.21938V9.26254C18.8406 9.38359 18.7425 9.48194 18.6215 9.48194Z" fill="white"/> <path d="M20.394 11.8665C20.273 11.8665 20.1746 11.7684 20.1746 11.6474V9.29253C20.1746 9.1715 20.273 9.07315 20.394 9.07315C20.515 9.07315 20.6132 9.1715 20.6132 9.29253V11.6474C20.6132 11.7684 20.515 11.8665 20.394 11.8665Z" fill="white"/> <path d="M16.8494 11.8665C16.7284 11.8665 16.63 11.7684 16.63 11.6474V9.29253C16.63 9.1715 16.7284 9.07315 16.8494 9.07315C16.9704 9.07315 17.0686 9.1715 17.0686 9.29253V11.6474C17.0686 11.7684 16.9704 11.8665 16.8494 11.8665Z" fill="white"/> <path d="M21.3989 26.7425C20.6211 26.8009 19.9747 27.0804 19.5189 27.558C19.0943 28.0043 18.8613 28.6036 18.8406 29.3012C19.5378 29.2803 20.1376 29.0478 20.5836 28.6227C21.0614 28.1673 21.3412 27.5208 21.3989 26.7425ZM18.7377 29.7414C18.6968 29.7414 18.6562 29.7407 18.6151 29.7396C18.499 29.7359 18.4059 29.6428 18.4027 29.527C18.3746 28.6188 18.651 27.8332 19.2015 27.2555C19.7847 26.6439 20.6229 26.3112 21.6251 26.2933C21.6839 26.2919 21.7423 26.3155 21.7845 26.3575C21.8261 26.3996 21.8495 26.457 21.8483 26.5165C21.8308 27.5192 21.4979 28.3572 20.8861 28.9401C20.3345 29.4657 19.5939 29.7414 18.7377 29.7414Z" fill="white"/> <path d="M15.8438 26.7425C15.9019 27.5208 16.1817 28.1673 16.6591 28.6226C17.1058 29.0478 17.7049 29.2803 18.4028 29.3012C18.3814 28.6036 18.1491 28.0043 17.7238 27.558C17.2684 27.0804 16.6221 26.8008 15.8438 26.7425ZM18.5054 29.7414C17.6488 29.7414 16.9082 29.4656 16.3566 28.9399C15.7448 28.3572 15.4125 27.5192 15.3948 26.5165C15.3937 26.457 15.4166 26.3995 15.4589 26.3575C15.5011 26.3155 15.5588 26.2917 15.6176 26.2933C16.6205 26.3112 17.4587 26.6439 18.0412 27.2555C18.5917 27.8332 18.8681 28.6186 18.8406 29.527C18.8372 29.6428 18.7441 29.7359 18.6283 29.7395C18.5872 29.7407 18.5459 29.7414 18.5054 29.7414Z" fill="white"/> <path d="M21.3989 22.9777C20.6211 23.0359 19.9747 23.3154 19.5189 23.7932C19.0943 24.2395 18.8613 24.8388 18.8406 25.5362C19.5378 25.5156 20.1376 25.283 20.5836 24.8577C21.0614 24.4023 21.3412 23.7558 21.3989 22.9777ZM18.7377 25.9766C18.6968 25.9766 18.6562 25.9759 18.6151 25.9745C18.499 25.9711 18.4059 25.878 18.4027 25.762C18.3746 24.8538 18.651 24.0683 19.2015 23.4907C19.7847 22.8791 20.6229 22.5462 21.6251 22.5285C21.6839 22.5271 21.7423 22.5505 21.7845 22.5925C21.8261 22.6348 21.8495 22.6922 21.8483 22.7518C21.8308 23.7542 21.4979 24.5924 20.8861 25.1752C20.3352 25.7007 19.5939 25.9766 18.7377 25.9766Z" fill="white"/> <path d="M15.8438 22.9777C15.9019 23.7558 16.1817 24.4023 16.6591 24.8577C17.1058 25.2828 17.7049 25.5156 18.4028 25.5363C18.3814 24.8388 18.1491 24.2395 17.7238 23.7933C17.2684 23.3154 16.6221 23.0359 15.8438 22.9777ZM18.5054 25.9766C17.6488 25.9766 16.9082 25.7007 16.3566 25.1752C15.7448 24.5922 15.4125 23.7542 15.3948 22.7518C15.3937 22.6923 15.4166 22.6348 15.4589 22.5926C15.5011 22.5505 15.5588 22.5269 15.6176 22.5285C16.6205 22.5462 17.4587 22.8792 18.0412 23.4905C18.5917 24.0683 18.8681 24.8538 18.8406 25.762C18.8372 25.8781 18.7441 25.9712 18.6283 25.9746C18.5872 25.9759 18.5459 25.9766 18.5054 25.9766Z" fill="white"/> <path d="M21.3989 19.2125C20.6211 19.2709 19.9747 19.5505 19.5189 20.0281C19.0943 20.4743 18.8613 21.0736 18.8406 21.7713C19.5385 21.7504 20.1376 21.5179 20.5836 21.0927C21.0614 20.6374 21.3412 19.9908 21.3989 19.2125ZM18.7377 22.2114C18.6968 22.2114 18.6562 22.2107 18.6151 22.2096C18.499 22.206 18.4059 22.1129 18.4027 21.997C18.3746 21.0886 18.651 20.3033 19.2015 19.7256C19.7847 19.114 20.6229 18.7813 21.6251 18.7633C21.6839 18.762 21.7423 18.7856 21.7845 18.8276C21.8261 18.8696 21.8495 18.9271 21.8483 18.9866C21.8308 19.9892 21.4979 20.8272 20.8861 21.4102C20.3352 21.9355 19.5939 22.2114 18.7377 22.2114Z" fill="white"/> <path d="M15.8438 19.2125C15.9019 19.9908 16.1817 20.6372 16.6591 21.0927C17.1058 21.5179 17.7049 21.7504 18.4028 21.7713C18.3814 21.0736 18.1491 20.4743 17.7238 20.0281C17.2684 19.5505 16.6221 19.2709 15.8438 19.2125ZM18.5054 22.2114C17.6488 22.2114 16.9082 21.9355 16.3566 21.41C15.7448 20.8272 15.4125 19.9892 15.3948 18.9866C15.3937 18.9271 15.4166 18.8696 15.4589 18.8276C15.5011 18.7856 15.5588 18.762 15.6176 18.7633C16.6205 18.7813 17.4587 19.114 18.0412 19.7256C18.5917 20.3031 18.8681 21.0886 18.8406 21.997C18.8372 22.1129 18.7441 22.206 18.6283 22.2096C18.5872 22.2107 18.5459 22.2114 18.5054 22.2114Z" fill="white"/> <path d="M21.3989 15.4477C20.6211 15.5058 19.9747 15.7854 19.5189 16.2632C19.0943 16.7092 18.8613 17.3085 18.8406 18.0062C19.5385 17.9855 20.1376 17.7527 20.5836 17.3276C21.0614 16.8723 21.3412 16.2257 21.3989 15.4477ZM18.7377 18.4463C18.6968 18.4463 18.6562 18.4456 18.6151 18.4445C18.499 18.4409 18.4059 18.348 18.4027 18.2319C18.3746 17.3238 18.651 16.5382 19.2015 15.9605C19.7847 15.3491 20.6229 15.0162 21.6251 14.9984C21.6839 14.9971 21.7423 15.0205 21.7845 15.0625C21.8261 15.1047 21.8495 15.162 21.8483 15.2215C21.8308 16.2241 21.4979 17.0623 20.8861 17.6451C20.3352 18.1706 19.5939 18.4463 18.7377 18.4463Z" fill="white"/> <path d="M15.8438 15.4476C15.9019 16.2257 16.1817 16.8723 16.6591 17.3276C17.1058 17.7527 17.7049 17.9853 18.4028 18.0062C18.3814 17.3085 18.1491 16.7092 17.7238 16.2632C17.2684 15.7853 16.6221 15.5058 15.8438 15.4476ZM18.5054 18.4463C17.6488 18.4463 16.9082 18.1706 16.3566 17.6451C15.7448 17.0621 15.4125 16.2241 15.3948 15.2215C15.3937 15.162 15.4166 15.1047 15.4589 15.0625C15.5011 15.0205 15.5588 14.9968 15.6176 14.9984C16.6205 15.0162 17.4587 15.3491 18.0412 15.9604C18.5917 16.5382 18.8681 17.3237 18.8406 18.2319C18.8372 18.348 18.7441 18.4408 18.6283 18.4445C18.5872 18.4456 18.5459 18.4463 18.5054 18.4463Z" fill="white"/> <path d="M10.6139 15.3098C10.3016 15.6954 10.1429 16.1169 10.1531 16.5403C10.1622 16.9343 10.3209 17.3119 10.6139 17.6433C10.9075 17.3119 11.0656 16.9343 11.0754 16.5403C11.0856 16.1169 10.9268 15.6954 10.6139 15.3098ZM10.6139 18.177C10.5603 18.177 10.506 18.1572 10.4638 18.1175C9.98752 17.6692 9.72818 17.1273 9.71456 16.5507C9.69957 15.9414 9.95596 15.3455 10.4563 14.8272C10.4974 14.7845 10.5546 14.7602 10.6139 14.7602C10.6734 14.7602 10.7304 14.7845 10.7722 14.8272C11.2718 15.3455 11.5282 15.9414 11.5139 16.5507C11.4996 17.1273 11.2409 17.6692 10.764 18.1175C10.7224 18.1572 10.6681 18.177 10.6139 18.177Z" fill="white"/> <path d="M10.6139 31C10.4928 31 10.3947 30.9019 10.3947 30.7808V17.9578C10.3947 17.8365 10.4928 17.7384 10.6139 17.7384C10.7349 17.7384 10.8332 17.8365 10.8332 17.9578V30.7808C10.8332 30.9019 10.7349 31 10.6139 31Z" fill="white"/> <path d="M10.6139 13.9367C10.4928 13.9367 10.3947 13.8386 10.3947 13.7176V11.3044C10.3947 11.1831 10.4928 11.085 10.6139 11.085C10.7349 11.085 10.8332 11.1831 10.8332 11.3044V13.7176C10.8332 13.8386 10.7349 13.9367 10.6139 13.9367Z" fill="white"/> <path d="M12.0192 15.8278C11.8984 15.8278 11.8 15.7295 11.8 15.6084V13.7412C11.8 13.6202 11.8984 13.5221 12.0192 13.5221C12.1404 13.5221 12.2386 13.6202 12.2386 13.7412V15.6084C12.2386 15.7295 12.1404 15.8278 12.0192 15.8278Z" fill="white"/> <path d="M9.20857 15.8278C9.08752 15.8278 8.98919 15.7295 8.98919 15.6084V13.7412C8.98919 13.6202 9.08752 13.5221 9.20857 13.5221C9.32962 13.5221 9.42773 13.6202 9.42773 13.7412V15.6084C9.42773 15.7295 9.32962 15.8278 9.20857 15.8278Z" fill="white"/> <path d="M12.7668 27.6283C12.185 27.6822 11.7017 27.8972 11.3585 28.2567C11.0395 28.5919 10.8605 29.0393 10.8349 29.5603C11.3563 29.5351 11.8034 29.3561 12.1386 29.0366C12.4984 28.6939 12.7132 28.2104 12.7668 27.6283ZM10.7065 30.0022C10.6734 30.0022 10.6407 30.0015 10.6075 30.0006C10.4917 29.9972 10.3986 29.9039 10.3947 29.7881C10.3729 29.0554 10.5962 28.4211 11.0417 27.9542C11.5123 27.4603 12.1877 27.1916 12.9953 27.1771C13.0541 27.1764 13.1122 27.1993 13.154 27.2413C13.1963 27.2833 13.2196 27.3408 13.2185 27.4003C13.2037 28.2077 12.9353 28.8833 12.4414 29.3541C11.9953 29.779 11.3969 30.0022 10.7065 30.0022Z" fill="white"/> <path d="M8.46096 27.6283C8.51477 28.2104 8.73007 28.6939 9.08912 29.0366C9.42431 29.3561 9.87216 29.5351 10.3929 29.5603C10.3679 29.0393 10.1885 28.5919 9.86922 28.2567C9.52674 27.8972 9.04302 27.6822 8.46096 27.6283ZM10.5219 30.0022C9.83105 30.0022 9.2331 29.779 8.78707 29.3541C8.29312 28.8833 8.02425 28.2077 8.00993 27.4003C8.0088 27.3408 8.03219 27.2833 8.07375 27.2413C8.11598 27.1993 8.17321 27.1764 8.23318 27.1771C9.04052 27.1916 9.71614 27.4603 10.1867 27.9542C10.6323 28.4211 10.8555 29.0554 10.8332 29.7881C10.8298 29.9039 10.7367 29.9972 10.6207 30.0006C10.5878 30.0015 10.5546 30.0022 10.5219 30.0022Z" fill="white"/> <path d="M12.7668 24.6427C12.185 24.6966 11.7013 24.9116 11.3585 25.2711C11.0395 25.6063 10.8605 26.0537 10.8349 26.5747C11.3563 26.5495 11.8034 26.3703 12.1386 26.051C12.4984 25.7083 12.7132 25.2248 12.7668 24.6427ZM10.7065 27.0164C10.6734 27.0164 10.6407 27.016 10.6075 27.015C10.4917 27.0114 10.3986 26.9183 10.3947 26.8025C10.3729 26.0698 10.5962 25.4356 11.041 24.9686C11.5123 24.4747 12.1877 24.206 12.9953 24.1915C13.0541 24.1899 13.1122 24.2135 13.154 24.2557C13.1962 24.2978 13.2196 24.3552 13.2185 24.4147C13.2037 25.2221 12.9353 25.8977 12.4414 26.3685C11.9953 26.7934 11.3974 27.0164 10.7065 27.0164Z" fill="white"/> <path d="M8.46096 24.6427C8.51477 25.2248 8.73007 25.7083 9.08912 26.051C9.42431 26.3703 9.87216 26.5495 10.3929 26.5747C10.3679 26.0537 10.1885 25.6063 9.86922 25.2711C9.52674 24.9116 9.04302 24.6966 8.46096 24.6427ZM10.5219 27.0164C9.83105 27.0164 9.2331 26.7934 8.78707 26.3685C8.29312 25.8977 8.02425 25.2221 8.00993 24.4147C8.0088 24.3552 8.03219 24.2978 8.07375 24.2557C8.11598 24.2135 8.17321 24.1899 8.23318 24.1915C9.04052 24.206 9.71614 24.4747 10.1867 24.9686C10.6323 25.4356 10.8555 26.0698 10.8332 26.8025C10.8298 26.9183 10.7367 27.0114 10.6207 27.015C10.5878 27.016 10.5546 27.0164 10.5219 27.0164Z" fill="white"/> <path d="M12.7668 21.6573C12.185 21.7112 11.7017 21.926 11.3585 22.2857C11.0395 22.6207 10.8605 23.0683 10.8349 23.5893C11.3563 23.5641 11.8034 23.3849 12.1386 23.0656C12.4984 22.7227 12.7132 22.2394 12.7668 21.6573ZM10.7065 24.031C10.6734 24.031 10.6407 24.0303 10.6075 24.0294C10.4917 24.026 10.3986 23.9326 10.3947 23.8168C10.3729 23.0842 10.5962 22.4501 11.0417 21.983C11.5123 21.489 12.1877 21.2204 12.9953 21.2058C13.0541 21.2049 13.1122 21.2281 13.154 21.2701C13.1963 21.3121 13.2196 21.3696 13.2185 21.4291C13.2037 22.2366 12.9353 22.9123 12.4414 23.3831C11.9953 23.808 11.3969 24.031 10.7065 24.031Z" fill="white"/> <path d="M8.46096 21.6573C8.51477 22.2394 8.73007 22.7227 9.08912 23.0656C9.42431 23.3849 9.87216 23.5641 10.3929 23.5893C10.3679 23.0683 10.1885 22.6207 9.86922 22.2857C9.52674 21.926 9.04302 21.7112 8.46096 21.6573ZM10.5219 24.031C9.83105 24.031 9.2331 23.808 8.78707 23.3831C8.29312 22.9123 8.02425 22.2367 8.00993 21.4291C8.0088 21.3696 8.03219 21.3121 8.07375 21.2701C8.11598 21.2281 8.17321 21.2047 8.23318 21.2058C9.04052 21.2204 9.71614 21.4891 10.1867 21.983C10.6323 22.4501 10.8555 23.0842 10.8332 23.8169C10.8298 23.9327 10.7367 24.026 10.6207 24.0294C10.5878 24.0303 10.5546 24.031 10.5219 24.031Z" fill="white"/> <path d="M12.7668 18.6718C12.185 18.7254 11.7017 18.9405 11.3585 19.3002C11.0395 19.6352 10.8605 20.0826 10.8349 20.6038C11.3563 20.5783 11.8034 20.3994 12.1386 20.0799C12.4984 19.7372 12.7132 19.2537 12.7668 18.6718ZM10.7065 21.0455C10.6734 21.0455 10.6407 21.0448 10.6075 21.0439C10.4917 21.0403 10.3986 20.9472 10.3947 20.8313C10.3729 20.0987 10.5962 19.4646 11.041 18.9975C11.5123 18.5035 12.1877 18.2349 12.9953 18.2204C13.0541 18.2197 13.1122 18.2426 13.154 18.2846C13.1962 18.3266 13.2196 18.3841 13.2185 18.4436C13.2037 19.2512 12.9353 19.9268 12.4414 20.3973C11.9953 20.8223 11.3969 21.0455 10.7065 21.0455Z" fill="white"/> <path d="M8.46096 18.6718C8.51477 19.2537 8.73007 19.7372 9.08912 20.0799C9.42431 20.3994 9.87216 20.5783 10.3929 20.6038C10.3679 20.0826 10.1885 19.6352 9.86922 19.3002C9.52674 18.9405 9.04302 18.7254 8.46096 18.6718ZM10.5219 21.0455C9.83105 21.0455 9.2331 20.8225 8.78707 20.3973C8.29312 19.9268 8.02425 19.2512 8.00993 18.4436C8.0088 18.3841 8.03219 18.3266 8.07375 18.2846C8.11598 18.2426 8.17321 18.2197 8.23318 18.2204C9.04052 18.2349 9.71614 18.5035 10.1867 18.9975C10.6323 19.4644 10.8555 20.0987 10.8332 20.8313C10.8298 20.9472 10.7367 21.0403 10.6207 21.0439C10.5878 21.0448 10.5546 21.0455 10.5219 21.0455Z" fill="white"/> <path d="M26.6289 15.3098C26.3165 15.6954 26.1577 16.1169 26.1675 16.5403C26.1772 16.9343 26.336 17.3119 26.6289 17.6433C26.9224 17.3119 27.0806 16.9343 27.0904 16.5403C27.1006 16.1169 26.9419 15.6954 26.6289 15.3098ZM26.6289 18.177C26.5753 18.177 26.5208 18.1572 26.4788 18.1175C26.0024 17.6692 25.743 17.1273 25.7294 16.5507C25.7146 15.9416 25.971 15.3455 26.4713 14.8272C26.5124 14.7845 26.5694 14.7602 26.6289 14.7602C26.6889 14.7602 26.7454 14.7845 26.787 14.8272C27.2868 15.3455 27.5433 15.9416 27.5289 16.5507C27.5146 17.1273 27.256 17.6692 26.779 18.1175C26.7375 18.1572 26.6832 18.177 26.6289 18.177Z" fill="white"/> <path d="M26.6289 31C26.5079 31 26.4096 30.9019 26.4096 30.7808V17.9578C26.4096 17.8365 26.5079 17.7384 26.6289 17.7384C26.75 17.7384 26.8483 17.8365 26.8483 17.9578V30.7808C26.8483 30.9019 26.75 31 26.6289 31Z" fill="white"/> <path d="M26.6289 13.9367C26.5079 13.9367 26.4096 13.8386 26.4096 13.7176V11.3044C26.4096 11.1831 26.5079 11.085 26.6289 11.085C26.75 11.085 26.8483 11.1831 26.8483 11.3044V13.7176C26.8483 13.8386 26.75 13.9367 26.6289 13.9367Z" fill="white"/> <path d="M28.0341 15.8278C27.9131 15.8278 27.815 15.7295 27.815 15.6084V13.7412C27.815 13.6202 27.9131 13.5221 28.0341 13.5221C28.1556 13.5221 28.2533 13.6202 28.2533 13.7412V15.6084C28.2533 15.7295 28.1556 15.8278 28.0341 15.8278Z" fill="white"/> <path d="M25.2236 15.8278C25.1023 15.8278 25.0042 15.7295 25.0042 15.6084V13.7412C25.0042 13.6202 25.1023 13.5221 25.2236 13.5221C25.3444 13.5221 25.4428 13.6202 25.4428 13.7412V15.6084C25.4428 15.7295 25.3444 15.8278 25.2236 15.8278Z" fill="white"/> <path d="M28.7824 27.6283C28.1988 27.6822 27.7166 27.8972 27.3739 28.2567C27.0542 28.5919 26.8755 29.0393 26.8498 29.5603C27.3712 29.5351 27.8184 29.3561 28.1534 29.0366C28.5122 28.6939 28.7279 28.2104 28.7824 27.6283ZM26.7213 30.0022C26.6881 30.0022 26.6556 30.0015 26.6225 30.0006C26.5066 29.9972 26.4135 29.9039 26.4101 29.7881C26.3879 29.0554 26.6111 28.4211 27.0565 27.9542C27.527 27.4603 28.2033 27.1916 29.0095 27.1771C29.0686 27.1764 29.1276 27.1993 29.1685 27.2413C29.2117 27.2833 29.2343 27.3408 29.2343 27.4003C29.2185 28.2077 28.9505 28.8833 28.4554 29.3541C28.0103 29.779 27.4123 30.0022 26.7213 30.0022Z" fill="white"/> <path d="M24.4759 27.6283C24.5295 28.2104 24.7448 28.6939 25.1041 29.0366C25.4393 29.3561 25.8869 29.5351 26.4079 29.5603C26.3827 29.0393 26.2033 28.5919 25.8842 28.2567C25.5415 27.8972 25.0578 27.6822 24.4759 27.6283ZM26.5369 30.0022C25.8458 30.0022 25.2479 29.779 24.8021 29.3541C24.3081 28.8833 24.039 28.2077 24.0247 27.4003C24.0235 27.3408 24.0472 27.2833 24.0887 27.2413C24.131 27.1993 24.1887 27.1764 24.2481 27.1771C25.0555 27.1916 25.7311 27.4603 26.2017 27.9542C26.647 28.4211 26.8705 29.0554 26.8482 29.7881C26.8446 29.9039 26.7517 29.9972 26.6357 30.0006C26.6025 30.0015 26.5693 30.0022 26.5369 30.0022Z" fill="white"/> <path d="M28.7824 24.6427C28.1988 24.6966 27.7166 24.9116 27.3735 25.2711C27.0542 25.6063 26.8755 26.0537 26.8498 26.5747C27.3712 26.5495 27.8184 26.3703 28.1534 26.051C28.5122 25.7083 28.7279 25.2248 28.7824 24.6427ZM26.7213 27.0164C26.6888 27.0164 26.6556 27.016 26.6225 27.015C26.5066 27.0114 26.4135 26.9183 26.4101 26.8025C26.3879 26.0698 26.6111 25.4356 27.056 24.9686C27.527 24.4747 28.2033 24.206 29.0095 24.1915C29.0686 24.1899 29.1276 24.2135 29.1685 24.2557C29.2117 24.2978 29.2343 24.3552 29.2343 24.4147C29.2185 25.2221 28.9505 25.8977 28.4554 26.3685C28.0103 26.7934 27.4123 27.0164 26.7213 27.0164Z" fill="white"/> <path d="M24.4759 24.6427C24.5295 25.2248 24.7448 25.7083 25.1041 26.051C25.4393 26.3703 25.8869 26.5495 26.4079 26.5747C26.3827 26.0537 26.2033 25.6063 25.8842 25.2711C25.5415 24.9116 25.0578 24.6966 24.4759 24.6427ZM26.5369 27.0164C25.8458 27.0164 25.2479 26.7934 24.8021 26.3685C24.3081 25.8977 24.039 25.2221 24.0247 24.4147C24.0235 24.3552 24.0472 24.2978 24.0887 24.2557C24.131 24.2135 24.1887 24.1899 24.2481 24.1915C25.0555 24.206 25.7311 24.4747 26.2017 24.9686C26.647 25.4356 26.8705 26.0698 26.8482 26.8025C26.8446 26.9183 26.7517 27.0114 26.6357 27.015C26.6025 27.016 26.5693 27.0164 26.5369 27.0164Z" fill="white"/> <path d="M28.7824 21.6573C28.1988 21.7112 27.7166 21.926 27.3735 22.2857C27.0542 22.6207 26.8755 23.0683 26.8498 23.5893C27.3712 23.5641 27.8184 23.3849 28.1534 23.0656C28.5122 22.7227 28.7279 22.2394 28.7824 21.6573ZM26.7213 24.031C26.6881 24.031 26.6556 24.0303 26.6225 24.0294C26.5066 24.026 26.4135 23.9326 26.4101 23.8168C26.3879 23.0842 26.6111 22.4501 27.056 21.983C27.527 21.489 28.2033 21.2204 29.0095 21.2058C29.0686 21.2049 29.1276 21.2281 29.1685 21.2701C29.2117 21.3121 29.2343 21.3696 29.2343 21.4291C29.2185 22.2366 28.9505 22.9123 28.4554 23.3831C28.0103 23.808 27.4123 24.031 26.7213 24.031Z" fill="white"/> <path d="M24.4759 21.6573C24.5295 22.2394 24.7448 22.7227 25.1041 23.0656C25.4393 23.3849 25.8869 23.5641 26.4079 23.5893C26.3827 23.0683 26.2033 22.6207 25.8842 22.2857C25.5415 21.926 25.0578 21.7112 24.4759 21.6573ZM26.5369 24.031C25.8458 24.031 25.2479 23.808 24.8021 23.3831C24.3081 22.9123 24.039 22.2367 24.0247 21.4291C24.0235 21.3696 24.0472 21.3121 24.0887 21.2701C24.131 21.2281 24.1887 21.2047 24.2481 21.2058C25.0555 21.2204 25.7311 21.4891 26.2017 21.983C26.647 22.4501 26.8705 23.0842 26.8482 23.8169C26.8446 23.9327 26.7517 24.026 26.6357 24.0294C26.6025 24.0303 26.5693 24.031 26.5369 24.031Z" fill="white"/> <path d="M28.7824 18.6718C28.1988 18.7254 27.7166 18.9405 27.3735 19.3002C27.0542 19.6352 26.8755 20.0826 26.8498 20.6038C27.3712 20.5783 27.8184 20.3994 28.1534 20.0799C28.5122 19.7372 28.7279 19.2537 28.7824 18.6718ZM26.7213 21.0455C26.6881 21.0455 26.6556 21.0448 26.6225 21.0439C26.5066 21.0403 26.4135 20.9472 26.4101 20.8313C26.3879 20.0987 26.6111 19.4644 27.056 18.9975C27.527 18.5035 28.2033 18.2349 29.0095 18.2204C29.0686 18.2197 29.1276 18.2426 29.1685 18.2846C29.2117 18.3266 29.2343 18.3841 29.2343 18.4436C29.2185 19.2512 28.9505 19.9268 28.4554 20.3973C28.0103 20.8223 27.4123 21.0455 26.7213 21.0455Z" fill="white"/> <path d="M24.4759 18.6718C24.5295 19.2537 24.7448 19.7372 25.1041 20.0799C25.4393 20.3994 25.8869 20.5783 26.4079 20.6038C26.3827 20.0826 26.2033 19.6352 25.8842 19.3002C25.5415 18.9405 25.0578 18.7254 24.4759 18.6718ZM26.5369 21.0455C25.8458 21.0455 25.2479 20.8225 24.8021 20.3973C24.3081 19.9268 24.039 19.2512 24.0247 18.4436C24.0235 18.3841 24.0472 18.3266 24.0887 18.2846C24.131 18.2426 24.1887 18.2197 24.2481 18.2204C25.0555 18.2349 25.7311 18.5035 26.2017 18.9975C26.647 19.4644 26.8705 20.0987 26.8482 20.8313C26.8446 20.9472 26.7517 21.0403 26.6357 21.0439C26.6025 21.0448 26.5693 21.0455 26.5369 21.0455Z" fill="white"/> </svg> ',

			'fructose-free'     => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>fructose-free</defs> <rect width="38" height="38" rx="19" fill="#FFAB48"/> <path d="M32.7071 5.70709L26.7071 11.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M11.7071 26.7071L5.70709 32.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M20.4724 9.6437C20.5621 9.46439 20.5621 9.28507 20.5621 9.10577C20.5621 8.88162 20.5397 8.5454 20.3155 8.27641C20.1138 8.07469 19.8224 7.96262 19.4189 7.96263C19.0155 7.96262 18.7241 8.0747 18.5224 8.27643C18.5224 8.27643 18.5224 8.27643 18.4999 8.29884C18.3206 8.07469 18.0068 7.98504 17.6034 7.98504C16.5723 7.98505 16.4602 8.63506 16.4826 9.10576C16.4826 9.32991 16.5275 9.50922 16.5723 9.64371C16.5274 9.64371 16.4602 9.62129 16.393 9.6437C15.4964 9.64371 15.0929 10.0472 15.1154 10.9213C15.1154 11.1903 15.1378 11.4817 15.3171 11.7058C15.2274 11.7507 15.1153 11.7731 15.0481 11.8403C14.7567 12.0421 14.6222 12.4007 14.5998 12.8714C14.5998 13.3645 14.7567 13.7007 15.0481 13.9024L15.0705 13.9249C14.8016 14.1042 14.5998 14.4404 14.5998 14.9783C14.5998 15.5163 14.8016 15.8525 15.0705 16.0318L15.0481 16.0542C14.7567 16.256 14.6222 16.6146 14.5998 17.0853C14.5998 17.5336 14.7343 17.8922 15.0257 18.0939C14.824 18.206 14.6895 18.4302 14.6446 18.6991C13.8826 18.4302 13.0981 18.3181 12.4256 18.3181C12.2911 18.3181 12.1791 18.4302 12.1791 18.5646C12.1791 18.6991 12.2911 18.8112 12.4256 18.8112C12.9636 18.8112 13.367 18.9457 13.636 19.2147C14.3084 19.8871 14.2636 21.4113 14.1964 23.182C14.1964 23.6751 14.1739 24.1907 14.1515 24.7062C14.1515 26.9925 14.936 28.4494 16.5723 29.1891C16.6395 29.2115 17.3344 29.5477 17.9172 29.5926C19.5086 29.7046 20.6965 29.3684 21.5707 28.5839C21.5931 28.5615 21.6379 28.5167 21.6604 28.4942C22.669 27.4856 23.1397 25.8493 23.1397 23.3389C23.1397 22.6665 23.1173 22.0165 23.1173 21.3889C23.0501 18.9905 23.0276 16.906 23.8121 16.0766C24.0587 15.8301 24.4397 15.6732 24.888 15.6732C25.0225 15.6732 25.1346 15.5611 25.1346 15.4266C25.1346 15.3818 25.1122 15.3145 25.0674 15.2697C25.0225 15.2249 24.9553 15.2025 24.9105 15.2025C24.1035 15.2025 23.0725 15.4266 22.019 16.0318L21.9966 16.0094L22.019 15.987C22.0638 15.9422 22.0862 15.9197 22.1311 15.8749C22.3328 15.6732 22.4449 15.337 22.4449 14.9335C22.4449 14.4404 22.288 14.1042 21.9966 13.9024C21.9966 13.9024 21.9742 13.88 21.9517 13.8576C21.9966 13.8128 22.0638 13.7904 22.1086 13.7455C22.3104 13.5438 22.4224 13.2524 22.4225 12.8041C22.4224 12.0869 22.1086 11.7731 21.6828 11.6162C21.8173 11.4369 21.8845 11.1903 21.8845 10.8317C21.8845 9.97992 21.4586 9.55404 20.5621 9.55405C20.5845 9.6213 20.5397 9.62129 20.4724 9.6437ZM19.8448 11.8851C19.8224 11.9076 19.7776 11.9524 19.7551 11.9748C19.7327 11.9524 19.7103 11.93 19.6879 11.9076C19.6431 11.8627 19.5758 11.7955 19.5086 11.7731C19.531 11.7507 19.531 11.7507 19.5534 11.7283L19.5758 11.7058C19.5982 11.7283 19.6207 11.7507 19.6431 11.7731C19.7103 11.8403 19.7776 11.8627 19.8448 11.8851ZM17.1775 20.2906C17.1999 20.2682 17.2223 20.2457 17.2447 20.2233C17.2671 20.2009 17.2896 20.1785 17.312 20.1561C17.3568 20.2009 17.3792 20.2233 17.424 20.2681C17.4464 20.2906 17.4464 20.2906 17.4689 20.313C17.424 20.3578 17.3792 20.4026 17.3344 20.4475C17.2671 20.3802 17.2223 20.3354 17.1775 20.2906ZM17.4689 11.7507L17.4913 11.7282C17.5137 11.7507 17.5361 11.7731 17.5585 11.7955L17.5809 11.8179C17.5137 11.8403 17.4464 11.9076 17.4016 11.9524C17.3792 11.9748 17.3568 11.9972 17.3344 12.0196C17.312 11.9972 17.2895 11.93 17.2671 11.9076C17.3344 11.8403 17.4016 11.8179 17.4689 11.7507ZM19.5534 12.9386C19.5534 13.3421 19.4862 13.4542 19.3741 13.5214C19.2396 13.6559 18.9706 13.7007 18.5448 13.6783C17.6034 13.6783 17.5137 13.4542 17.5361 12.9386C17.5361 12.6248 17.5809 12.4007 17.7154 12.2662C17.8723 12.1093 18.1189 12.0421 18.5223 12.0421C18.9258 12.0421 19.1948 12.1317 19.3293 12.2662C19.5086 12.4007 19.5758 12.6024 19.5534 12.9386ZM19.5534 15.0456C19.5534 15.5835 19.4637 15.8077 18.5448 15.7853C18.0965 15.7853 17.8275 15.7404 17.693 15.6059C17.6258 15.5387 17.5137 15.4266 17.5137 15.0232C17.5137 14.7094 17.5585 14.4852 17.693 14.3507C17.8499 14.1938 18.0965 14.1266 18.4999 14.1266C19.3965 14.1714 19.5534 14.4628 19.5534 15.0456ZM19.3741 16.5249C19.5086 16.6594 19.5534 16.8836 19.5534 17.1974C19.5534 17.6008 19.4862 17.7129 19.3741 17.7801C19.2396 17.9146 18.9706 17.9595 18.5448 17.937C17.6034 17.937 17.5137 17.7129 17.5361 17.1974C17.5361 16.8836 17.5809 16.6594 17.7154 16.5249C17.8723 16.368 18.1189 16.3008 18.5448 16.3232C18.9482 16.2784 19.2172 16.368 19.3741 16.5249ZM17.5137 13.9473C17.4689 13.9921 17.4464 14.0145 17.4016 14.0593C17.3792 14.0818 17.3568 14.1042 17.3344 14.1266C17.2896 14.0369 17.2223 13.9697 17.1551 13.9473C17.1775 13.9249 17.1775 13.9249 17.1999 13.9024C17.2223 13.88 17.2223 13.88 17.2447 13.8576C17.2671 13.8352 17.2896 13.8128 17.312 13.7904C17.3568 13.8352 17.424 13.9024 17.5137 13.9473ZM17.424 16.0094C17.4464 16.0318 17.4689 16.0542 17.5361 16.0766C17.4689 16.1439 17.424 16.1887 17.3568 16.256C17.312 16.1663 17.2447 16.0991 17.1775 16.0766C17.1999 16.0542 17.1999 16.0542 17.2223 16.0318C17.2447 16.0094 17.2447 16.0094 17.2671 15.987C17.2896 15.9646 17.312 15.9422 17.3344 15.9197C17.312 15.9422 17.3568 15.987 17.424 16.0094ZM17.5137 18.206C17.4689 18.2509 17.424 18.2509 17.3792 18.2957C17.3568 18.3181 17.3344 18.3405 17.312 18.3629C17.2671 18.2733 17.1999 18.206 17.1326 18.1836C17.1551 18.1612 17.1551 18.1612 17.1775 18.1388C17.1999 18.1163 17.1999 18.1163 17.2223 18.0939C17.2447 18.0715 17.2671 18.0491 17.2896 18.0267C17.3568 18.0939 17.424 18.1612 17.5137 18.206ZM17.6482 20.8061C17.7154 20.6044 17.8499 20.5147 18.5672 20.5147C18.612 20.5147 18.6568 20.5147 18.7016 20.5147C18.5672 20.963 18.4551 21.4337 18.3654 21.9268C18.1413 21.5234 17.8947 21.1423 17.6482 20.8061ZM18.8586 20.044C18.7689 20.044 18.6568 20.0664 18.5448 20.044C18.0965 20.044 17.8275 19.9992 17.693 19.8647C17.6258 19.7974 17.5137 19.6854 17.5137 19.2819C17.5137 18.9681 17.5585 18.744 17.693 18.6095C17.8499 18.4526 18.0965 18.3853 18.4999 18.3853C19.1275 18.3853 19.3069 18.5646 19.4189 18.7215C19.2396 19.125 19.0379 19.5509 18.8586 20.044ZM19.5982 18.1836C19.6206 18.1612 19.6431 18.1388 19.7103 18.1164C19.7327 18.0939 19.7551 18.0715 19.7551 18.0715C19.7776 18.0491 19.7776 18.0491 19.8 18.0267C19.8224 18.0491 19.8448 18.0715 19.8672 18.0939C19.8 18.1612 19.7776 18.2284 19.7327 18.3181C19.6879 18.2733 19.6431 18.2284 19.5982 18.1836ZM19.5982 16.0766C19.6431 16.0318 19.7103 16.0094 19.7551 15.9646C19.7776 15.9422 19.7776 15.9422 19.8 15.9197C19.8448 15.9646 19.8672 15.987 19.912 16.0318C19.9344 16.0542 19.9344 16.0542 19.9569 16.0766C19.8896 16.1439 19.8448 16.1887 19.7776 16.256C19.7551 16.2335 19.7327 16.2111 19.7103 16.1887C19.6655 16.1439 19.6431 16.1215 19.5982 16.0766ZM19.5982 13.9697C19.6207 13.9473 19.6879 13.9248 19.7103 13.9024C19.7327 13.88 19.7551 13.8576 19.7551 13.8576C19.7775 13.8352 19.7776 13.8352 19.8 13.8128C19.8448 13.8576 19.8896 13.9024 19.9344 13.9473C19.912 13.9697 19.912 13.9697 19.8896 13.9921C19.8448 14.0369 19.8 14.0818 19.7551 14.1266C19.7327 14.0593 19.6655 13.9921 19.5982 13.9697ZM19.2396 11.4144C19.1275 11.5265 18.9034 11.5713 18.5448 11.5714C18.1861 11.5714 17.962 11.5265 17.8723 11.4368C17.8051 11.3696 17.7379 11.2575 17.7378 10.8989C17.7378 10.6299 17.7827 10.4058 17.8947 10.2937C18.0068 10.1816 18.2085 10.1144 18.5672 10.1144C18.881 10.1144 19.1275 10.1816 19.2396 10.2937C19.3517 10.4058 19.3965 10.5851 19.3965 10.8989C19.3517 11.2576 19.2844 11.3696 19.2396 11.4144ZM18.9034 8.61264C19.0155 8.50057 19.1948 8.45574 19.4413 8.47817C19.7103 8.47815 19.8896 8.52298 20.0017 8.63505C20.0914 8.72471 20.1138 8.88162 20.1362 9.12817C20.1362 9.39715 20.0689 9.50922 20.0241 9.55405C19.9345 9.6437 19.7327 9.66612 19.4638 9.6661C19.1948 9.66612 18.9931 9.64369 18.9034 9.55404C18.8586 9.50921 18.8137 9.41956 18.7913 9.12817C18.7913 8.8592 18.8137 8.70229 18.9034 8.61264ZM16.9758 9.10576C16.9757 8.65747 17.0654 8.47816 17.6258 8.45575C18.2085 8.45575 18.2982 8.63505 18.2982 9.08336C18.2982 9.48681 18.2534 9.62129 17.6258 9.62129C16.9982 9.62129 16.9757 9.50921 16.9758 9.10576ZM15.586 10.8989C15.5861 10.6299 15.6309 10.4058 15.743 10.2937C15.855 10.1816 16.0568 10.1144 16.4154 10.1144C17.1326 10.1144 17.2671 10.3834 17.2447 10.8989C17.2447 11.392 17.1775 11.5489 16.4154 11.5489C16.0568 11.5489 15.8326 11.5041 15.743 11.4144C15.6533 11.3696 15.5861 11.2576 15.586 10.8989ZM15.3395 13.5214C15.2274 13.4542 15.1154 13.2973 15.0929 12.8714C15.0929 12.5128 15.205 12.3558 15.2947 12.2662C15.3171 12.2438 15.3395 12.2214 15.3395 12.2214C15.4964 12.1093 15.7654 12.0645 16.1688 12.0645C16.8861 12.0645 17.0878 12.2214 17.0878 12.8938C17.0878 13.3421 16.9757 13.499 16.9085 13.5662C16.7964 13.6783 16.5723 13.7231 16.1688 13.7231C15.7654 13.6783 15.4964 13.6335 15.3395 13.5214ZM15.0929 14.9783C15.0929 14.7094 15.1602 14.5076 15.2947 14.3732C15.4516 14.2163 15.743 14.149 16.1912 14.149C16.9085 14.149 17.1102 14.3059 17.1102 14.9783C17.1102 15.4266 16.9982 15.5835 16.9309 15.6508C16.8188 15.7628 16.5947 15.8077 16.1912 15.8077C15.3395 15.8077 15.0929 15.6059 15.0929 14.9783ZM15.0929 17.1301C15.0929 16.7715 15.205 16.6146 15.2947 16.5249C15.3171 16.5025 15.3395 16.4801 15.3395 16.4801C15.4964 16.368 15.7654 16.3232 16.1688 16.3232C16.8861 16.3232 17.0878 16.4801 17.0878 17.1525C17.0878 17.6008 16.9982 17.7353 16.9085 17.825C16.7964 17.937 16.5723 17.9819 16.1688 17.9819C15.7654 17.9819 15.4964 17.9371 15.3395 17.825C15.2274 17.6681 15.0929 17.5336 15.0929 17.1301ZM15.3171 18.5198C15.4964 18.4302 15.8326 18.4078 16.1688 18.4302C16.8861 18.4302 17.0878 18.5871 17.0878 19.2595C17.0878 19.7526 16.9757 19.9095 16.8413 19.9544C16.3033 19.5061 15.7205 19.1474 15.1154 18.9009C15.1378 18.6543 15.2274 18.5646 15.3171 18.5198ZM14.6671 24.7286C14.6671 24.2355 14.6895 23.72 14.6895 23.2269C14.7567 21.4561 14.7791 19.9095 14.1067 19.0578C14.3308 19.1026 14.555 19.1923 14.7567 19.2595C14.7567 19.2595 14.7791 19.2819 14.8016 19.2595C16.2136 19.8199 17.5361 20.9182 18.2309 22.7786C18.1637 23.3837 18.1413 23.9889 18.1413 24.6614C18.1413 26.6787 17.5809 27.7322 16.662 28.696C15.3171 28.0235 14.6447 26.7235 14.6671 24.7286ZM22.6242 21.4337C22.6242 22.0613 22.6466 22.7113 22.6466 23.3837C22.6466 25.8045 22.1759 27.3959 21.2569 28.2701C20.5172 28.9649 19.4189 29.2563 17.9844 29.1667C17.693 29.1443 17.3568 29.0322 17.1102 28.9201C18.0068 27.9339 18.5896 26.7683 18.612 24.6838C18.612 19.0802 21.1672 16.7491 23.319 15.987C22.5345 17.0405 22.5794 19.1026 22.6242 21.4337ZM21.5483 16.368C21.0552 16.7267 20.5845 17.1525 20.1362 17.6905C20.0689 17.5784 20.0241 17.4439 20.0241 17.0853C20.0241 16.4129 20.181 16.256 20.9431 16.256C21.1897 16.2784 21.4138 16.3232 21.5483 16.368ZM21.75 14.3283C21.8621 14.3956 21.9742 14.5525 21.9966 14.9783C22.019 15.4042 21.8621 15.5611 21.75 15.6284C21.5931 15.7404 21.3241 15.7853 20.9207 15.7853C20.5172 15.7853 20.3155 15.718 20.181 15.6284C20.1138 15.5611 20.0017 15.4042 20.0017 14.9559C20.0017 14.5076 20.0914 14.3731 20.1586 14.3059C20.1586 14.3059 20.1586 14.3059 20.181 14.2835C20.2931 14.1714 20.5172 14.1266 20.9207 14.1266C21.3241 14.1714 21.5931 14.2162 21.75 14.3283ZM21.9966 12.8714C21.9966 13.499 21.7276 13.6783 20.9207 13.6783C20.1586 13.6783 20.0241 13.5438 20.0241 12.8714C20.0241 12.5576 20.0689 12.3334 20.181 12.2214C20.3155 12.0869 20.5396 12.042 20.9207 12.0645C21.7276 12.0196 21.9966 12.2438 21.9966 12.8714ZM21.5035 10.8989C21.5035 11.392 21.4362 11.5489 20.6741 11.5489C20.3155 11.5489 20.0914 11.5041 20.0017 11.4144C19.9345 11.3472 19.8672 11.2351 19.8672 10.8765C19.8672 10.6075 19.912 10.3834 20.0241 10.2713C20.1362 10.1592 20.3379 10.092 20.6965 10.092C21.369 10.092 21.4811 10.3834 21.5035 10.8989Z" fill="white"/> </svg> ',

			'msg-free'          => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>msg-free</defs> <rect width="38" height="38" rx="19" fill="#2E9BFF"/> <path d="M32.7071 5.70709L26.7071 11.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M10.7071 27.7071L5.7071 32.7071" stroke="white" stroke-width="1.82574" stroke-linecap="round"/> <path d="M14.0455 12C13.8173 12.0216 13.6238 12.235 13.6248 12.4643V14.3214C13.6229 14.4797 13.7083 14.6363 13.8424 14.7204C13.783 16.2166 13.3851 16.9766 12.544 18.2388C12.2858 18.6263 12.1182 19.069 12.058 19.9727C11.9978 20.8763 12 22.3261 12 24.9928C12.0447 25.4916 12.4106 25.9196 12.8632 25.9286H24.0116C24.5403 25.8844 24.9754 25.4321 24.9836 24.9928C25.0116 22.3114 25.0079 20.8629 24.94 19.9582C24.8722 19.0534 24.6914 18.6169 24.4396 18.2388C23.6 16.9789 23.2133 16.2189 23.1557 14.7204C23.2898 14.6363 23.3752 14.4797 23.3733 14.3214V12.4643C23.3733 12.2212 23.1522 12 22.9091 12C19.9556 12 16.9959 12 14.0455 12ZM14.5532 12.9286H22.4449V13.8571H14.5532V12.9286ZM14.749 14.7857H22.249C22.3309 16.4176 22.8051 17.4658 23.6635 18.7539C23.8751 19.0716 23.9568 19.2029 24.0189 20.0307C24.1017 22.0463 24.0696 23.597 24.0696 24.9898C20.3559 24.9898 16.6227 25 12.9284 25C12.9284 23.3648 12.8837 21.6838 12.9865 20.038C13.0421 19.2026 13.1075 19.0621 13.3129 18.754C14.1132 17.3718 14.6731 16.0063 14.749 14.7857ZM15.0537 18.9643C14.7019 18.9643 14.3211 19.2249 14.3211 19.6535V23.3896C14.3211 23.8182 14.7019 24.0714 15.0537 24.0714H21.9444C22.2962 24.0714 22.677 23.8182 22.677 23.3896V19.6535C22.677 19.2249 22.2962 18.9643 21.9444 18.9643H15.0537ZM15.2495 19.8929H21.7486V23.1429H15.2495V19.8929Z" fill="white"/> </svg> ',

			'halal'             => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>halal</defs> <rect width="38" height="38" rx="19" fill="#2DB68D"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M24.0769 15.4615C25.4143 15.462 26.34 16.2079 27.5385 16.8462C28.6051 17.2795 30.0769 17.0769 30.0769 17.0769L29.1538 18.6923C29.1538 18.6923 27.6951 18.4075 25.9231 19.1538C24.326 19.7846 22.8554 19.8893 22.4615 19.8462C21.9508 19.8893 20.8182 19.8153 20.1538 18.4615C19.7095 17.1831 19.6923 15 19.6923 15V13.6154C19.6923 13.6154 19.4586 15.1891 19 16.8462C18.5771 17.7395 18.3981 18.2403 17.8462 18.9231C17.1675 19.7033 16.2208 20.264 15.3077 20.5385C13.7442 20.9161 12.0769 20.7692 12.0769 20.7692C12.0769 20.7692 13.8863 20.1796 15.0769 19.3846C16.2632 18.6636 17.709 17.7395 18.5385 15.6923C19.3191 13.5145 19.4724 11.0385 19.6923 10.6154C20.1064 10.3291 20.338 11.4384 20.3846 12C20.4074 12.6513 20.6154 15.6923 20.6154 15.6923C20.6154 15.6923 20.72 17.9976 21.5385 18.4615C22.2736 18.7291 23.9864 18.5991 25 18.2308C26.0105 17.9717 26.1538 17.7692 26.1538 17.7692C26.1538 17.7692 24.2043 16.6049 23.1538 17.0769C22.2065 17.453 21.8156 17.9717 21.7692 18C21.8156 17.9717 22.6499 15.462 24.0769 15.4615ZM13.6923 11.7692C13.6923 11.7692 13.4146 13.6406 14.3846 14.0769C15.3769 14.5971 15.8058 14.5441 16.2308 15.2308C16.7778 15.7074 17.3846 17.3077 17.3846 17.3077C17.3846 17.3077 17.6152 15.7598 16.9231 14.3077C16.1468 12.9944 13.6923 11.7692 13.6923 11.7692ZM11.7378 10.1538L12.6496 9C12.6496 9 12.7852 9.65153 12.9999 9.9972C13.2146 10.3429 13.5614 10.6154 13.5614 10.6154C13.5614 10.6154 13.1055 11.3283 13.1055 12.0503C13.1055 12.7722 13.7312 15 13.7312 16.1741C13.7312 16.1741 13.7582 16.8763 13.6866 18.1338C13.6866 18.1338 13.6788 18.7708 12.9116 19.8347C12.9116 19.8347 11.924 21 9.91424 21C9.91424 21 8.38071 20.9138 7.60325 20.0769C7.60325 20.0769 7.13163 19.6741 7 18.7098C7 18.7098 7 18.2308 7.21364 17.5663C7.21364 17.5663 7.49718 16.8552 8.09067 15.9231C8.09067 15.9231 7.22346 17.5735 7.60325 18.541C7.98304 19.5086 9.21876 19.7263 9.91424 19.7263C10.6097 19.7263 11.5084 19.4808 12.1969 18.9676C12.9999 18.369 12.9116 17.0121 12.9116 16.1741C12.9116 15.3361 12.4969 13.2412 12.4969 13.2412C12.4114 12.6704 12.0241 11.4085 11.9658 11.0769C11.9075 10.7453 11.7378 10.1538 11.7378 10.1538Z" fill="white"/> <path d="M9.54846 27.8V23.8H10.3942V25.4504H12.111V23.8H12.9547V27.8H12.111V26.1476H10.3942V27.8H9.54846ZM14.3414 27.8H13.4352L14.816 23.8H15.9059L17.2848 27.8H16.3785L15.3766 24.7141H15.3453L14.3414 27.8ZM14.2848 26.2277H16.4254V26.8879H14.2848V26.2277ZM17.7662 27.8V23.8H18.6119V27.1027H20.3268V27.8H17.7662ZM21.6783 27.8H20.7721L22.153 23.8H23.2428L24.6217 27.8H23.7155L22.7135 24.7141H22.6823L21.6783 27.8ZM21.6217 26.2277H23.7623V26.8879H21.6217V26.2277ZM25.1031 27.8V23.8H25.9489V27.1027H27.6637V27.8H25.1031Z" fill="white"/> </svg> ',

			'yeast-free'        => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>yeast-free</defs> <rect width="38" height="38" rx="19" fill="#DA225C"/> <path d="M27.7893 24.1053C27.8743 23.3559 27.6626 22.6182 27.193 22.0278L25.0148 19.2918C24.7745 18.9902 24.4844 18.7514 24.1658 18.5767C24.3275 18.3656 24.4629 18.1293 24.5625 17.8684L25.81 14.6015C26.3656 13.1467 25.6336 11.5111 24.1793 10.9562C22.7247 10.4008 21.0892 11.1324 20.5342 12.5868L19.2867 15.8544C19.1849 16.1209 19.1268 16.3934 19.1079 16.6643L17.4597 15.5806C16.2301 14.7724 14.5969 15.0531 13.699 16.1841C13.3604 15.8775 12.9495 15.6565 12.4938 15.5418C10.984 15.1627 9.44677 16.0828 9.06714 17.5928L8.21515 20.9847C8.03146 21.7162 8.14365 22.4755 8.53099 23.1227C8.65293 23.3257 8.79702 23.5101 8.96013 23.6732C9.31682 24.0299 9.76434 24.2847 10.2662 24.4108C10.9979 24.5946 11.7574 24.4827 12.4045 24.0952C13.0518 23.7079 13.5093 23.0916 13.6932 22.36L14.2334 20.2082C14.2746 20.2387 14.314 20.2706 14.3572 20.299L17.2796 22.2205C18.2097 22.8319 19.3697 22.818 20.2688 22.2872C20.3574 22.4689 20.4651 22.6444 20.5961 22.8091L22.7743 25.5451C22.8421 25.6301 22.9132 25.7098 22.988 25.7846C23.9891 26.7857 25.6087 26.8977 26.742 25.9958C27.332 25.5261 27.7041 24.8548 27.7893 24.1053ZM21.956 13.1299C22.2119 12.4594 22.9659 12.1222 23.6362 12.378C23.8124 12.4454 23.9656 12.5468 24.0911 12.6723C24.4433 13.0244 24.577 13.5642 24.3883 14.0584L23.1408 17.3254C22.8845 17.9962 22.1307 18.3333 21.4604 18.0774C20.7891 17.821 20.453 17.0678 20.7085 16.3972L21.956 13.1299ZM11.6228 22.7893C11.3245 22.9678 10.9743 23.0195 10.6372 22.9345C10.3002 22.8504 10.0161 22.6392 9.83757 22.3407C9.65906 22.0424 9.60721 21.6923 9.69222 21.3553L10.544 17.9635C10.7191 17.2671 11.4278 16.8428 12.1237 17.0177C12.355 17.0755 12.5617 17.1932 12.726 17.3576C12.8013 17.4329 12.8676 17.5177 12.9237 17.6115C13.0109 17.7572 13.0668 17.9156 13.0925 18.0785C13.0945 18.1202 13.0969 18.1618 13.1006 18.2033C13.1083 18.3342 13.102 18.4662 13.0692 18.5971L12.2177 21.9892C12.1321 22.3265 11.9212 22.6106 11.6228 22.7893ZM18.1157 20.9488L15.1934 19.0273C14.8742 18.8166 14.682 18.4883 14.6275 18.1378C14.6244 18.0724 14.6162 18.0077 14.6085 17.9428C14.6076 17.6958 14.6756 17.4458 14.8211 17.2245C15.2154 16.625 16.0241 16.458 16.6238 16.8521L19.546 18.7737C19.8373 18.9645 20.0353 19.257 20.1059 19.5979C20.1761 19.9384 20.1098 20.2857 19.9187 20.5764C19.5243 21.176 18.7154 21.3429 18.1157 20.9488ZM25.794 24.8049C25.2327 25.2521 24.4126 25.1601 23.9653 24.5971L21.7869 21.8609C21.3398 21.2995 21.4332 20.4789 21.9946 20.0319C22.5169 19.6158 23.2636 19.6676 23.725 20.129C23.7594 20.1634 23.7926 20.2006 23.8235 20.2398L26.0019 22.976C26.2183 23.248 26.3162 23.5882 26.2769 23.9335C26.2376 24.2789 26.0663 24.5885 25.794 24.8049Z" fill="white"/> <path d="M32.7071 5.70709L26.7071 11.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M11.7071 26.7071L5.70709 32.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> </svg> ',

			'high-iron'         => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>high-iron</defs> <rect x="37.5" y="37.5" width="37" height="37" rx="18.5" transform="rotate(-180 37.5 37.5)" fill="white"/> <rect x="37.5" y="37.5" width="37" height="37" rx="18.5" transform="rotate(-180 37.5 37.5)" stroke="#FF7B5F"/> <mask id="mask0_215_3938" style="mask-type:alpha" maskUnits="userSpaceOnUse" x="0" y="0" width="38" height="38"> <rect x="38" y="38" width="38" height="38" rx="19" transform="rotate(-180 38 38)" fill="#2DB68D"/> </mask> <g mask="url(#mask0_215_3938)"> <rect x="39" y="19" width="39" height="19" transform="rotate(-180 39 19)" fill="#FF7B5F"/> </g> <g clip-path="url(#clip0_215_3938)"> <path d="M21.8125 19H16.1875V11.5H11.65L19 4.15002L26.35 11.5H21.8125V19Z" fill="white"/> </g> <path d="M12.6421 25V21.3636H13.1908V22.9439H15.0037V21.3636H15.5541V25H15.0037V23.4144H13.1908V25H12.6421ZM16.9017 21.3636V25H16.3531V21.3636H16.9017ZM20.1617 22.5124C20.1274 22.4047 20.0812 22.3082 20.0232 22.223C19.9664 22.1366 19.8983 22.0632 19.819 22.0028C19.7397 21.9413 19.6491 21.8945 19.5473 21.8626C19.4467 21.8306 19.336 21.8146 19.2153 21.8146C19.0105 21.8146 18.8259 21.8673 18.6613 21.9727C18.4968 22.078 18.3666 22.2325 18.2707 22.4361C18.176 22.6385 18.1287 22.8865 18.1287 23.18C18.1287 23.4748 18.1766 23.724 18.2725 23.9276C18.3684 24.1312 18.4997 24.2856 18.6667 24.391C18.8336 24.4963 19.0235 24.549 19.2366 24.549C19.4343 24.549 19.6065 24.5088 19.7533 24.4283C19.9013 24.3478 20.0155 24.2341 20.096 24.0874C20.1777 23.9394 20.2185 23.7654 20.2185 23.5653L20.3605 23.592H19.3201V23.1392H20.7494V23.5529C20.7494 23.8583 20.6843 24.1235 20.5541 24.3484C20.4251 24.5721 20.2463 24.7449 20.0179 24.8668C19.7906 24.9888 19.5302 25.0497 19.2366 25.0497C18.9075 25.0497 18.6187 24.974 18.3701 24.8224C18.1227 24.6709 17.9298 24.4561 17.7913 24.1779C17.6528 23.8986 17.5836 23.5671 17.5836 23.1836C17.5836 22.8936 17.6238 22.6332 17.7043 22.4023C17.7848 22.1715 17.8978 21.9756 18.0434 21.8146C18.1902 21.6525 18.3624 21.5288 18.5601 21.4435C18.759 21.3571 18.9762 21.3139 19.2118 21.3139C19.4082 21.3139 19.5911 21.3429 19.7604 21.4009C19.9309 21.4589 20.0824 21.5412 20.2149 21.6477C20.3487 21.7543 20.4594 21.8809 20.547 22.0277C20.6346 22.1733 20.6938 22.3349 20.7245 22.5124H20.1617ZM21.4459 25V21.3636H21.9945V22.9439H23.8074V21.3636H24.3578V25H23.8074V23.4144H21.9945V25H21.4459Z" fill="#FF7B5F"/> <path d="M12.4395 27.6364V32H11.5169V27.6364H12.4395ZM13.1986 32V27.6364H14.9202C15.2497 27.6364 15.531 27.6953 15.7639 27.8132C15.9983 27.9297 16.1765 28.0952 16.2987 28.3097C16.4223 28.5227 16.4841 28.7734 16.4841 29.0618C16.4841 29.3516 16.4216 29.6009 16.2966 29.8097C16.1716 30.017 15.9905 30.1761 15.7533 30.2869C15.5175 30.3977 15.2319 30.4531 14.8967 30.4531H13.744V29.7116H14.7476C14.9237 29.7116 15.07 29.6875 15.1865 29.6392C15.303 29.5909 15.3896 29.5185 15.4464 29.4219C15.5047 29.3253 15.5338 29.2053 15.5338 29.0618C15.5338 28.9169 15.5047 28.7947 15.4464 28.6953C15.3896 28.5959 15.3023 28.5206 15.1844 28.4695C15.0679 28.4169 14.9209 28.3906 14.7433 28.3906H14.1212V32H13.1986ZM15.5551 30.0142L16.6396 32H15.6212L14.5601 30.0142H15.5551ZM21.146 29.8182C21.146 30.294 21.0558 30.6989 20.8754 31.0327C20.6964 31.3665 20.4521 31.6214 20.1425 31.7976C19.8342 31.9723 19.4876 32.0597 19.1027 32.0597C18.7149 32.0597 18.3669 31.9716 18.0587 31.7955C17.7504 31.6193 17.5068 31.3643 17.3278 31.0305C17.1489 30.6967 17.0594 30.2926 17.0594 29.8182C17.0594 29.3423 17.1489 28.9375 17.3278 28.6037C17.5068 28.2699 17.7504 28.0156 18.0587 27.8409C18.3669 27.6648 18.7149 27.5767 19.1027 27.5767C19.4876 27.5767 19.8342 27.6648 20.1425 27.8409C20.4521 28.0156 20.6964 28.2699 20.8754 28.6037C21.0558 28.9375 21.146 29.3423 21.146 29.8182ZM20.2106 29.8182C20.2106 29.5099 20.1645 29.25 20.0721 29.0384C19.9812 28.8267 19.8527 28.6662 19.6865 28.5568C19.5203 28.4474 19.3257 28.3928 19.1027 28.3928C18.8797 28.3928 18.6851 28.4474 18.5189 28.5568C18.3527 28.6662 18.2234 28.8267 18.1311 29.0384C18.0402 29.25 17.9947 29.5099 17.9947 29.8182C17.9947 30.1264 18.0402 30.3864 18.1311 30.598C18.2234 30.8097 18.3527 30.9702 18.5189 31.0795C18.6851 31.1889 18.8797 31.2436 19.1027 31.2436C19.3257 31.2436 19.5203 31.1889 19.6865 31.0795C19.8527 30.9702 19.9812 30.8097 20.0721 30.598C20.1645 30.3864 20.2106 30.1264 20.2106 29.8182ZM25.4793 27.6364V32H24.6824L22.784 29.2536H22.752V32H21.8294V27.6364H22.6391L24.5226 30.3807H24.561V27.6364H25.4793Z" fill="black"/> <defs> <clipPath id="clip0_215_3938"> <rect width="15" height="15" fill="white" transform="translate(11.5 4)"/> </clipPath> </defs> </svg> ',

			'plant-based'       => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>plant-based</defs> <rect x="-6.10352e-05" width="38" height="38" rx="19" fill="#00C68A"/> <path d="M27.5564 15.5725C27.87 15.4157 28.2097 15.2066 28.5234 14.9191C27.5825 13.4817 26.328 12.2794 24.8382 11.4169C24.8382 11.4692 24.8382 11.5215 24.8382 11.5476C24.8382 11.9396 24.786 12.3317 24.7076 12.6976C25.8575 13.4555 26.8245 14.4225 27.5564 15.5725Z" fill="white"/> <path d="M28.8109 24.0144C28.7064 24.0928 28.628 24.1713 28.5234 24.2235C27.2166 27.778 23.8974 30.3654 19.977 30.6268C19.8986 24.119 20.1861 20.7475 17.5725 18.3691C14.8544 15.9124 11.9795 18.1078 9.28749 15.3374C9.05227 15.1021 8.86932 14.8669 8.68637 14.6317C8.60796 14.5272 8.50342 14.4749 8.39888 14.4749C8.21593 14.4749 8.03298 14.6056 8.03298 14.8408C7.77162 19.5452 9.10454 22.1588 10.4897 23.3088C11.9533 24.5372 14.384 25.0076 16.0567 25.1906C16.1351 23.9099 15.9521 22.6031 15.3771 21.7668C14.6192 20.7213 13.1556 20.0679 12.084 19.702C12.0579 19.702 12.0579 19.6759 12.0318 19.6759C11.9272 19.5975 11.9533 19.4668 12.0318 19.4145L12.0579 19.3884C12.084 19.3884 12.1102 19.3623 12.1363 19.3623H12.1624C13.3647 19.4668 15.0896 19.7804 16.1089 20.7736C18.2259 22.7861 18.6702 29.2416 18.7486 30.6268C14.8283 30.3654 11.5352 27.8041 10.2284 24.2497C10.1238 24.1713 10.0193 24.0928 9.94089 24.0144C9.52272 23.6747 9.02614 23.152 8.58183 22.3679C9.26136 27.6996 13.8351 31.8029 19.3498 31.8029C24.8644 31.8029 29.4382 27.6996 30.1961 22.3679C29.7257 23.152 29.2291 23.6747 28.8109 24.0144Z" fill="white"/> <path d="M13.9658 12.6976C13.8873 12.3317 13.8351 11.9396 13.8351 11.5476C13.8351 11.4953 13.8351 11.4431 13.8351 11.4169C12.3453 12.2794 11.0908 13.4817 10.1499 14.9191C10.4636 15.2066 10.7772 15.4157 11.117 15.5725C11.8749 14.4225 12.8419 13.4555 13.9658 12.6976Z" fill="white"/> <path d="M19.3498 16.1476C21.8849 16.1476 23.9235 14.0828 23.9235 11.5738C23.9235 9.03862 21.8588 7.00003 19.3498 7.00003C16.8146 7.00003 14.776 9.06476 14.776 11.5738C14.776 14.0828 16.8146 16.1476 19.3498 16.1476ZM17.1021 11.0249C17.3634 10.7897 17.7555 10.7897 17.9907 11.0511L18.827 11.9658L20.6827 9.97951C20.9179 9.71815 21.3099 9.71815 21.5713 9.95337C21.8327 10.1886 21.8327 10.5806 21.5974 10.842L19.2975 13.3249C19.1668 13.4556 19.01 13.534 18.827 13.534C18.6441 13.534 18.4873 13.4556 18.3566 13.3249L17.0498 11.9136C16.8407 11.6522 16.8669 11.2602 17.1021 11.0249Z" fill="white"/> <path d="M19.8725 20.0156C20.3952 21.1656 20.6304 22.4985 20.735 24.0667C21.0747 22.8122 21.5975 21.5576 22.4599 20.7474C23.5054 19.7804 25.2042 19.4406 26.4326 19.3361C26.6417 19.31 26.6939 19.5975 26.511 19.6759C25.4133 20.0418 23.9497 20.7213 23.2179 21.7406C22.6168 22.5508 22.4338 23.8837 22.5383 25.1644C24.211 25.0076 26.7201 24.5371 28.2621 23.2565C29.6473 22.1065 30.9802 19.4929 30.7188 14.7885C30.6927 14.4487 30.2745 14.2919 30.0655 14.5794C29.9086 14.8146 29.6996 15.0498 29.4643 15.2851C26.7723 18.0554 23.8974 15.86 21.1793 18.3168C20.5782 18.8657 20.1861 19.4145 19.8725 20.0156Z" fill="white"/> </svg> ',

			'high-calcium'      => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>high-calcium</defs> <rect x="-9.15527e-05" width="38" height="38" rx="19" fill="#F59C7B"/> <path d="M19.2953 14.6029L18.7478 15.8833L19.4225 16.1712L19.9696 14.8904C20.3401 14.0231 20.5323 13.0901 20.5346 12.147V10.9333H24.9346V12.147C24.9368 13.0904 25.129 14.0236 25.4996 14.8911L26.629 17.5311C26.8365 18.019 26.9818 18.5311 27.0613 19.0554L26.7118 19.1606C25.3567 19.545 23.9164 19.5051 22.5846 19.0462C22.1754 18.904 21.7544 18.7982 21.3266 18.7298L21.21 19.4536C21.5952 19.5153 21.9742 19.6107 22.3426 19.7388C23.1409 20.0154 23.9798 20.156 24.8246 20.155C25.5314 20.1553 26.2348 20.0574 26.9146 19.8642L27.1221 19.8019C27.1221 19.8631 27.1313 19.9255 27.1313 19.9852V28.1777C27.131 28.4658 27.0166 28.7422 26.8132 28.9462C26.6097 29.1503 26.3338 29.2655 26.0456 29.2667H20.5346V30H26.0456C26.5288 29.9995 26.992 29.8074 27.3336 29.4657C27.6753 29.1241 27.8674 28.6608 27.8679 28.1777V19.9863C27.8679 19.7531 27.8562 19.5152 27.8313 19.2772C27.7577 18.5761 27.5791 17.8901 27.3014 17.2422L26.1721 14.6022C25.8411 13.8259 25.6696 12.9909 25.6679 12.147V10.9333H26.0346C26.1318 10.9333 26.2251 10.8947 26.2939 10.8259C26.3626 10.7572 26.4013 10.6639 26.4013 10.5667V9.46667C26.4013 9.07768 26.2467 8.70463 25.9717 8.42958C25.6966 8.15452 25.3236 8 24.9346 8H20.5346C20.1456 8 19.7726 8.15452 19.4975 8.42958C19.2225 8.70463 19.0679 9.07768 19.0679 9.46667V10.5667C19.0679 10.6639 19.1066 10.7572 19.1753 10.8259C19.2441 10.8947 19.3374 10.9333 19.4346 10.9333H19.8013V12.147C19.7992 12.9913 19.6271 13.8266 19.2953 14.6029ZM19.8013 9.46667C19.8013 9.27217 19.8785 9.08565 20.0161 8.94812C20.1536 8.81059 20.3401 8.73333 20.5346 8.73333H24.9346C25.1291 8.73333 25.3156 8.81059 25.4531 8.94812C25.5907 9.08565 25.6679 9.27217 25.6679 9.46667V10.2H19.8013V9.46667Z" fill="white"/> <path d="M22.0013 12.1455V12.0333H21.2679V12.1455C21.2679 12.3443 21.2604 12.542 21.2452 12.7388L21.9767 12.7945C21.9932 12.5804 22.0013 12.3633 22.0013 12.1455Z" fill="white"/> <path d="M20.6446 15.1786L20.326 15.9222L20.9995 16.2111L21.3185 15.4672C21.5804 14.8559 21.7693 14.2159 21.8814 13.5605L21.1587 13.4373C21.0562 14.0359 20.8837 14.6203 20.6446 15.1786Z" fill="white"/> <path d="M11.0977 16.9188C11.063 16.9562 11.0365 17.0004 11.0199 17.0487C11.0033 17.0969 10.997 17.1481 11.0013 17.1989L11.3914 21.6191L12.1013 29.6656C12.1094 29.7571 12.1515 29.8423 12.2194 29.9042C12.2873 29.9662 12.376 30.0003 12.4679 30H19.0679C19.1598 30.0003 19.2485 29.9662 19.3164 29.9042C19.3843 29.8423 19.4265 29.7571 19.4346 29.6656L20.1617 21.4277L20.5346 17.1989C20.5391 17.1481 20.5329 17.097 20.5165 17.0487C20.5001 17.0004 20.4738 16.9561 20.4393 16.9185C20.4048 16.881 20.3629 16.8511 20.3161 16.8306C20.2694 16.8102 20.2189 16.7998 20.1679 16.8H11.3679C11.3171 16.8 11.2669 16.8105 11.2203 16.831C11.1738 16.8514 11.1321 16.8813 11.0977 16.9188ZM12.8038 29.2667L12.1493 21.8505L12.3715 21.7819C13.3648 21.4501 14.4464 21.5024 15.4031 21.9286C16.5688 22.4495 17.8916 22.4956 19.0907 22.0569H19.0958L19.3781 21.9491L18.7321 29.2667H12.8038ZM19.7675 17.5333L19.4496 21.1347L18.8373 21.3683C17.8173 21.7407 16.6923 21.7013 15.7008 21.2583C14.5798 20.7585 13.3124 20.6964 12.1478 21.0841L12.0833 21.1039L11.7683 17.5333H19.7675Z" fill="white"/> <path d="M14.0027 23.9418L13.272 24.0045L13.6609 28.5346L14.3915 28.4718L14.0027 23.9418Z" fill="white"/> <path d="M13.8719 22.4212L13.1413 22.4843L13.2109 23.2912L13.9415 23.2282L13.8719 22.4212Z" fill="white"/> </svg> ',

			'no-added-colors'   => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>no-added-colors</defs> <rect x="-9.15527e-05" width="38" height="38" rx="19" fill="#DA225C"/> <path d="M21.2488 21.5443C21.2925 21.549 21.3361 21.5538 21.3797 21.5586C21.6198 21.5847 21.8693 21.5236 22.0871 21.3486C22.3291 21.1542 22.4473 20.88 22.4346 20.5916C22.4103 19.3966 21.9885 18.2024 21.2302 17.2586L18.4062 13.7933L18.8176 13.4627C19.35 13.035 19.4333 12.2713 19.0055 11.7389C18.5778 11.2065 17.814 11.1232 17.2816 11.551L16.7469 12.0005L14.1586 8.80376C13.5753 8.07776 12.5061 7.96121 11.7801 8.5445C11.0541 9.12778 10.9375 10.197 11.5208 10.923L14.0849 14.1392L13.5283 14.5864C12.9959 15.0142 12.9126 15.7779 13.3404 16.3103C13.7681 16.8427 14.5319 16.9259 15.0643 16.4982L15.4757 16.1677L18.2754 19.6525C19.0142 20.5721 20.0677 21.2389 21.2488 21.5443ZM17.656 14.396L20.4558 17.8808C21.078 18.6552 21.4363 19.6216 21.4868 20.5766C20.5191 20.3386 19.6525 19.7805 19.0304 19.0061L16.2282 15.5431L17.656 14.396Z" fill="white"/> <path d="M22.3876 22.9319C22.2559 22.7124 22.0364 22.6027 21.7949 22.6027C21.5534 22.6027 21.312 22.7344 21.2022 22.9319C20.7413 23.7002 19.2267 26.2904 19.2267 27.4318C19.2267 28.8586 20.3681 30 21.7949 30C23.2217 30 24.3631 28.8586 24.3631 27.4318C24.3412 26.2904 22.8485 23.7002 22.3876 22.9319ZM21.4437 26.3123C21.4217 26.3562 20.8071 27.3659 21.2681 28.0464C21.3339 28.1342 21.3559 28.244 21.3339 28.3537C21.312 28.4635 21.2461 28.5513 21.1583 28.6171C21.0925 28.661 21.0047 28.683 20.9388 28.683C20.8071 28.683 20.6754 28.6171 20.5876 28.5074C19.8194 27.3879 20.7193 25.9392 20.7632 25.8733C20.8291 25.7855 20.9169 25.7196 21.0266 25.6977C21.1364 25.6757 21.2461 25.6977 21.3339 25.7416C21.5095 25.8733 21.5754 26.1367 21.4437 26.3123Z" fill="white"/> <path d="M32.707 5.70709L26.707 11.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M11.707 26.7071L5.707 32.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> </svg> ',

			'high-omega-3'      => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>high-omega-3</defs> <rect x="-0.00012207" width="38" height="38" rx="19" fill="#8EBB5C"/> <path d="M14.6018 26.459C14.5886 26.459 14.5753 26.4585 14.5618 26.4575C12.9361 26.3283 10.9984 25.9599 10.9984 24.9375C10.9984 24.3044 11.7051 23.8656 13.1585 23.5967C13.4294 23.5477 13.6906 23.7259 13.7411 23.9974C13.7913 24.269 13.6119 24.5298 13.3405 24.58C12.6916 24.7001 12.3192 24.8393 12.1284 24.9389C12.4539 25.1075 13.2485 25.3504 14.6407 25.4608C14.9159 25.4828 15.1214 25.7238 15.0994 25.9987C15.0786 26.2605 14.8596 26.459 14.6017 26.459L14.6018 26.459Z" fill="white"/> <path d="M18.7559 24.9992H18.7384H18.7307C15.7102 24.9847 13.2568 22.4146 13.2568 19.2576C13.2568 16.903 16.0609 13.5906 17.7975 11.7653C17.8165 11.7331 17.8398 11.7026 17.8665 11.6743C18.099 11.4319 18.421 11.3036 18.7469 11.3019C19.0796 11.2926 19.3953 11.4319 19.6273 11.6743C19.6541 11.7026 19.6773 11.7331 19.6963 11.7651C21.4328 13.5903 24.2371 16.9026 24.2371 19.2576C24.2371 22.4146 21.7839 24.9848 18.7634 24.9992H18.7558L18.7559 24.9992ZM18.7471 12.3021C18.7126 12.3036 18.6554 12.3149 18.6064 12.3664C17.586 13.4317 14.2569 17.0771 14.2569 19.2578C14.2567 21.8694 16.2701 23.9945 18.7472 23.9995C21.2241 23.9945 23.2377 21.8696 23.2377 19.2578C23.2377 17.0767 19.9085 13.4314 18.8882 12.3662C18.839 12.3149 18.7818 12.3037 18.7473 12.3022L18.7471 12.3021Z" fill="white"/> <path d="M19.0957 22.9995C18.8198 22.9995 18.5958 22.7755 18.5958 22.4996C18.5958 22.2237 18.8198 21.9997 19.0957 21.9997C19.5051 21.9997 19.8973 21.8482 20.2 21.573C20.404 21.3876 20.7202 21.4023 20.9061 21.607C21.0915 21.8113 21.0766 22.1274 20.8721 22.3131C20.3853 22.7556 19.7541 22.9995 19.0957 22.9995Z" fill="white"/> <path d="M19.0534 20.873H17.8603C17.6369 20.873 17.4409 20.7247 17.3797 20.51C17.3184 20.2955 17.4072 20.066 17.5969 19.9483C18.0991 19.6366 18.3039 19.3049 18.3039 18.8027C18.3039 18.1799 17.7972 17.6732 17.1743 17.6732C16.5515 17.6732 16.0448 18.1799 16.0448 18.8027C16.0448 19.3659 16.4104 19.7636 16.7234 19.9328C16.9248 20.0418 17.0263 20.2733 16.9703 20.4952C16.9143 20.7172 16.7146 20.8727 16.4856 20.8727H15.2948C15.0189 20.8727 14.7949 20.6487 14.7949 20.3728C14.7949 20.0969 15.0189 19.8729 15.2948 19.8729H15.3167C15.1521 19.5702 15.0448 19.2114 15.0448 18.8031C15.0448 17.629 16.0001 16.6736 17.1743 16.6736C18.3484 16.6736 19.3037 17.6289 19.3037 18.8031C19.3037 19.202 19.2188 19.5577 19.0465 19.8734H19.0537C19.3297 19.8734 19.5537 20.0974 19.5537 20.3733C19.5537 20.6493 19.3295 20.873 19.0535 20.873H19.0534Z" fill="white"/> <path d="M21.0545 20.873C20.2209 20.873 19.5425 20.2898 19.5425 19.5732C19.5425 19.2973 19.7665 19.0733 20.0424 19.0733C20.3183 19.0733 20.5423 19.2973 20.5423 19.5732C20.5423 19.712 20.766 19.8732 21.0545 19.8732C21.343 19.8732 21.5667 19.7116 21.5667 19.5732C21.5667 19.4348 21.343 19.2733 21.0545 19.2733C20.7786 19.2733 20.5546 19.0493 20.5546 18.7734C20.5546 18.4975 20.7786 18.2735 21.0545 18.2735C21.343 18.2735 21.5667 18.112 21.5667 17.9733C21.5667 17.8346 21.343 17.6734 21.0545 17.6734C20.7661 17.6734 20.5423 17.8346 20.5423 17.9733C20.5423 18.2492 20.3183 18.4733 20.0424 18.4733C19.7665 18.4733 19.5425 18.2492 19.5425 17.9733C19.5425 17.2567 20.2208 16.6736 21.0545 16.6736C21.8881 16.6736 22.5666 17.2567 22.5666 17.9733C22.5666 18.2748 22.4465 18.5527 22.2457 18.7734C22.4466 18.9942 22.5666 19.2719 22.5666 19.5732C22.5665 20.2899 21.8881 20.873 21.0545 20.873Z" fill="white"/> <path d="M30.3391 25.134C30.3841 24.9975 30.42 24.8581 30.4451 24.715C30.4768 24.6495 30.496 24.577 30.496 24.4993C30.496 24.4436 30.4925 24.3888 30.4865 24.3346C30.4888 24.2883 30.496 24.2429 30.496 24.1959V14.8043C30.496 13.7095 29.8996 12.7019 28.9401 12.1747L20.1913 7.37196C19.2877 6.87601 18.2084 6.87601 17.3044 7.37196L8.55586 12.1749C7.59605 12.7019 6.99988 13.7094 6.99988 14.8043V24.1957C6.99988 25.2907 7.59605 26.2981 8.55586 26.8251L17.3044 31.6278C17.7563 31.876 18.2523 32 18.748 32C19.2437 32 19.7396 31.876 20.1915 31.6278L23.3385 29.9001C24.0784 31.1533 25.4392 31.9981 26.9967 31.9981C29.3396 31.9981 31.246 30.0917 31.246 27.7488C31.246 26.7624 30.9053 25.8561 30.3391 25.134L30.3391 25.134ZM29.3385 28.1062C29.2408 28.2039 29.1131 28.2527 28.9851 28.2527C28.8571 28.2527 28.7293 28.2039 28.6316 28.1062L27.4972 26.9719V29.7525C27.4972 30.0284 27.2732 30.2524 26.9973 30.2524C26.7214 30.2524 26.4974 30.0284 26.4974 29.7525V26.96L25.3511 28.1063C25.2534 28.204 25.1257 28.2527 24.9977 28.2527C24.8697 28.2527 24.7419 28.204 24.6442 28.1063C24.449 27.9111 24.449 27.5947 24.6442 27.3994L26.5786 25.4649C26.5946 25.4422 26.6116 25.4199 26.6321 25.3997C26.7313 25.3004 26.8615 25.2522 26.9915 25.2537C27.0263 25.2532 27.061 25.2562 27.0953 25.263C27.1103 25.266 27.124 25.2714 27.1385 25.2758C27.1553 25.2807 27.1723 25.2845 27.1885 25.2912C27.2057 25.2982 27.2212 25.308 27.2374 25.3169C27.2497 25.3237 27.2627 25.3292 27.2745 25.3372C27.3267 25.3722 27.3717 25.4167 27.4074 25.4682L29.3383 27.3991C29.5337 27.5946 29.5337 27.9108 29.3385 28.1063L29.3385 28.1062ZM29.4962 22.8883C28.5633 22.2132 27.126 21.7579 25.8378 21.4636C25.5683 21.4008 25.3007 21.5703 25.2389 21.8392C25.1771 22.1085 25.3456 22.3764 25.6146 22.4382C27.9788 22.9791 29.1902 23.7145 29.4439 24.2814C28.7513 23.7906 27.9081 23.4994 26.9966 23.4994C26.6804 23.4994 26.373 23.537 26.0763 23.6027C25.817 23.4654 25.5051 23.3383 25.1362 23.2208C24.8745 23.1368 24.592 23.2823 24.508 23.5455C24.4325 23.7832 24.5448 24.0337 24.7607 24.1422C24.1158 24.5436 23.5851 25.1103 23.2315 25.7856C22.0965 25.9848 20.608 26.1293 18.7479 26.1293C18.472 26.1293 18.248 26.3533 18.248 26.6292C18.248 26.9051 18.472 27.1291 18.7479 27.1291C19.9362 27.1291 21.4424 27.0549 22.8431 26.8575C22.7813 27.1451 22.7473 27.4431 22.7473 27.7488C22.7473 28.178 22.8125 28.5921 22.9315 28.9831L19.7104 30.7515C19.1079 31.0819 18.3883 31.0819 17.7859 30.7515L9.03735 25.9488C8.41522 25.6073 8.02522 24.9622 8.00399 24.2558C8.05273 24.223 8.09797 24.184 8.13394 24.1335C8.56112 23.5304 9.83093 22.935 11.6173 22.5C11.8855 22.4345 12.05 22.164 11.9848 21.8959C11.9193 21.6279 11.6486 21.4634 11.3806 21.5284C10.2699 21.7989 8.90687 22.2379 7.99973 22.8952V14.8042C7.99973 14.0744 8.39712 13.4027 9.03708 13.0512L17.7857 8.24848C18.0869 8.08323 18.4176 8.00057 18.748 8.00057C19.0784 8.00057 19.4091 8.08332 19.7103 8.24873L28.4589 13.0515C29.0989 13.4028 29.4963 14.0744 29.4963 14.8042L29.4962 22.8883Z" fill="white"/> </svg> ',

			'low-cholestrol'    => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>low-cholestrol</defs> <rect x="0.503174" y="0.5" width="37" height="37" rx="18.5" fill="white"/> <rect x="0.503174" y="0.5" width="37" height="37" rx="18.5" stroke="#58C3FF"/> <mask id="mask0_217_4105" style="mask-type:alpha" maskUnits="userSpaceOnUse" x="0" y="0" width="39" height="38"> <rect x="0.00317383" width="38" height="38" rx="19" fill="#2DB68D"/> </mask> <g mask="url(#mask0_217_4105)"> <rect x="0.00317383" y="19" width="39" height="19" fill="#58C3FF"/> </g> <g clip-path="url(#clip0_217_4105)"> <path d="M16.1907 19L21.8157 19L21.8157 26.5L26.3532 26.5L19.0032 33.85L11.6532 26.5L16.1907 26.5L16.1907 19Z" fill="white"/> </g> <path d="M14.6885 10V7.09091H15.2155V9.55824H16.4968V10H14.6885ZM19.4403 8.54545C19.4403 8.8589 19.3816 9.12737 19.2642 9.35085C19.1477 9.57339 18.9886 9.74384 18.7869 9.86222C18.5862 9.98059 18.3584 10.0398 18.1037 10.0398C17.8489 10.0398 17.6207 9.98059 17.419 9.86222C17.2183 9.7429 17.0592 9.57197 16.9417 9.34943C16.8253 9.12595 16.767 8.85795 16.767 8.54545C16.767 8.23201 16.8253 7.96402 16.9417 7.74148C17.0592 7.51799 17.2183 7.34706 17.419 7.22869C17.6207 7.11032 17.8489 7.05114 18.1037 7.05114C18.3584 7.05114 18.5862 7.11032 18.7869 7.22869C18.9886 7.34706 19.1477 7.51799 19.2642 7.74148C19.3816 7.96402 19.4403 8.23201 19.4403 8.54545ZM18.9105 8.54545C18.9105 8.32481 18.8759 8.13873 18.8068 7.98722C18.7386 7.83475 18.6439 7.7197 18.5227 7.64205C18.4015 7.56345 18.2618 7.52415 18.1037 7.52415C17.9455 7.52415 17.8058 7.56345 17.6846 7.64205C17.5634 7.7197 17.4683 7.83475 17.3991 7.98722C17.3309 8.13873 17.2969 8.32481 17.2969 8.54545C17.2969 8.7661 17.3309 8.95265 17.3991 9.10511C17.4683 9.25663 17.5634 9.37169 17.6846 9.45028C17.8058 9.52794 17.9455 9.56676 18.1037 9.56676C18.2618 9.56676 18.4015 9.52794 18.5227 9.45028C18.6439 9.37169 18.7386 9.25663 18.8068 9.10511C18.8759 8.95265 18.9105 8.7661 18.9105 8.54545ZM20.4765 10L19.6555 7.09091H20.2223L20.7464 9.22869H20.7734L21.3331 7.09091H21.8487L22.4098 9.23011H22.4353L22.9595 7.09091H23.5263L22.7052 10H22.1853L21.603 7.95881H21.5802L20.9964 10H20.4765Z" fill="#58C3FF"/> <path d="M5.52215 13.873H4.82222C4.80944 13.7825 4.78334 13.7021 4.74392 13.6317C4.7045 13.5604 4.6539 13.4996 4.59211 13.4496C4.53032 13.3995 4.45894 13.3612 4.37797 13.3345C4.29807 13.3079 4.21125 13.2946 4.1175 13.2946C3.94811 13.2946 3.80056 13.3366 3.67485 13.4208C3.54914 13.5039 3.45166 13.6254 3.38241 13.7852C3.31317 13.9439 3.27854 14.1367 3.27854 14.3636C3.27854 14.5969 3.31317 14.793 3.38241 14.9517C3.45273 15.1104 3.55074 15.2303 3.67645 15.3113C3.80216 15.3922 3.94758 15.4327 4.1127 15.4327C4.20539 15.4327 4.29115 15.4205 4.36998 15.396C4.44988 15.3714 4.52073 15.3358 4.58252 15.2889C4.64431 15.2409 4.69545 15.1829 4.73593 15.1147C4.77748 15.0465 4.80624 14.9687 4.82222 14.8814L5.52215 14.8846C5.50404 15.0348 5.45876 15.1797 5.38632 15.3192C5.31494 15.4577 5.21853 15.5819 5.09708 15.6916C4.9767 15.8002 4.83287 15.8865 4.66562 15.9505C4.49942 16.0133 4.31139 16.0447 4.10152 16.0447C3.80961 16.0447 3.54861 15.9787 3.31849 15.8466C3.08944 15.7145 2.90834 15.5233 2.77517 15.2729C2.64307 15.0225 2.57702 14.7195 2.57702 14.3636C2.57702 14.0067 2.64413 13.7031 2.77836 13.4528C2.9126 13.2024 3.09477 13.0117 3.32488 12.8807C3.555 12.7486 3.81388 12.6825 4.10152 12.6825C4.29115 12.6825 4.46693 12.7092 4.62886 12.7624C4.79186 12.8157 4.93621 12.8935 5.06192 12.9957C5.18763 13.0969 5.28991 13.2211 5.36874 13.3681C5.44864 13.5151 5.49978 13.6834 5.52215 13.873ZM6.01833 16L6.01833 12.7273H6.71027V14.0776H8.11492V12.7273H8.80526V16H8.11492V14.6481H6.71027L6.71027 16H6.01833ZM12.3832 14.3636C12.3832 14.7205 12.3156 15.0241 12.1803 15.2745C12.046 15.5249 11.8628 15.7161 11.6305 15.8482C11.3994 15.9792 11.1394 16.0447 10.8507 16.0447C10.5599 16.0447 10.2989 15.9787 10.0677 15.8466C9.83651 15.7145 9.65381 15.5233 9.51958 15.2729C9.38534 15.0225 9.31823 14.7195 9.31823 14.3636C9.31823 14.0067 9.38534 13.7031 9.51958 13.4528C9.65381 13.2024 9.83651 13.0117 10.0677 12.8807C10.2989 12.7486 10.5599 12.6825 10.8507 12.6825C11.1394 12.6825 11.3994 12.7486 11.6305 12.8807C11.8628 13.0117 12.046 13.2024 12.1803 13.4528C12.3156 13.7031 12.3832 14.0067 12.3832 14.3636ZM11.6817 14.3636C11.6817 14.1325 11.6471 13.9375 11.5778 13.7788C11.5096 13.62 11.4132 13.4996 11.2886 13.4176C11.1639 13.3356 11.018 13.2946 10.8507 13.2946C10.6835 13.2946 10.5375 13.3356 10.4129 13.4176C10.2882 13.4996 10.1913 13.62 10.122 13.7788C10.0538 13.9375 10.0198 14.1325 10.0198 14.3636C10.0198 14.5948 10.0538 14.7898 10.122 14.9485C10.1913 15.1072 10.2882 15.2276 10.4129 15.3097C10.5375 15.3917 10.6835 15.4327 10.8507 15.4327C11.018 15.4327 11.1639 15.3917 11.2886 15.3097C11.4132 15.2276 11.5096 15.1072 11.5778 14.9485C11.6471 14.7898 11.6817 14.5948 11.6817 14.3636ZM12.8958 16V12.7273H13.5877V15.4295H14.9908V16H12.8958ZM15.449 16V12.7273H17.6543V13.2978H16.1409V14.0776H17.5408V14.6481H16.1409V15.4295H17.6606V16H15.449ZM19.9798 13.6685C19.967 13.5396 19.9121 13.4395 19.8152 13.3681C19.7182 13.2967 19.5866 13.261 19.4205 13.261C19.3075 13.261 19.2122 13.277 19.1344 13.3089C19.0566 13.3398 18.997 13.383 18.9554 13.4384C18.915 13.4938 18.8947 13.5566 18.8947 13.627C18.8926 13.6855 18.9048 13.7367 18.9315 13.7804C18.9592 13.824 18.997 13.8619 19.0449 13.8938C19.0929 13.9247 19.1483 13.9519 19.2111 13.9753C19.274 13.9977 19.3411 14.0169 19.4125 14.0328L19.7065 14.1032C19.8493 14.1351 19.9803 14.1777 20.0996 14.231C20.2189 14.2843 20.3223 14.3498 20.4096 14.4276C20.497 14.5053 20.5646 14.5969 20.6126 14.7024C20.6616 14.8079 20.6866 14.9288 20.6877 15.0652C20.6866 15.2654 20.6355 15.4391 20.5343 15.5861C20.4341 15.7321 20.2892 15.8455 20.0996 15.9265C19.911 16.0064 19.6836 16.0463 19.4173 16.0463C19.1531 16.0463 18.9229 16.0059 18.7269 15.9249C18.532 15.8439 18.3796 15.7241 18.2699 15.5653C18.1612 15.4055 18.1042 15.2079 18.0989 14.9725H18.7685C18.7759 15.0822 18.8074 15.1738 18.8627 15.2473C18.9192 15.3198 18.9943 15.3746 19.0881 15.4119C19.1829 15.4482 19.29 15.4663 19.4093 15.4663C19.5265 15.4663 19.6282 15.4492 19.7145 15.4151C19.8018 15.381 19.8695 15.3336 19.9174 15.2729C19.9654 15.2122 19.9893 15.1424 19.9893 15.0636C19.9893 14.9901 19.9675 14.9283 19.9238 14.8782C19.8812 14.8281 19.8184 14.7855 19.7353 14.7504C19.6532 14.7152 19.5526 14.6832 19.4332 14.6545L19.0769 14.565C18.801 14.4979 18.5831 14.3929 18.4233 14.2502C18.2635 14.1074 18.1841 13.9151 18.1852 13.6733C18.1841 13.4751 18.2369 13.302 18.3434 13.1539C18.451 13.0059 18.5985 12.8903 18.786 12.8072C18.9735 12.7241 19.1866 12.6825 19.4252 12.6825C19.6681 12.6825 19.8801 12.7241 20.0613 12.8072C20.2434 12.8903 20.3851 13.0059 20.4863 13.1539C20.5875 13.302 20.6397 13.4735 20.6429 13.6685H19.9798ZM21.0285 13.2978V12.7273L23.7163 12.7273V13.2978L22.7144 13.2978V16H22.0304V13.2978H21.0285ZM24.159 16V12.7273H26.3642V13.2978H24.8509V14.0776H26.2508V14.6481H24.8509V15.4295H26.3706V16H24.159ZM26.9143 16V12.7273H28.2055C28.4527 12.7273 28.6636 12.7715 28.8383 12.8599C29.0141 12.9473 29.1478 13.0714 29.2394 13.2322C29.3321 13.392 29.3785 13.5801 29.3785 13.7963C29.3785 14.0137 29.3316 14.2006 29.2378 14.3572C29.1441 14.5128 29.0083 14.6321 28.8303 14.7152C28.6535 14.7983 28.4394 14.8398 28.1879 14.8398H27.3234V14.2837H28.0761C28.2082 14.2837 28.3179 14.2656 28.4053 14.2294C28.4926 14.1932 28.5576 14.1388 28.6002 14.0664C28.6439 13.994 28.6657 13.9039 28.6657 13.7963C28.6657 13.6877 28.6439 13.5961 28.6002 13.5215C28.5576 13.4469 28.4921 13.3904 28.4037 13.3521C28.3163 13.3127 28.2061 13.293 28.0729 13.293H27.6063V16H26.9143ZM28.6817 14.5107L29.4951 16H28.7313L27.9355 14.5107H28.6817ZM32.8749 14.3636C32.8749 14.7205 32.8073 15.0241 32.672 15.2745C32.5377 15.5249 32.3545 15.7161 32.1222 15.8482C31.8911 15.9792 31.6311 16.0447 31.3424 16.0447C31.0516 16.0447 30.7906 15.9787 30.5594 15.8466C30.3282 15.7145 30.1455 15.5233 30.0113 15.2729C29.877 15.0225 29.8099 14.7195 29.8099 14.3636C29.8099 14.0067 29.877 13.7031 30.0113 13.4528C30.1455 13.2024 30.3282 13.0117 30.5594 12.8807C30.7906 12.7486 31.0516 12.6825 31.3424 12.6825C31.6311 12.6825 31.8911 12.7486 32.1222 12.8807C32.3545 13.0117 32.5377 13.2024 32.672 13.4528C32.8073 13.7031 32.8749 14.0067 32.8749 14.3636ZM32.1734 14.3636C32.1734 14.1325 32.1388 13.9375 32.0695 13.7788C32.0013 13.62 31.9049 13.4996 31.7803 13.4176C31.6556 13.3356 31.5097 13.2946 31.3424 13.2946C31.1752 13.2946 31.0292 13.3356 30.9046 13.4176C30.7799 13.4996 30.683 13.62 30.6137 13.7788C30.5455 13.9375 30.5115 14.1325 30.5115 14.3636C30.5115 14.5948 30.5455 14.7898 30.6137 14.9485C30.683 15.1072 30.7799 15.2276 30.9046 15.3097C31.0292 15.3917 31.1752 15.4327 31.3424 15.4327C31.5097 15.4327 31.6556 15.3917 31.7803 15.3097C31.9049 15.2276 32.0013 15.1072 32.0695 14.9485C32.1388 14.7898 32.1734 14.5948 32.1734 14.3636ZM33.3875 16V12.7273H34.0794V15.4295H35.4825V16H33.3875Z" fill="black"/> <defs> <clipPath id="clip0_217_4105"> <rect width="15" height="15" fill="white" transform="translate(26.5032 34) rotate(180)"/> </clipPath> </defs> </svg> ',

			'ayurvedic'         => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>ayurvedic</defs> <rect x="0.00317383" width="38" height="38" rx="19" fill="#009669"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M22.4112 26.5425H13.2074C10.3378 26.5425 8.00316 24.2079 8.00316 21.3382V16.2637C8.00316 16.0276 8.1952 15.8356 8.43129 15.8356H27.1872C27.4233 15.8356 27.6154 16.0276 27.6154 16.2637V21.3382C27.6154 24.2079 25.2809 26.5425 22.4112 26.5425ZM8.46151 16.2939V21.3381C8.46151 23.9551 10.5905 26.084 13.2074 26.084L22.4112 26.0841C25.0281 26.0841 27.1571 23.955 27.1571 21.3382V16.2939L8.46151 16.2939Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M22.4112 25.5111C22.2846 25.5111 22.182 25.4085 22.182 25.2819C22.182 25.1554 22.2846 25.0528 22.4112 25.0528C24.4594 25.0528 26.1258 23.3863 26.1258 21.3381C26.1258 21.2115 26.2284 21.1089 26.3549 21.1089C26.4815 21.1089 26.5841 21.2115 26.5841 21.3381C26.5841 23.6391 24.7121 25.5111 22.4112 25.5111Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M27.3862 18.2155H8.23231C8.10574 18.2155 8.00316 18.1129 8.00316 17.9863C8.00316 17.8597 8.10574 17.7571 8.23231 17.7571H27.3862C27.5128 17.7571 27.6154 17.8597 27.6154 17.9863C27.6154 18.1129 27.5128 18.2155 27.3862 18.2155Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M23.3068 30H12.3119C12.1853 30 12.0827 29.8974 12.0827 29.7708V26.2317C12.0827 26.1051 12.1853 26.0025 12.3119 26.0025C12.4385 26.0025 12.5411 26.1051 12.5411 26.2317V29.5417H23.0776V26.2317C23.0776 26.1051 23.1802 26.0025 23.3068 26.0025C23.4333 26.0025 23.5359 26.1051 23.5359 26.2317V29.7708C23.5359 29.8975 23.4334 30 23.3068 30Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M17.3792 25.0502C16.6305 25.0502 15.2099 24.5991 14.1079 21.578C14.1058 21.5722 14.1039 21.5665 14.1023 21.5606C13.5569 19.5905 13.6717 19.4395 13.7404 19.3494C13.8171 19.2485 13.9597 19.2202 14.0737 19.2857C14.1161 19.3102 14.1447 19.3369 14.1643 19.3819C14.1661 19.3846 14.6473 20.1842 17.2398 20.5731C19.0561 20.8455 19.6281 21.9502 19.5577 22.8931C19.48 23.9348 18.5861 25.0502 17.3792 25.0502ZM14.5415 21.4295C15.5474 24.1809 16.7509 24.5919 17.3792 24.5919C18.3316 24.5919 19.0382 23.6959 19.1006 22.859C19.1718 21.9033 18.4508 21.2183 17.1718 21.0264C15.5211 20.7788 14.6514 20.3662 14.2027 20.0444C14.2795 20.4242 14.4109 20.957 14.5415 21.4295Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M18.6195 24.5447C18.5222 24.5447 18.4319 24.4822 18.4011 24.3845C18.0208 23.1818 16.5261 22.3592 16.5111 22.3511C16.3999 22.2908 16.3586 22.1517 16.419 22.0404C16.4793 21.9292 16.6185 21.888 16.7297 21.9483C16.7978 21.9852 18.4026 22.8691 18.8381 24.2464C18.8763 24.367 18.8093 24.4959 18.6886 24.5339C18.6656 24.5413 18.6424 24.5447 18.6195 24.5447Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M18.24 25.0492C18.0757 25.0492 17.9114 25.0281 17.7516 24.9865C17.6506 24.9602 17.5801 24.8691 17.5801 24.7647C17.5801 24.6604 17.6506 24.5692 17.7516 24.543C18.4914 24.3504 19.0855 23.5729 19.104 22.773C19.1142 22.3297 18.9509 21.5301 17.7414 21.1549C17.6455 21.1252 17.5801 21.0365 17.5801 20.936C17.5801 20.8356 17.6455 20.7469 17.7414 20.7172C17.942 20.6549 18.1565 20.6065 18.3789 20.5731C20.9713 20.1842 21.4525 19.3846 21.457 19.3766C21.4767 19.3316 21.5024 19.3102 21.5449 19.2857C21.659 19.2202 21.8016 19.2486 21.8783 19.3493C21.9469 19.4396 22.0618 19.5905 21.5164 21.5606C21.5148 21.5665 21.5128 21.5722 21.5108 21.5779C20.4091 24.5982 18.9887 25.0492 18.24 25.0492ZM18.5476 21.0109C19.2172 21.4106 19.5796 22.0313 19.5622 22.7836C19.547 23.443 19.2314 24.0754 18.7519 24.5033C19.4081 24.2807 20.3014 23.551 21.077 21.4294C21.2076 20.9569 21.339 20.4242 21.4159 20.0443C20.9766 20.3595 20.1332 20.7618 18.5476 21.0109Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M23.4995 16.2938H19.838C19.7445 16.2938 19.6603 16.2369 19.6254 16.1502C19.5905 16.0634 19.6118 15.9641 19.6793 15.8993L23.8512 11.8923C23.662 11.0163 23.9298 10.0943 24.567 9.45716C25.0673 8.95688 25.7325 8.68134 26.4401 8.68134C27.1476 8.68134 27.8128 8.95688 28.3131 9.45716C29.3459 10.49 29.346 12.1705 28.3131 13.2033C27.8128 13.7035 27.1477 13.979 26.4401 13.9791C26.2509 13.9791 26.0624 13.959 25.8779 13.9191L23.6647 16.2234C23.6216 16.2684 23.5619 16.2938 23.4995 16.2938ZM20.4074 15.8355H23.4019L25.6387 13.5065C25.6968 13.446 25.7831 13.4222 25.864 13.4441C26.0515 13.495 26.2453 13.5207 26.4401 13.5207C27.0252 13.5207 27.5753 13.2929 27.989 12.8792C28.8431 12.0251 28.8431 10.6354 27.989 9.78126C27.5752 9.3675 27.0252 9.13966 26.4401 9.13966C25.8549 9.13966 25.3049 9.36753 24.891 9.78126C24.3372 10.3352 24.1208 11.1494 24.3262 11.9063C24.3481 11.9871 24.3242 12.0735 24.2638 12.1315L20.4074 15.8355Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M16.779 16.2938H14.3115C14.2334 16.2938 14.1607 16.254 14.1185 16.1882C13.4792 15.1901 13.1412 14.0334 13.1412 12.8429C13.1412 10.9837 13.9643 9.23759 15.3993 8.05242C15.484 7.98249 15.6065 7.98249 15.6912 8.05242C17.1262 9.23758 17.9493 10.9837 17.9493 12.843C17.9493 14.0334 17.6113 15.1902 16.972 16.1883C16.9299 16.254 16.8571 16.2938 16.779 16.2938ZM14.4385 15.8355H16.6521C17.2013 14.9357 17.491 13.9039 17.491 12.843C17.491 11.1849 16.7843 9.62405 15.5453 8.53043C14.3063 9.62411 13.5996 11.1849 13.5996 12.843C13.5996 13.9039 13.8893 14.9358 14.4385 15.8355Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M15.5453 16.2939C15.4187 16.2939 15.3162 16.1913 15.3162 16.0647V11.2613C15.3162 11.1347 15.4187 11.0321 15.5453 11.0321C15.6719 11.0321 15.7745 11.1347 15.7745 11.2613V16.0647C15.7745 16.1913 15.6718 16.2939 15.5453 16.2939Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M15.5453 14.7367C15.4803 14.7367 15.4184 14.7091 15.375 14.6608L14.4853 13.6723C14.4006 13.5782 14.4082 13.4333 14.5023 13.3486C14.5964 13.264 14.7412 13.2716 14.8259 13.3656L15.5453 14.1649L16.2647 13.3656C16.3494 13.2716 16.4943 13.2639 16.5884 13.3486C16.6824 13.4333 16.6901 13.5782 16.6054 13.6723L15.7156 14.6608C15.6722 14.7091 15.6102 14.7367 15.5453 14.7367Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M15.5453 16.2938C15.4803 16.2938 15.4184 16.2663 15.375 16.218L14.4853 15.2294C14.4006 15.1354 14.4082 14.9905 14.5023 14.9058C14.5964 14.8211 14.7412 14.8288 14.8259 14.9228L15.5453 15.7221L16.2647 14.9228C16.3494 14.8288 16.4943 14.8211 16.5884 14.9058C16.6824 14.9904 16.6901 15.1354 16.6054 15.2294L15.7156 16.218C15.6722 16.2663 15.6102 16.2938 15.5453 16.2938Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M15.5453 13.1794C15.4803 13.1794 15.4184 13.1518 15.375 13.1035L14.4853 12.1149C14.4006 12.0208 14.4082 11.876 14.5023 11.7913C14.5964 11.7066 14.7413 11.7142 14.8259 11.8083L15.5453 12.6076L16.2647 11.8083C16.3494 11.7143 16.4943 11.7066 16.5884 11.7913C16.6824 11.876 16.6901 12.0209 16.6054 12.1149L15.7156 13.1035C15.6722 13.1518 15.6102 13.1794 15.5453 13.1794Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M14.3127 16.2938C14.3122 16.2938 14.3131 16.2938 14.3127 16.2938H11.1057C11.0513 16.2938 10.9987 16.2745 10.9573 16.2392C10.8278 16.1291 10.7019 16.013 10.583 15.8942C9.26824 14.5795 8.61558 12.7629 8.79225 10.9101C8.80268 10.8007 8.88921 10.7142 8.9986 10.7037C9.19477 10.6851 9.39416 10.6756 9.59133 10.6756C11.0329 10.6756 12.4364 11.1797 13.5434 12.095C13.602 12.1434 13.6327 12.2176 13.6256 12.2932C13.6083 12.4755 13.5995 12.6605 13.5995 12.843C13.5995 13.9388 13.9085 15.0036 14.4934 15.9238C14.5238 15.9626 14.5418 16.0116 14.5418 16.0647C14.5418 16.1913 14.4393 16.2938 14.3127 16.2938ZM11.1909 15.8356H13.9087C13.4055 14.9221 13.1412 13.8957 13.1412 12.843C13.1412 12.6872 13.1472 12.5297 13.1589 12.3733C12.056 11.4997 10.652 11.0571 9.2334 11.145C9.13058 12.7944 9.7346 14.3978 10.907 15.5701C10.9978 15.6609 11.093 15.7499 11.1909 15.8356Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M14.1532 16.2938C14.0946 16.2938 14.036 16.2715 13.9912 16.2267L11.0024 13.2379C10.9128 13.1484 10.9128 13.0033 11.0024 12.9138C11.0919 12.8243 11.237 12.8243 11.3265 12.9138L14.3153 15.9026C14.4049 15.9921 14.4049 16.1372 14.3153 16.2267C14.2705 16.2715 14.2119 16.2938 14.1532 16.2938Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M13.6071 15.7476C13.603 15.7476 13.5989 15.7475 13.5948 15.7473L12.2667 15.6774C12.1403 15.6707 12.0432 15.5629 12.0499 15.4365C12.0565 15.3101 12.1648 15.2129 12.2907 15.2197L13.6189 15.2896C13.7453 15.2962 13.8424 15.4041 13.8357 15.5305C13.8293 15.6527 13.7281 15.7476 13.6071 15.7476Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M12.472 14.6125C12.468 14.6125 12.464 14.6125 12.4599 14.6122L11.1318 14.5424C11.0054 14.5357 10.9083 14.4279 10.915 14.3015C10.9216 14.1751 11.0301 14.0779 11.1559 14.0847L12.2298 14.1412L12.1733 13.0673C12.1666 12.9409 12.2637 12.8331 12.3901 12.8265C12.5167 12.8194 12.6243 12.9169 12.6309 13.0433L12.7008 14.3714C12.7042 14.4362 12.6799 14.4995 12.634 14.5455C12.5909 14.5885 12.5326 14.6125 12.472 14.6125Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M20.9574 28.2304H14.6612C14.5347 28.2304 14.4321 28.1278 14.4321 28.0012C14.4321 27.8746 14.5347 27.7721 14.6612 27.7721H20.9574C21.084 27.7721 21.1865 27.8746 21.1865 28.0012C21.1865 28.1278 21.0839 28.2304 20.9574 28.2304Z" fill="white"/> </svg> ',

			'wild-caught'       => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>wild-caught</defs> <rect x="0.00317383" width="38" height="38" rx="19" fill="#58C3FF"/> <g clip-path="url(#clip0_222_4263)"> <path d="M28.0032 15C27.7832 15 27.6032 15.18 27.6032 15.4V18C27.0032 18.06 26.6632 18.42 26.3832 18.72C26.1032 19.02 25.9232 19.2 25.5432 19.2C25.1632 19.2 24.9832 19.02 24.7033 18.72C24.4032 18.4 24.0232 18 23.2832 18C22.5432 18 22.1632 18.4 21.8632 18.72C21.5632 19.02 21.3832 19.2 21.0032 19.2C20.7832 19.2 20.6032 19.38 20.6032 19.6C20.6032 19.82 20.7832 20 21.0032 20C21.7432 20 22.1232 19.6 22.4232 19.28C22.7032 18.98 22.8832 18.8 23.2631 18.8C23.6432 18.8 23.8232 18.98 24.1031 19.28C24.4232 19.6 24.8032 20 25.5431 20C26.2831 20 26.6631 19.6 26.9631 19.28C27.1831 19.0401 27.3431 18.88 27.6031 18.82L27.6032 21.2C27.0032 21.26 26.6632 21.62 26.3832 21.92C26.1032 22.22 25.9231 22.4 25.5432 22.4C25.1632 22.4 24.9832 22.22 24.7032 21.92C24.4032 21.6 24.0232 21.2 23.2832 21.2C22.5432 21.2 22.1632 21.6 21.8632 21.92C21.5632 22.22 21.3832 22.4 21.0032 22.4C20.6232 22.4 20.4431 22.22 20.1632 21.92C20.0832 21.84 20.0032 21.76 19.9232 21.68C19.7632 21.52 19.5033 21.54 19.3632 21.7C19.2032 21.86 19.2232 22.12 19.3832 22.26L19.5832 22.46C19.8832 22.78 20.2632 23.18 21.0032 23.18C21.7431 23.18 22.1232 22.78 22.4232 22.46C22.7032 22.16 22.8832 21.98 23.2631 21.98C23.6431 21.98 23.8232 22.16 24.1031 22.46C24.4031 22.78 24.7831 23.18 25.5231 23.18C26.263 23.18 26.6431 22.78 26.9431 22.46C27.163 22.2201 27.3231 22.06 27.583 22V22.74C27.583 23.32 27.263 23.86 26.7431 24.14L24.7831 25.22L24.6631 25.1C24.3631 24.78 23.9831 24.38 23.2431 24.38C22.5031 24.38 22.1231 24.78 21.8231 25.1C21.5431 25.4 21.3631 25.58 20.9831 25.58C20.6031 25.58 20.4231 25.4 20.1432 25.1C19.8432 24.78 19.4632 24.38 18.7232 24.38C17.9832 24.38 17.6032 24.78 17.3032 25.1C17.0232 25.4 16.8432 25.58 16.4632 25.58C16.0832 25.58 15.9032 25.4 15.6233 25.1L15.5632 25.04C15.4032 24.8799 15.1632 24.8799 15.0032 25.02C14.8432 25.18 14.8432 25.42 14.9832 25.58L15.0432 25.64C15.3432 25.96 15.7233 26.36 16.4632 26.36C17.2032 26.36 17.5832 25.96 17.8832 25.64C18.1632 25.34 18.3432 25.16 18.7232 25.16C19.1032 25.16 19.2832 25.34 19.5631 25.64C19.8631 25.96 20.2432 26.36 20.9831 26.36C21.7231 26.36 22.1031 25.96 22.4031 25.64C22.6831 25.34 22.8631 25.16 23.2431 25.16C23.6031 25.16 23.783 25.3201 24.0631 25.6L19.7632 28C19.2832 28.26 18.7032 28.26 18.2233 28L11.2233 24.16C10.7033 23.88 10.4033 23.34 10.4033 22.76L10.4032 19.08C10.5232 19.14 10.6032 19.22 10.7232 19.3C10.9832 19.52 11.3232 19.8 11.9432 19.8C12.5632 19.8 12.9032 19.52 13.1632 19.3C13.3632 19.14 13.5031 19.02 13.7832 19C14.0031 18.98 14.1632 18.78 14.1432 18.56C14.1232 18.34 13.9232 18.18 13.7032 18.2C13.1832 18.24 12.9032 18.48 12.6633 18.68C12.4433 18.86 12.2832 18.98 11.9433 18.98C11.6033 18.98 11.4433 18.86 11.2233 18.68C11.0233 18.52 10.7833 18.32 10.4033 18.24V15.24C10.4033 14.66 10.7233 14.12 11.2233 13.84L18.2433 10C18.7233 9.73999 19.3033 9.73999 19.7833 10L24.1633 12.4C23.9433 12.98 22.9233 13.62 22.0033 14.2C20.9232 14.88 19.9033 15.54 19.5433 16.32C18.7233 16.1801 16.7633 16.06 15.3833 17.44C14.1433 18.68 14.1232 20.36 14.2232 21.3C13.5232 21.14 12.5833 21.18 11.7832 21.96C11.6832 22.06 11.6432 22.18 11.6632 22.3C11.6832 22.42 11.7632 22.54 11.8832 22.6C11.8832 22.6 12.6432 22.98 13.1832 23.54C13.7432 24.1 14.0432 24.74 14.0432 24.76C14.1033 24.8799 14.2033 24.96 14.3432 24.9799H14.4033C14.5033 24.9799 14.6033 24.94 14.6833 24.86C15.4633 24.0799 15.5033 23.14 15.3433 22.44C15.5433 22.4599 15.7833 22.4799 16.0433 22.4799C17.0034 22.4799 18.2433 22.2599 19.2233 21.2799C20.6233 19.8799 20.4633 17.8799 20.3233 17.0799C20.3033 16.9799 20.2633 16.8799 20.2233 16.7799C20.3433 16.1799 21.4433 15.4999 22.4233 14.8799C23.7233 14.0599 24.9633 13.2799 25.0033 12.2399V12.2199V12.2V12.18V12.14V12.1C25.0033 12.08 25.0033 12.08 24.9833 12.0601C24.9833 12.0401 24.9833 12.0401 24.9634 12.0201C24.9634 12.0001 24.9434 12.0001 24.9434 11.9801C24.9434 11.9601 24.9234 11.9601 24.9234 11.9601C24.9234 11.9401 24.9034 11.9401 24.9034 11.9201L24.8834 11.9002C24.8834 11.9002 24.8634 11.8802 24.8434 11.8802C24.8234 11.8802 24.8234 11.8602 24.8035 11.8602L24.7835 11.8402L20.1633 9.30003C19.4433 8.90003 18.5833 8.90003 17.8633 9.30003L10.8633 13.14C10.0833 13.5601 9.60327 14.3601 9.60327 15.24V22.76C9.60327 23.64 10.0833 24.44 10.8432 24.86L17.8432 28.7C18.2033 28.9 18.6033 29 19.0033 29C19.4033 29 19.8033 28.9 20.1633 28.7L24.9633 26.06H24.9833L27.1833 24.86C27.9433 24.44 28.4233 23.64 28.4233 22.76V15.4C28.4033 15.18 28.2233 15 28.0033 15L28.0032 15ZM19.4432 17.12C19.4831 17.12 19.5432 17.18 19.5432 17.22C19.6232 17.72 19.7432 18.78 19.3032 19.76C18.6632 19.84 18.0032 19.62 17.5232 19.14C17.1232 18.74 16.8632 17.98 16.8232 17.4C17.8232 16.94 18.9232 17.04 19.4432 17.12H19.4432ZM14.4632 23.86C14.2832 23.6 14.0632 23.28 13.7632 22.98C13.4432 22.66 13.0832 22.4 12.7832 22.22C13.4832 21.8599 14.1832 22.12 14.4432 22.24C14.5632 22.48 14.8231 23.16 14.4632 23.86ZM15.0632 21.58C14.9432 20.96 14.7432 19.18 15.9232 18C15.9632 17.96 16.0232 17.9 16.0832 17.86C16.2032 18.5201 16.5032 19.2401 16.9632 19.68C17.4632 20.18 18.1233 20.48 18.8032 20.5601C18.7633 20.6 18.7232 20.6601 18.6833 20.7C17.4832 21.9 15.7033 21.7 15.0633 21.58L15.0632 21.58Z" fill="white"/> <path d="M17.9432 18.08C18.1032 18.08 18.2432 17.94 18.2432 17.78C18.2432 17.62 18.1032 17.48 17.9432 17.48C17.7831 17.48 17.6432 17.62 17.6432 17.78C17.6632 17.94 17.7832 18.08 17.9432 18.08Z" fill="white"/> </g> <defs> <clipPath id="clip0_222_4263"> <rect width="20" height="20" fill="white" transform="translate(9.00317 9)"/> </clipPath> </defs> </svg> ',

			'oil-free'          => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>oil-free</defs> <rect width="38" height="38" rx="19" fill="#F2C142"/> <path d="M32.7071 5.70709L26.7071 11.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M11.7071 26.7071L5.70709 32.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M19.2516 27C17.2952 27 15.4736 26.1005 14.2818 24.5488C12.5727 22.3226 12.5727 19.1968 14.2818 16.9705L18.8918 11.1687C19.0717 10.9438 19.4315 10.9438 19.5889 11.1687L24.1989 16.948C25.9079 19.1968 25.9079 22.3001 24.1989 24.5264C23.0295 26.1005 21.208 27 19.2516 27ZM19.2516 12.1581L15.0014 17.5102C13.5397 19.4216 13.5397 22.0752 15.0014 23.9866C16.0134 25.3134 17.565 26.078 19.2516 26.078C20.9382 26.078 22.4673 25.3134 23.5018 23.9866C24.9635 22.0752 24.9635 19.3992 23.5018 17.4877L19.2516 12.1581Z" fill="white" stroke="white" stroke-width="0.224877" stroke-miterlimit="10"/> <path d="M19.9263 25.0661C19.6789 25.0661 19.4765 24.8637 19.4765 24.6163C19.4765 24.3689 19.6789 24.1666 19.9263 24.1666C21.5454 24.1666 22.8497 22.8623 22.8497 21.2431C22.8497 20.9958 23.052 20.7934 23.2994 20.7934C23.5468 20.7934 23.7492 20.9958 23.7492 21.2431C23.7492 23.357 22.0401 25.0661 19.9263 25.0661Z" fill="white" stroke="white" stroke-width="0.224877" stroke-miterlimit="10"/> </svg> ',

			'peanut-free'       => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>peanut-free</defs> <rect width="38" height="38" rx="19" fill="#C27363"/> <path d="M26.1105 14.0371C25.1149 13.0415 23.647 12.7615 22.2521 13.151L22.3056 13.1267C22.0208 12.4889 21.568 11.9533 20.9935 11.5784C19.1556 10.3783 16.5436 11.1062 15.1706 13.2021C14.7665 13.8229 14.4987 14.5264 14.394 15.2348C14.3015 15.8653 13.8926 16.4033 13.2961 16.6759C12.3467 17.1093 11.5312 17.8201 10.9348 18.733C9.38172 21.1065 9.80773 24.1445 11.8867 25.5053C12.5488 25.9386 13.3254 26.1602 14.1336 26.1602C14.3234 26.1602 14.5133 26.1431 14.7056 26.1188C15.4359 26.6933 16.3439 27 17.3323 27C17.4516 27 17.5733 26.9951 17.6974 26.9854C18.973 26.8905 20.2048 26.3086 21.1664 25.3471C21.9381 24.5754 22.4663 23.6309 22.6951 22.6157C22.8388 21.9755 23.2818 21.4619 23.8807 21.2452C24.555 20.9993 25.1903 20.5904 25.7137 20.067C27.4859 18.2948 27.6636 15.5902 26.1105 14.0371ZM11.7503 19.2661C12.2445 18.509 12.9188 17.9199 13.6978 17.5645C14.5961 17.1579 15.2168 16.34 15.358 15.3784C15.4408 14.8064 15.6599 14.2392 15.9861 13.7353C17.0645 12.0872 19.0728 11.4859 20.4604 12.3939C20.8645 12.6593 21.1859 13.0366 21.3952 13.4845C20.9254 13.7207 20.4799 14.0322 20.0807 14.4315C19.5573 14.9549 19.1483 15.5902 18.9024 16.2645C18.6833 16.861 18.1721 17.304 17.5319 17.4476C16.5168 17.6765 15.5723 18.2047 14.8006 18.9764C13.839 19.938 13.2548 21.1722 13.1623 22.4453C13.0868 23.4556 13.3278 24.3953 13.839 25.1694C13.3278 25.1231 12.8434 24.9625 12.4246 24.6874C10.7961 23.6211 10.4942 21.1868 11.7552 19.2612L11.7503 19.2661ZM25.0248 19.3781C24.5988 19.8041 24.09 20.1327 23.5472 20.3299C22.6367 20.661 21.9648 21.4351 21.7457 22.4015C21.5583 23.2365 21.1201 24.0179 20.4775 24.6582C19.679 25.4566 18.6663 25.9362 17.6244 26.0141C16.6069 26.0896 15.6891 25.7682 15.0318 25.1109C14.3746 24.4537 14.0557 23.5335 14.1287 22.5184C14.2066 21.4765 14.6886 20.4638 15.4846 19.6653C16.1249 19.0251 16.9063 18.5869 17.7413 18.397C18.7053 18.1804 19.4818 17.5061 19.8129 16.5956C20.0101 16.0528 20.3411 15.5415 20.7647 15.118C21.5388 14.3438 22.5272 13.947 23.4522 13.947C24.1923 13.947 24.8958 14.2027 25.4167 14.7236C26.5901 15.897 26.4124 17.9856 25.0199 19.3756L25.0248 19.3781ZM23.3987 15.5196C23.4474 15.785 23.2721 16.0381 23.0067 16.0868C21.8918 16.2913 21.6703 17.1993 21.6605 17.2383C21.6094 17.4647 21.4074 17.618 21.1859 17.618C21.1518 17.618 21.1177 17.6156 21.0812 17.6059C20.8207 17.5499 20.6527 17.2943 20.7087 17.0338C20.7209 16.9705 21.0568 15.4539 22.8315 15.1277C23.0992 15.079 23.35 15.2543 23.3987 15.5196ZM18.0821 19.8528C18.1989 20.0962 18.0942 20.3859 17.8532 20.5027C15.9325 21.4205 15.9934 23.1854 15.9983 23.2608C16.0104 23.5286 15.806 23.7575 15.5382 23.7721C15.5284 23.7721 15.5211 23.7721 15.5114 23.7721C15.2558 23.7721 15.0416 23.5749 15.027 23.3144C15.0197 23.2146 14.9101 20.8314 17.4345 19.6239C17.6755 19.5071 17.9676 19.6093 18.0845 19.8528H18.0821Z" fill="white"/> <path d="M32.7071 5.70709L26.7071 11.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> <path d="M10.7071 27.7071L5.70709 32.7071" stroke="white" stroke-width="2" stroke-linecap="round"/> </svg> ',

			'smoothies'         => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>smoothies</defs> <rect width="38" height="38" rx="19" fill="#FF6E4E"/> <path d="M22.8539 10.1922C22.5253 10.1922 22.23 10.3748 22.0798 10.6758L20.7365 13.5624C20.7009 13.4688 20.6589 13.3777 20.6124 13.2884C20.5662 12.0545 19.1743 10.3679 19.1142 10.2953L18.8681 10L18.6547 10.32C18.1428 11.088 17.4386 11.2833 16.8173 11.4562C16.4158 11.5677 16.0076 11.6826 15.7419 11.9554C15.2086 11.2141 14.363 10.7666 13.4361 10.7666C12.6442 10.7666 12 11.4108 12 12.2027C12 13.5733 12.9659 14.7211 14.2521 15.0057C13.4961 15.2447 13.1489 15.5425 13.1489 15.9366V24.8405C13.1489 25.7525 13.7598 26.5234 14.5933 26.7687C14.5893 26.7957 14.585 26.823 14.585 26.8511V27.4256C14.585 27.7424 14.8426 28 15.1595 28H20.3295C20.6463 28 20.9039 27.7424 20.9039 27.4256V26.8511C20.9039 26.823 20.8996 26.7957 20.8956 26.7687C21.7291 26.5234 22.34 25.7525 22.34 24.8405V15.9366C22.34 15.6684 22.1697 15.4317 21.838 15.2298L23.3347 12.0226C23.3534 11.9824 23.3982 11.9155 23.4889 11.9155H26.6484V10.1922H22.8539ZM22.8539 10.7666H23.2017V11.4071C23.1153 11.4444 23.0366 11.4961 22.9693 11.5642L22.5124 11.1072L22.5971 10.9254C22.6456 10.8275 22.7442 10.7666 22.8539 10.7666ZM20.0423 22.83V17.199C20.7411 17.0847 21.3503 16.9129 21.7656 16.6794V24.5533C20.8152 24.5533 20.0423 23.7804 20.0423 22.83ZM16.0211 22.83V17.2782C16.5772 17.3411 17.166 17.3727 17.7445 17.3727C18.3229 17.3727 18.9117 17.3411 19.4678 17.2782V22.83V23.4044C19.4678 24.3548 18.6949 25.1278 17.7445 25.1278C16.794 25.1278 16.0211 24.3548 16.0211 23.4044V22.83ZM13.7233 24.5533V16.6791C14.1387 16.9126 14.7479 17.0847 15.4467 17.1987V22.83C15.4467 23.7804 14.6738 24.5533 13.7233 24.5533ZM17.7445 16.7983C16.3152 16.7983 15.2629 16.6228 14.5925 16.4209C14.6672 15.791 15.5343 15.6491 16.925 15.4992C18.1397 15.3679 19.4997 15.2209 20.1221 14.3644C20.8686 15.0477 20.9025 15.768 20.9039 16.4189C20.2338 16.6216 19.1786 16.7983 17.7445 16.7983ZM20.1497 13.6477L19.9222 13.4768L19.7783 13.8129C19.4187 14.6519 18.1196 14.792 16.8636 14.9279C16.5028 14.967 16.1395 15.0066 15.7982 15.0649C15.4636 14.8569 15.447 14.5129 15.4467 14.5005C15.4467 14.0455 16.0766 13.9203 17.114 13.7687C18.0041 13.6383 18.9996 13.4926 19.5787 12.8952C19.836 13.125 20.0253 13.3749 20.1497 13.6477ZM20.9588 14.4459L22.2544 11.6616L22.6735 12.0806L21.2828 15.0609C21.2075 14.8572 21.1041 14.6513 20.9588 14.4459ZM16.9707 12.0097C17.5397 11.8514 18.2939 11.6421 18.8968 10.9484C19.1499 11.2865 19.4816 11.7785 19.7229 12.2722C19.7177 12.2682 19.7134 12.2642 19.7082 12.2602L19.4621 12.0755L19.2946 12.3337C18.912 12.9248 17.9125 13.071 17.0304 13.2005C16.7915 13.2356 16.5511 13.2712 16.3213 13.316C16.0234 13.1281 16.0211 12.9354 16.0211 12.7772C16.0211 12.3124 16.2532 12.2087 16.9707 12.0097ZM12.5744 12.2027C12.5744 11.7276 12.961 11.341 13.4361 11.341C14.2912 11.341 15.0618 11.8181 15.4582 12.5862C15.4513 12.6465 15.4467 12.7094 15.4467 12.7772C15.4467 12.9274 15.4478 13.2123 15.6661 13.4941C15.613 13.5145 15.5616 13.5366 15.5125 13.5601C15.1465 13.146 14.2007 12.1697 13.2184 11.9241L13.0794 12.4813C13.8802 12.6815 14.7102 13.5268 15.0552 13.912C14.9392 14.0665 14.8722 14.2584 14.8722 14.5005C13.6053 14.5005 12.5744 13.4697 12.5744 12.2027ZM14.4871 15.5336C14.2639 15.7008 14.1076 15.9205 14.0447 16.2126C13.8336 16.1075 13.7233 16.0084 13.7233 15.9366C13.7233 15.9343 13.7518 15.7534 14.4871 15.5336ZM20.3301 27.4256H15.1595V26.8511H20.3295L20.3301 27.4256ZM20.3295 26.2767H15.1595C14.4655 26.2767 13.885 25.7818 13.7523 25.1263C14.5049 25.1169 15.1698 24.7437 15.5825 24.175C15.9002 25.0634 16.7481 25.7022 17.7445 25.7022C18.7408 25.7022 19.5887 25.0634 19.9064 24.175C20.3191 24.7437 20.9841 25.1171 21.7366 25.1263C21.6039 25.7818 21.0234 26.2767 20.3295 26.2767ZM21.7656 15.9366C21.7656 16.005 21.6665 16.0978 21.4758 16.1971C21.4744 16.1394 21.4721 16.0805 21.4689 16.0211L21.5935 15.7539C21.7271 15.845 21.7656 15.9139 21.7656 15.9366ZM26.0739 11.341H23.7762V10.7666H26.0739V11.341Z" fill="white"/> <path d="M18.8934 17.9472H18.3189V20.8194H18.8934V17.9472Z" fill="white"/> <path d="M18.8934 21.3939H18.3189V22.5428H18.8934V21.3939Z" fill="white"/> <path d="M21.1911 17.6599H20.6167V19.6705H21.1911V17.6599Z" fill="white"/> <path d="M21.1911 20.245H20.6167V20.8194H21.1911V20.245Z" fill="white"/> <path d="M14.8722 17.6599H14.2978V19.6705H14.8722V17.6599Z" fill="white"/> <path d="M14.8722 20.245H14.2978V20.8194H14.8722V20.245Z" fill="white"/> </svg> ',

			'juice'             => '<svg width="38"g height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>juice</defs> <rect width="38" height="38" rx="19" fill="#58C3FF"/> <path d="M25.9195 11.4082L21.9847 9.04736C21.8535 8.94899 21.6568 9.01457 21.5584 9.14573L18.8696 12.9166H16.8694C16.8038 12.228 16.4759 11.6378 16.0169 11.1787C15.4922 10.6213 14.7709 10.2934 13.9511 10.2934C13.1314 10.2934 12.41 10.6213 11.8853 11.1459C11.3279 11.7034 11 12.4247 11 13.2445C11 14.0642 11.3279 14.7856 11.8525 15.3103C12.3116 15.7693 12.9346 16.0644 13.5904 16.1628V25.7047C13.5904 25.967 13.6888 26.1965 13.8527 26.3605C13.6888 26.5244 13.5904 26.754 13.5904 27.0163C13.5904 27.5737 14.0167 28 14.5741 28H22.4437C23.0012 28 23.4274 27.5737 23.4274 27.0163C23.4274 26.754 23.3291 26.5244 23.1651 26.3605C23.3291 26.1965 23.4274 25.967 23.4274 25.7047V13.2445C23.4274 13.0477 23.2963 12.9166 23.0995 12.9166H21.2961L22.3126 11.5394L24.8702 13.0805C24.9358 13.1133 25.0341 13.1461 25.1325 13.1133C25.2309 13.0805 25.2965 13.0477 25.3292 12.9494L26.0178 11.8345C26.1162 11.7034 26.0834 11.5066 25.9195 11.4082ZM12.1149 14.6217C11.8853 14.3266 11.7214 13.9659 11.6886 13.5724H13.1641L12.1149 14.6217ZM12.1149 11.8673L13.1641 12.9166H11.6886C11.7542 12.5231 11.8853 12.1624 12.1149 11.8673ZM14.279 10.982C14.6725 11.0476 15.0332 11.1787 15.3283 11.4082L14.279 12.4575V10.982ZM13.6232 12.4575L12.5739 11.4082C12.869 11.1787 13.2297 11.0148 13.6232 10.982V12.4575ZM15.7873 11.8673C16.0169 12.1624 16.1808 12.5231 16.2136 12.9166H14.7381L15.7873 11.8673ZM12.5739 15.0807L13.6232 14.0314V15.507C13.2297 15.4414 12.869 15.3103 12.5739 15.0807ZM22.4765 27.3442H14.6069C14.4102 27.3442 14.279 27.213 14.279 27.0163C14.279 26.8196 14.4102 26.6884 14.6069 26.6884H22.4765C22.6733 26.6884 22.8044 26.8196 22.8044 27.0163C22.8044 27.213 22.6733 27.3442 22.4765 27.3442ZM22.8044 25.7047C22.8044 25.9014 22.6733 26.0326 22.4765 26.0326H14.6069C14.4102 26.0326 14.279 25.9014 14.279 25.7047V13.5724H22.8044V25.7047ZM24.9686 12.3592L22.4437 10.818C22.3126 10.7197 22.1158 10.7524 22.0174 10.9164L20.5091 12.9166H19.6566L21.8863 9.73595L25.2637 11.7689L24.9686 12.3592Z" fill="white"/> <path d="M15.0332 16.294C14.9676 16.3595 14.9348 16.4579 14.9348 16.5235V25.0489C14.9348 25.2456 15.066 25.3768 15.2627 25.3768H21.8207C22.0175 25.3768 22.1486 25.2456 22.1486 25.0489V16.5235C22.1486 16.3923 22.0503 16.2612 21.9519 16.2284C21.8207 16.1956 21.6896 16.2284 21.5912 16.3268C21.5584 16.3923 20.4763 17.5728 18.7385 16.294C16.5415 14.6217 15.0332 16.294 15.0332 16.294ZM21.4928 17.1793V24.721H15.5906V16.6547C15.8857 16.3923 16.8694 15.671 18.345 16.7858C19.7222 17.8351 20.8698 17.54 21.4928 17.1793Z" fill="white"/> <path d="M17.8859 18.163H16.5743C16.3776 18.163 16.2464 18.2942 16.2464 18.4909V19.8025C16.2464 19.9992 16.3776 20.1304 16.5743 20.1304H17.8859C18.0826 20.1304 18.2138 19.9992 18.2138 19.8025V18.4909C18.2138 18.2942 18.0826 18.163 17.8859 18.163ZM17.558 19.4746H16.9022V18.8188H17.558V19.4746Z" fill="white"/> <path d="M19.1975 22.0978H20.5091C20.7059 22.0978 20.837 21.9666 20.837 21.7699V20.4583C20.837 20.2616 20.7059 20.1304 20.5091 20.1304H19.1975C19.0008 20.1304 18.8696 20.2616 18.8696 20.4583V21.7699C18.8696 21.9666 19.0008 22.0978 19.1975 22.0978ZM19.5254 20.7862H20.1812V21.442H19.5254V20.7862Z" fill="white"/> <path d="M17.8859 22.0978H16.5743C16.3776 22.0978 16.2464 22.229 16.2464 22.4257V23.7373C16.2464 23.9341 16.3776 24.0652 16.5743 24.0652H17.8859C18.0826 24.0652 18.2138 23.9341 18.2138 23.7373V22.4257C18.2138 22.229 18.0826 22.0978 17.8859 22.0978ZM17.558 23.4094H16.9022V22.7536H17.558V23.4094Z" fill="white"/> </svg> ',

			'bbq'               => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>bbq</defs> <rect width="38" height="38" rx="19" fill="#FF3C5F"/> <path d="M27.3353 15.1499C27.3353 13.6359 25.0832 12.4247 21.7714 11.9516C21.8092 12.1976 21.8281 12.4247 21.7903 12.6708C24.7426 13.125 26.6351 14.1658 26.6351 15.131C26.6351 16.399 23.3611 17.8373 18.6677 17.8373C13.9743 17.8373 10.7002 16.4368 10.7002 15.1499C10.7002 14.1469 12.7063 13.0682 15.8289 12.6329C15.8289 12.614 15.8289 12.5951 15.8478 12.5951L15.9425 12.3869C15.8478 12.2544 15.7721 12.103 15.6964 11.9516C12.3089 12.4058 10 13.6359 10 15.1689C10 19.7109 12.9145 21.4141 15.7532 22.0008L13.9175 27.5269C13.8607 27.7161 13.9553 27.9054 14.1446 27.9811C14.1824 28 14.2203 28 14.2581 28C14.4095 28 14.542 27.9054 14.5988 27.754L15.4693 25.1045H21.9038L22.7744 27.754C22.8312 27.9054 22.9636 28 23.115 28C23.1529 28 23.1907 28 23.2286 27.9811C23.4178 27.9243 23.5125 27.7161 23.4557 27.5269L21.62 21.9629C24.4209 21.3952 27.3353 19.6919 27.3353 15.1499ZM21.6389 24.4043H15.6964L16.4534 22.0954C17.2483 22.209 18.0053 22.2468 18.6677 22.2468C19.33 22.2468 20.087 22.209 20.8819 22.0954L21.6389 24.4043ZM18.6677 21.5655C14.1257 21.5655 11.4572 19.8812 10.8516 16.6829C12.2332 17.8184 15.1476 18.5754 18.6677 18.5754C22.1877 18.5754 25.1211 17.8184 26.4837 16.6829C25.8781 19.8812 23.2097 21.5655 18.6677 21.5655Z" fill="white"/> <path d="M20.2763 12.614L20.1438 12.9168C20.0681 13.0871 20.1438 13.2953 20.3331 13.371C20.3709 13.3899 20.4277 13.3899 20.4655 13.3899C20.598 13.3899 20.7305 13.3142 20.7873 13.1817L20.9197 12.8789C21.109 12.4247 21.0333 11.9137 20.7305 11.5163C20.5602 11.3081 20.5412 11.0243 20.6926 10.7972L20.844 10.5511C20.9387 10.3808 20.9008 10.1726 20.7305 10.0591C20.5602 9.94553 20.352 10.0023 20.2384 10.1726L20.087 10.4187C19.7842 10.8918 19.8221 11.5163 20.1817 11.9516C20.3331 12.1598 20.3709 12.4058 20.2763 12.614Z" fill="white"/> <path d="M18.4784 12.6139L18.3459 12.9167C18.2702 13.0871 18.3459 13.2952 18.5352 13.3709C18.573 13.3899 18.6298 13.3899 18.6677 13.3899C18.8001 13.3899 18.9326 13.3142 18.9894 13.1817L19.1219 12.8789C19.3111 12.4247 19.2354 11.9137 18.9326 11.5163C18.7623 11.3081 18.7434 11.0242 18.8948 10.7971L19.0462 10.5511C19.1408 10.3808 19.1029 10.1726 18.9326 10.0591C18.7623 9.96443 18.5541 10.0023 18.4406 10.1726L18.2892 10.4376C17.9864 10.9107 18.0242 11.5352 18.3838 11.9705C18.5352 12.1597 18.573 12.4058 18.4784 12.6139Z" fill="white"/> <path d="M16.6995 12.614L16.567 12.9168C16.4913 13.0871 16.567 13.2953 16.7562 13.371C16.7941 13.3899 16.8509 13.3899 16.8887 13.3899C17.0212 13.3899 17.1537 13.3142 17.2104 13.1817L17.3429 12.8789C17.5322 12.4247 17.4565 11.9137 17.1537 11.5163C16.9833 11.3081 16.9644 11.0243 17.1158 10.7972L17.2672 10.5511C17.3618 10.3808 17.324 10.1726 17.1537 10.0591C16.9833 9.94553 16.7752 10.0023 16.6616 10.1726L16.5102 10.4187C16.2074 10.8918 16.2453 11.5163 16.6048 11.9516C16.7562 12.1598 16.7941 12.4058 16.6995 12.614Z" fill="white"/> <path d="M23.6639 13.8819C23.4936 13.7684 23.2854 13.8252 23.1718 13.9955L23.0204 14.2226C22.7366 13.9576 22.3202 13.8252 21.9228 13.9198L21.8092 13.9387C20.8251 14.1469 20.2952 14.2604 19.0462 14.2604C17.7403 14.2604 16.4913 14.1469 15.4315 13.9198C15.1098 13.8441 14.7691 13.9198 14.4852 14.0901C14.4285 14.128 14.3717 14.1658 14.3149 14.2226L14.1635 13.9766C14.05 13.8062 13.8418 13.7684 13.6715 13.863C13.5011 13.9766 13.4633 14.1847 13.5579 14.3551L13.7282 14.62L13.4254 14.6579C13.2362 14.6768 13.0848 14.8471 13.1037 15.0364C13.1226 15.2256 13.274 15.3581 13.4633 15.3581H13.5011L13.9364 15.3202C13.9553 15.4906 14.031 15.6609 14.1257 15.8312C14.3149 16.1151 14.5799 16.3043 14.9205 16.38C16.1506 16.645 17.5889 16.7775 19.0462 16.7775C20.5602 16.7775 21.2604 16.6261 22.3391 16.399L22.4527 16.38C22.7744 16.3043 23.0583 16.1151 23.2475 15.8312C23.3422 15.6798 23.4179 15.5095 23.4368 15.3202L23.8721 15.3581H23.9099C24.0802 15.3581 24.2506 15.2256 24.2695 15.0364C24.2884 14.8471 24.137 14.6768 23.9478 14.6579L23.645 14.6389L23.8153 14.374C23.8721 14.2037 23.8342 13.9766 23.6639 13.8819ZM22.6419 15.4338C22.5662 15.5473 22.4337 15.642 22.3013 15.6798L22.1877 15.6987C21.1658 15.9069 20.4845 16.0583 19.0462 16.0583C17.6268 16.0583 16.2453 15.9258 15.053 15.6798C14.9016 15.642 14.788 15.5663 14.7123 15.4338C14.6366 15.3202 14.5988 15.1688 14.6366 15.0174C14.6745 14.866 14.7502 14.7525 14.8827 14.6768C14.9773 14.62 15.0719 14.5822 15.1855 14.5822C15.2233 14.5822 15.2612 14.5822 15.299 14.6011C15.3936 14.62 15.4883 14.6389 15.5829 14.6579L15.5072 14.9796C15.4693 15.1688 15.5829 15.3581 15.7721 15.3959C15.7911 15.3959 15.8289 15.3959 15.8478 15.3959C15.9992 15.3959 16.1506 15.2824 16.1885 15.131L16.2642 14.7714C16.4913 14.8093 16.7184 14.8282 16.9644 14.8471L16.9266 15.1499C16.9076 15.3392 17.0401 15.5095 17.2483 15.5284C17.2672 15.5284 17.2672 15.5284 17.2861 15.5284C17.4565 15.5284 17.6268 15.3959 17.6268 15.2067L17.6646 14.885C17.8917 14.9039 18.1188 14.9039 18.3459 14.9228V15.2256C18.3459 15.4149 18.4973 15.5852 18.7055 15.5852C18.9137 15.5852 19.0651 15.4338 19.0651 15.2256V14.9607H19.084C19.33 14.9607 19.5382 14.9607 19.7464 14.9417L19.7653 15.2256C19.7842 15.4149 19.9356 15.5473 20.106 15.5473C20.1249 15.5473 20.1249 15.5473 20.1438 15.5473C20.3331 15.5284 20.4845 15.3581 20.4655 15.1688L20.4466 14.9039C20.6926 14.885 20.9197 14.8471 21.1468 14.8093L21.2225 15.131C21.2604 15.3013 21.4118 15.3959 21.5632 15.3959C21.5821 15.3959 21.62 15.3959 21.6389 15.3959C21.8281 15.3581 21.9417 15.1688 21.9038 14.9796L21.7903 14.6579C21.8471 14.6389 21.9038 14.6389 21.9606 14.62L22.0742 14.6011C22.377 14.5443 22.6609 14.7336 22.7176 15.0174C22.7366 15.1688 22.7176 15.3202 22.6419 15.4338Z" fill="white"/> </svg> ',

			'salad'             => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>salad</defs> <rect width="38" height="38" rx="19" fill="#2DB68D"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M22.735 12.3299C22.2456 12.8233 21.8834 13.431 21.5938 14.0586C21.11 13.8297 20.5885 13.7441 20.0983 13.8753C19.3596 14.0734 18.8235 14.6753 18.5485 15.4222C18.2735 16.169 18.2385 17.0766 18.4848 17.9959C18.7117 18.8424 19.1316 19.7415 19.5944 20.5H17.228C17.0404 18.8162 15.6076 17.5 13.875 17.5C12.1423 17.5 10.7096 18.8162 10.522 20.5H10.125C9.50812 20.5 9 21.0081 9 21.625C9 25.1414 11.8586 28 15.375 28H23.625C27.1413 28 30 25.1414 30 21.625C30 21.0081 29.4918 20.5 28.875 20.5H27.0498C27.7779 19.8402 28.3083 18.9514 28.4545 18.0764C28.5898 17.2921 28.4269 16.5029 27.8862 15.9619H27.8854C27.5294 15.6061 27.0275 15.4511 26.499 15.4053C26.5804 14.8043 26.5812 14.1109 26.4038 13.4322C26.1863 12.6005 25.7346 11.7858 24.8921 11.5601C23.9875 11.3177 23.237 11.833 22.735 12.3299ZM24.6979 12.2845C25.1515 12.406 25.495 12.9232 25.6779 13.6226C25.8608 14.322 25.8581 15.1542 25.7372 15.639C25.6751 15.8871 25.8741 16.1232 26.1291 16.1041C26.6565 16.0646 27.074 16.2116 27.3551 16.4923C27.6929 16.8301 27.8202 17.3418 27.7155 17.9491C27.6107 18.5563 27.2641 19.2354 26.6923 19.8072C26.4551 20.0444 26.1734 20.2775 25.8749 20.5001H22.8844L23.0741 19.7926L26.0368 18.3731C26.2234 18.2835 26.3021 18.0596 26.2126 17.8729C26.1229 17.6863 25.899 17.6076 25.7123 17.6971L23.3298 18.839L24.4774 14.555C24.531 14.3549 24.4123 14.1494 24.2123 14.0957C23.9677 14.0489 23.8195 14.1683 23.7531 14.3609L22.6054 18.6449L21.1127 16.4644C20.9545 16.2542 20.739 16.2742 20.592 16.367C20.4212 16.4838 20.3773 16.7168 20.4938 16.8878L22.3498 19.5985L22.1081 20.5001H20.4857C19.9428 19.6764 19.4417 18.7309 19.2091 17.8018C18.9998 17.0208 19.0394 16.259 19.2523 15.6808C19.4652 15.1025 19.8309 14.7234 20.2924 14.5997C20.6763 14.497 21.1112 14.5785 21.5485 14.8766C21.7462 15.0119 22.0184 14.9296 22.108 14.7074C22.3725 14.0519 22.792 13.3374 23.2675 12.858C23.7479 12.3483 24.2641 12.1454 24.6979 12.2845ZM13.8751 18.25C15.2015 18.25 16.2893 19.2252 16.4708 20.5H15.712C15.5369 19.647 14.7778 19 13.8751 19C12.9724 19 12.2133 19.647 12.0382 20.5H11.2794C11.4609 19.2252 12.5487 18.25 13.8751 18.25ZM13.8751 19.75C14.3682 19.75 14.7784 20.0623 14.932 20.5H12.8182C12.9718 20.0623 13.3819 19.75 13.8751 19.75ZM10.1251 21.25H28.875C29.0866 21.25 29.25 21.4134 29.25 21.625C29.25 24.736 26.7361 27.2499 23.6251 27.2499H15.3751C12.264 27.2499 9.75008 24.736 9.75008 21.625C9.75008 21.4134 9.91353 21.25 10.1251 21.25Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M16.4999 13C15.8831 13 15.3749 13.5082 15.3749 14.125V16.375C15.3749 16.9918 15.8831 17.5 16.4999 17.5C17.1167 17.5 17.6249 16.9918 17.6249 16.375V14.125C17.6249 13.5082 17.1167 13 16.4999 13ZM16.4999 13.75C16.7141 13.75 16.8749 13.9108 16.8749 14.125V16.375C16.8749 16.5892 16.7141 16.75 16.4999 16.75C16.2857 16.75 16.1249 16.5892 16.1249 16.375V14.125C16.1249 13.9108 16.2857 13.75 16.4999 13.75Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M11.061 11.5506C10.8819 11.6543 10.8206 11.8834 10.924 12.0626L11.1306 12.42C10.4793 12.9981 10.2983 13.979 10.7497 14.7608C11.0075 15.2073 11.4665 15.6295 11.989 15.9239C12.5114 16.2183 13.1415 16.4111 13.686 16.0968C14.2304 15.7824 14.3784 15.1401 14.3847 14.5404C14.391 13.9407 14.255 13.3323 13.9973 12.8858C13.4525 12.0063 12.5294 11.8046 11.7795 12.045L11.5729 11.6876C11.4355 11.4729 11.2114 11.4659 11.061 11.5506ZM13.3476 13.2608C13.504 13.5318 13.6396 14.0663 13.6347 14.533C13.6298 14.9997 13.4839 15.3472 13.311 15.4471C13.138 15.5469 12.7633 15.4997 12.3566 15.2706C11.95 15.0415 11.5558 14.6568 11.3994 14.3858C11.0865 13.8439 11.2691 13.1621 11.811 12.8492C12.4014 12.5125 13.0807 12.7675 13.3476 13.2608Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M19.1249 10C18.3009 10 17.6249 10.676 17.6249 11.5C17.6249 11.912 17.7738 12.3355 18.0197 12.6946C18.2656 13.0537 18.6374 13.375 19.1249 13.375C19.6124 13.375 19.9842 13.0537 20.2301 12.6946C20.476 12.3354 20.6249 11.912 20.6249 11.5C20.6249 10.676 19.9489 10 19.1249 10ZM19.1249 10.75C19.5436 10.75 19.8749 11.0813 19.8749 11.5C19.8749 11.7093 19.7725 12.0357 19.6112 12.2712C19.45 12.5068 19.2588 12.625 19.1249 12.625C18.991 12.625 18.7998 12.5068 18.6386 12.2712C18.4773 12.0357 18.3749 11.7093 18.3749 11.5C18.3749 11.0813 18.7063 10.75 19.1249 10.75Z" fill="white"/> </svg> ',

			'low-glycemic'      => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>low-glycemic</defs> <rect x="0.5" y="0.5" width="37" height="37" rx="18.5" fill="white"/> <rect x="0.5" y="0.5" width="37" height="37" rx="18.5" stroke="#FF6999"/> <mask id="mask0_242_10138" style="mask-type:alpha" maskUnits="userSpaceOnUse" x="0" y="0" width="38" height="38"> <rect width="38" height="38" rx="19" fill="#2DB68D"/> </mask> <g mask="url(#mask0_242_10138)"> <rect x="-1" y="19" width="39" height="19" fill="#FF6999"/> </g> <g clip-path="url(#clip0_242_10138)"> <path d="M16.1875 19L21.8125 19L21.8125 26.5L26.35 26.5L19 33.85L11.65 26.5L16.1875 26.5L16.1875 19Z" fill="white"/> </g> <path d="M13.2159 9V5.36364H13.7646V8.5277H15.4123V9H13.2159ZM19.0553 7.18182C19.0553 7.57008 18.9843 7.90388 18.8422 8.18324C18.7002 8.46141 18.5055 8.67566 18.2581 8.82599C18.0119 8.97514 17.7319 9.04972 17.4182 9.04972C17.1034 9.04972 16.8222 8.97514 16.5748 8.82599C16.3286 8.67566 16.1345 8.46082 15.9925 8.18146C15.8504 7.90211 15.7794 7.56889 15.7794 7.18182C15.7794 6.79356 15.8504 6.46035 15.9925 6.18217C16.1345 5.90282 16.3286 5.68857 16.5748 5.53942C16.8222 5.38909 17.1034 5.31392 17.4182 5.31392C17.7319 5.31392 18.0119 5.38909 18.2581 5.53942C18.5055 5.68857 18.7002 5.90282 18.8422 6.18217C18.9843 6.46035 19.0553 6.79356 19.0553 7.18182ZM18.512 7.18182C18.512 6.88589 18.464 6.63672 18.3682 6.4343C18.2735 6.23071 18.1433 6.07682 17.9775 5.97266C17.813 5.86731 17.6266 5.81463 17.4182 5.81463C17.2087 5.81463 17.0217 5.86731 16.8572 5.97266C16.6926 6.07682 16.5624 6.23071 16.4665 6.4343C16.3718 6.63672 16.3245 6.88589 16.3245 7.18182C16.3245 7.47775 16.3718 7.72751 16.4665 7.93111C16.5624 8.13352 16.6926 8.28741 16.8572 8.39276C17.0217 8.49692 17.2087 8.54901 17.4182 8.54901C17.6266 8.54901 17.813 8.49692 17.9775 8.39276C18.1433 8.28741 18.2735 8.13352 18.3682 7.93111C18.464 7.72751 18.512 7.47775 18.512 7.18182ZM20.4074 9L19.3989 5.36364H19.9759L20.6844 8.17969H20.7181L21.455 5.36364H22.0267L22.7636 8.18146H22.7973L23.504 5.36364H24.0828L23.0725 9H22.5203L21.7551 6.27628H21.7267L20.9614 9H20.4074Z" fill="#FF6999"/> <path d="M7.12427 13.693C7.09692 13.5979 7.05851 13.5139 7.00903 13.441C6.95955 13.3668 6.89901 13.3043 6.82739 13.2535C6.75708 13.2014 6.67635 13.1617 6.58521 13.1344C6.49536 13.107 6.39575 13.0934 6.28638 13.0934C6.08195 13.0934 5.90226 13.1441 5.74731 13.2457C5.59367 13.3473 5.47388 13.495 5.38794 13.6891C5.302 13.8818 5.25903 14.1174 5.25903 14.3961C5.25903 14.6747 5.30135 14.9117 5.38599 15.107C5.47062 15.3023 5.59041 15.4514 5.74536 15.5543C5.90031 15.6559 6.08325 15.7066 6.29419 15.7066C6.4856 15.7066 6.64901 15.6728 6.78442 15.6051C6.92114 15.5361 7.02531 15.4391 7.09692 15.3141C7.16984 15.1891 7.2063 15.0413 7.2063 14.8707L7.37817 14.8961H6.34692V14.2594H8.02075V14.7633C8.02075 15.1148 7.94653 15.4169 7.7981 15.6695C7.64966 15.9208 7.44523 16.1148 7.18481 16.2516C6.9244 16.387 6.62622 16.4547 6.29028 16.4547C5.91528 16.4547 5.58586 16.372 5.302 16.2066C5.01815 16.04 4.79679 15.8036 4.63794 15.4977C4.48039 15.1904 4.40161 14.8258 4.40161 14.4039C4.40161 14.0797 4.44849 13.7906 4.54224 13.5367C4.63729 13.2815 4.7701 13.0654 4.94067 12.8883C5.11125 12.7112 5.30981 12.5764 5.53638 12.484C5.76294 12.3915 6.00838 12.3453 6.27271 12.3453C6.49927 12.3453 6.71021 12.3785 6.90552 12.4449C7.10083 12.51 7.27401 12.6025 7.42505 12.7223C7.57739 12.8421 7.70174 12.9846 7.7981 13.15C7.89445 13.3141 7.9563 13.495 7.98364 13.693H7.12427ZM8.65601 16.4V12.4H9.50171V15.7027H11.2166V16.4H8.65601ZM11.032 12.4H11.9792L12.8914 14.1227H12.9304L13.8425 12.4H14.7898L13.3308 14.9859V16.4H12.491V14.9859L11.032 12.4ZM18.5393 13.8004H17.6838C17.6682 13.6897 17.6363 13.5914 17.5881 13.5055C17.54 13.4182 17.4781 13.344 17.4026 13.2828C17.3271 13.2216 17.2398 13.1747 17.1409 13.1422C17.0432 13.1096 16.9371 13.0934 16.8225 13.0934C16.6155 13.0934 16.4351 13.1448 16.2815 13.2476C16.1278 13.3492 16.0087 13.4977 15.9241 13.693C15.8394 13.887 15.7971 14.1227 15.7971 14.4C15.7971 14.6852 15.8394 14.9247 15.9241 15.1187C16.01 15.3128 16.1298 15.4592 16.2834 15.5582C16.4371 15.6572 16.6148 15.7066 16.8167 15.7066C16.9299 15.7066 17.0347 15.6917 17.1311 15.6617C17.2288 15.6318 17.3153 15.5881 17.3909 15.5309C17.4664 15.4723 17.5289 15.4013 17.5784 15.318C17.6292 15.2346 17.6643 15.1396 17.6838 15.0328L18.5393 15.0367C18.5172 15.2203 18.4618 15.3974 18.3733 15.568C18.2861 15.7372 18.1682 15.8889 18.0198 16.023C17.8726 16.1559 17.6969 16.2613 17.4924 16.3394C17.2893 16.4163 17.0595 16.4547 16.803 16.4547C16.4462 16.4547 16.1272 16.374 15.8459 16.2125C15.566 16.051 15.3446 15.8173 15.1819 15.5113C15.0204 15.2053 14.9397 14.8349 14.9397 14.4C14.9397 13.9638 15.0217 13.5927 15.1858 13.2867C15.3499 12.9807 15.5725 12.7477 15.8538 12.5875C16.135 12.426 16.4514 12.3453 16.803 12.3453C17.0347 12.3453 17.2496 12.3779 17.4475 12.443C17.6467 12.5081 17.8232 12.6031 17.9768 12.7281C18.1305 12.8518 18.2555 13.0035 18.3518 13.1832C18.4495 13.3629 18.512 13.5686 18.5393 13.8004ZM19.1458 16.4V12.4H21.8411V13.0973H19.9915V14.0504H21.7024V14.7477H19.9915V15.7027H21.8489V16.4H19.1458ZM22.5134 12.4H23.5564L24.658 15.0875H24.7048L25.8064 12.4H26.8494V16.4H26.0291V13.7965H25.9958L24.9607 16.3805H24.4021L23.3669 13.7867H23.3337V16.4H22.5134V12.4ZM28.3918 12.4V16.4H27.5461V12.4H28.3918ZM32.6169 13.8004H31.7615C31.7458 13.6897 31.7139 13.5914 31.6658 13.5055C31.6176 13.4182 31.5557 13.344 31.4802 13.2828C31.4047 13.2216 31.3175 13.1747 31.2185 13.1422C31.1208 13.1096 31.0147 13.0934 30.9001 13.0934C30.6931 13.0934 30.5128 13.1448 30.3591 13.2476C30.2055 13.3492 30.0863 13.4977 30.0017 13.693C29.9171 13.887 29.8748 14.1227 29.8748 14.4C29.8748 14.6852 29.9171 14.9247 30.0017 15.1187C30.0876 15.3128 30.2074 15.4592 30.3611 15.5582C30.5147 15.6572 30.6925 15.7066 30.8943 15.7066C31.0076 15.7066 31.1124 15.6917 31.2087 15.6617C31.3064 15.6318 31.393 15.5881 31.4685 15.5309C31.544 15.4723 31.6065 15.4013 31.656 15.318C31.7068 15.2346 31.7419 15.1396 31.7615 15.0328L32.6169 15.0367C32.5948 15.2203 32.5395 15.3974 32.4509 15.568C32.3637 15.7372 32.2458 15.8889 32.0974 16.023C31.9503 16.1559 31.7745 16.2613 31.5701 16.3394C31.3669 16.4163 31.1371 16.4547 30.8806 16.4547C30.5238 16.4547 30.2048 16.374 29.9236 16.2125C29.6436 16.051 29.4223 15.8173 29.2595 15.5113C29.0981 15.2053 29.0173 14.8349 29.0173 14.4C29.0173 13.9638 29.0994 13.5927 29.2634 13.2867C29.4275 12.9807 29.6501 12.7477 29.9314 12.5875C30.2126 12.426 30.5291 12.3453 30.8806 12.3453C31.1124 12.3453 31.3272 12.3779 31.5251 12.443C31.7244 12.5081 31.9008 12.6031 32.0544 12.7281C32.2081 12.8518 32.3331 13.0035 32.4294 13.1832C32.5271 13.3629 32.5896 13.5686 32.6169 13.8004Z" fill="black"/> <defs> <clipPath id="clip0_242_10138"> <rect width="15" height="15" fill="white" transform="translate(26.5 34) rotate(180)"/> </clipPath> </defs> </svg> ',

			'steamed'           => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>steamed</defs> <rect width="38" height="38" rx="19" fill="#FF7B5F"/> <path d="M29.5954 19.2523H28.5339H9.48693H8.42537C8.1906 19.2523 8.00006 19.4428 8.00006 19.6776C8.00006 19.9124 8.1906 20.1029 8.42537 20.1029H9.20453L11.5114 25.6149C11.5216 25.6421 11.5386 25.6625 11.5488 25.6897C11.5488 25.6931 11.5522 25.6965 11.5556 25.6999C12.1034 26.9248 13.2262 27.7686 14.5328 27.9592C14.7199 27.9864 14.9104 28 15.101 28H22.9164C23.1104 28 23.2975 27.9864 23.4846 27.9592C24.7878 27.7686 25.9072 26.9282 26.4584 25.7067C26.4618 25.6999 26.4652 25.6965 26.4686 25.6897C26.4686 25.6897 26.4686 25.6897 26.4686 25.6863C26.4788 25.6625 26.4958 25.6387 26.506 25.6115L28.8129 20.0995H29.592C29.8268 20.0995 30.0173 19.909 30.0173 19.6742C30.0207 19.4428 29.8302 19.2523 29.5954 19.2523ZM17.401 21.7361H14.4715V20.1063H17.401V21.7361ZM25.8051 25.0943H12.2157L10.1266 20.1029H13.6209V22.1614V22.8759C13.6209 23.4339 14.0734 23.8864 14.6314 23.8864H17.2445C17.8025 23.8864 18.2551 23.4339 18.2551 22.8759V22.1614V20.1029H27.8976L25.8051 25.0943Z" fill="white"/> <path d="M15.5229 14.8699C15.4821 15.5708 14.9207 16.0642 14.8968 16.0846C14.7199 16.2377 14.6995 16.5065 14.8526 16.6834C14.9377 16.7821 15.0568 16.8297 15.1758 16.8297C15.2745 16.8297 15.3732 16.7957 15.4548 16.7243C15.5433 16.6494 16.3089 15.9553 16.3735 14.9176C16.4143 14.2507 16.1591 13.6144 15.6182 13.019C15.2609 12.6277 15.0976 12.2364 15.1282 11.8554C15.1792 11.2191 15.744 10.7666 15.7509 10.7598C15.9346 10.6169 15.9686 10.3481 15.8257 10.1643C15.6828 9.97721 15.414 9.94659 15.2303 10.0895C15.1963 10.1167 14.3695 10.7666 14.2844 11.7771C14.23 12.4066 14.4681 13.019 14.9921 13.594C15.3698 14.0125 15.5501 14.4412 15.5229 14.8699Z" fill="white"/> <path d="M18.3674 15.217C18.3435 15.6185 18.0645 15.9587 17.9182 16.0846C17.7413 16.2377 17.7209 16.5065 17.874 16.6834C17.9591 16.7821 18.0781 16.8297 18.1972 16.8297C18.2959 16.8297 18.3946 16.7957 18.4762 16.7243C18.5477 16.663 19.1669 16.105 19.218 15.2646C19.2486 14.7236 19.0444 14.2065 18.6089 13.7301C18.3537 13.4511 18.2347 13.1721 18.2551 12.9067C18.2891 12.4644 18.6906 12.1378 18.6974 12.131C18.8811 11.9881 18.9151 11.7193 18.7722 11.5355C18.6293 11.3484 18.3605 11.3178 18.1768 11.4607C18.1054 11.5151 17.4793 12.0255 17.4079 12.8285C17.3636 13.3422 17.5542 13.839 17.9795 14.3051C18.2551 14.6045 18.3844 14.9142 18.3674 15.217Z" fill="white"/> <path d="M12.396 15.217C12.3722 15.6185 12.0932 15.9587 11.9469 16.0846C11.77 16.2377 11.7495 16.5065 11.9027 16.6834C11.9877 16.7821 12.1068 16.8297 12.2259 16.8297C12.3246 16.8297 12.4232 16.7957 12.5049 16.7243C12.8213 16.4487 13.2092 15.9009 13.2466 15.2646C13.2772 14.7236 13.0731 14.2065 12.6376 13.7301C12.3824 13.4511 12.2633 13.1721 12.2837 12.9067C12.3178 12.4644 12.7192 12.1378 12.726 12.131C12.9098 11.9881 12.9438 11.7193 12.8009 11.5355C12.658 11.3484 12.3892 11.3178 12.2055 11.4607C12.134 11.5151 11.508 12.0255 11.4365 12.8285C11.3923 13.3422 11.5828 13.839 12.0081 14.3051C12.2837 14.6045 12.413 14.9108 12.396 15.217Z" fill="white"/> </svg> ',

			'baked'             => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>baked</defs> <rect width="38" height="38" rx="19" fill="#C27363"/> <g clip-path="url(#clip0_222_4492)"> <path fill-rule="evenodd" clip-rule="evenodd" d="M22.9554 27.9979H15.0294C14.8835 27.9979 14.7554 27.901 14.7156 27.7606L12.5636 20.1525C12.5358 20.0542 12.5557 19.9486 12.6174 19.8671C12.6791 19.7857 12.7752 19.7378 12.8774 19.7378L25.3213 19.7377C25.4249 19.7377 25.5223 19.787 25.5839 19.8705C25.6453 19.9539 25.6634 20.0616 25.6326 20.1606L23.2667 27.7687C23.2244 27.905 23.0981 27.9979 22.9554 27.9979ZM15.276 27.3458H22.7153L24.8785 20.3898H13.3085L15.276 27.3458Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M16.3525 27.998C16.1983 27.998 16.0612 27.8882 16.0323 27.7309L15.8445 26.7119C15.4463 24.5527 15.0347 22.3199 14.6323 20.1224C14.5998 19.9453 14.7172 19.7753 14.8943 19.7429C15.0712 19.7103 15.2412 19.8278 15.2738 20.0049C15.6762 22.202 16.0878 24.4346 16.4858 26.5937L16.6737 27.6127C16.7064 27.7899 16.5892 27.9598 16.4121 27.9925C16.392 27.9962 16.3721 27.998 16.3525 27.998Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M17.673 27.9979C17.5055 27.9979 17.3631 27.8696 17.3486 27.6997C17.2396 26.4259 17.1313 25.1531 17.0232 23.8802C16.9159 22.6176 16.8086 21.3549 16.7005 20.0916C16.6851 19.9121 16.8181 19.7543 16.9976 19.7389C17.177 19.7233 17.3349 19.8565 17.3503 20.0359C17.4584 21.2994 17.5656 22.5622 17.673 23.825C17.7811 25.0977 17.8893 26.3706 17.9983 27.6442C18.0136 27.8236 17.8807 27.9814 17.7013 27.9969C17.6917 27.9976 17.6824 27.9979 17.673 27.9979Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M18.9939 27.9979C18.9925 27.9979 18.991 27.9979 18.9895 27.9979C18.8094 27.9956 18.6653 27.8475 18.6677 27.6675C18.7012 25.1494 18.7381 22.591 18.7738 20.1168L18.7747 20.059C18.7773 19.8805 18.9227 19.7377 19.1006 19.7377C19.1022 19.7377 19.1037 19.7377 19.1053 19.7377C19.2854 19.7403 19.4292 19.8883 19.4266 20.0683L19.4257 20.1261C19.39 22.6002 19.353 25.1585 19.3197 27.6761C19.3175 27.8547 19.1719 27.9979 18.9939 27.9979Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M20.3148 27.9979C20.3026 27.9979 20.2902 27.9972 20.2778 27.9959C20.0989 27.9756 19.9702 27.8141 19.9905 27.6352L20.0662 26.9676C20.3238 24.6934 20.5901 22.3417 20.8494 20.0275C20.8695 19.8486 21.0309 19.7196 21.2098 19.7397C21.3886 19.7598 21.5175 19.9211 21.4975 20.1001C21.2382 22.4148 20.9717 24.7666 20.7141 27.041L20.6384 27.7086C20.6195 27.8751 20.4784 27.9979 20.3148 27.9979Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M21.6353 27.998C21.6128 27.998 21.5902 27.9957 21.5673 27.991C21.3911 27.9536 21.2786 27.7805 21.3159 27.6044C21.584 26.3402 21.8521 25.0767 22.1202 23.8132C22.39 22.5412 22.66 21.269 22.9299 19.9963C22.9672 19.8202 23.1405 19.7077 23.3165 19.7449C23.4926 19.7823 23.6052 19.9554 23.5679 20.1315C23.298 21.4044 23.0281 22.6764 22.7581 23.9485C22.49 25.212 22.2221 26.4754 21.9538 27.7396C21.9214 27.8928 21.7859 27.998 21.6353 27.998Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M25.3215 20.3898C25.2111 20.3898 25.1033 20.3337 25.042 20.2323C24.9489 20.0782 24.9985 19.8777 25.1526 19.7846C25.9324 19.3138 26.4561 18.4677 26.4561 17.6795C26.4561 16.339 25.3067 15.2483 23.8935 15.2483C23.7957 15.2483 23.7031 15.2044 23.6412 15.1287C23.5793 15.053 23.5545 14.9536 23.574 14.8576C23.6305 14.5778 23.6592 14.3002 23.6592 14.0327C23.6592 11.6203 21.595 9.65757 19.0578 9.65757C16.5291 9.65757 14.465 11.6125 14.4564 14.0155C14.4558 14.1718 14.3445 14.3057 14.191 14.3348C12.7224 14.6128 11.6565 15.8485 11.6565 17.2729C11.6565 18.2024 12.2214 19.2161 13.0621 19.7951C13.2105 19.8973 13.2479 20.1002 13.1457 20.2486C13.0437 20.3969 12.8406 20.4343 12.6923 20.3322C11.667 19.6259 11.0045 18.4251 11.0045 17.2729C11.0045 15.6205 12.1704 14.1784 13.8122 13.7518C13.8827 12.508 14.4242 11.3496 15.3539 10.4657C16.3444 9.52395 17.6598 9.00549 19.0579 9.00549C21.9548 9.00549 24.3114 11.2608 24.3114 14.0328C24.3114 14.2245 24.2988 14.4203 24.2739 14.6177C25.8679 14.7988 27.1084 16.1028 27.1084 17.6795C27.1084 18.7037 26.473 19.7491 25.4897 20.3428C25.4371 20.3747 25.3789 20.3898 25.3215 20.3898Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M18.2328 13.0079C17.5963 13.0079 17.0784 12.49 17.0784 11.8535C17.0784 11.217 17.5963 10.6991 18.2328 10.6991C18.8677 10.6991 19.3842 11.217 19.3842 11.8535C19.3842 12.49 18.8676 13.0079 18.2328 13.0079ZM18.2328 11.3512C17.9558 11.3512 17.7305 11.5765 17.7305 11.8535C17.7305 12.1305 17.9558 12.3558 18.2328 12.3558C18.5081 12.3558 18.7321 12.1305 18.7321 11.8535C18.7321 11.5765 18.5081 11.3512 18.2328 11.3512Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M14.5373 18.2929C13.9007 18.2929 13.3829 17.775 13.3829 17.1385C13.3829 16.502 13.9007 15.9841 14.5373 15.9841C15.1738 15.9841 15.6916 16.502 15.6916 17.1385C15.6916 17.775 15.1738 18.2929 14.5373 18.2929ZM14.5373 16.6362C14.2602 16.6362 14.0349 16.8615 14.0349 17.1385C14.0349 17.4155 14.2602 17.6408 14.5373 17.6408C14.8143 17.6408 15.0396 17.4155 15.0396 17.1385C15.0396 16.8615 14.8141 16.6362 14.5373 16.6362Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M23.8999 19.3108C23.2634 19.3108 22.7456 18.7943 22.7456 18.1594C22.7456 17.5229 23.2634 17.005 23.8999 17.005C24.5365 17.005 25.0543 17.5229 25.0543 18.1594C25.0543 18.7943 24.5365 19.3108 23.8999 19.3108ZM23.8999 17.6571C23.6229 17.6571 23.3976 17.8824 23.3976 18.1594C23.3976 18.4347 23.6229 18.6587 23.8999 18.6587C24.177 18.6587 24.4023 18.4347 24.4023 18.1594C24.4023 17.8824 24.1768 17.6571 23.8999 17.6571Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M19.5379 17.4982C18.9014 17.4982 18.3835 16.9803 18.3835 16.3438C18.3835 15.7073 18.9014 15.1894 19.5379 15.1894C20.1744 15.1894 20.6923 15.7073 20.6923 16.3438C20.6923 16.9803 20.1746 17.4982 19.5379 17.4982ZM19.5379 15.8415C19.2609 15.8415 19.0356 16.0667 19.0356 16.3438C19.0356 16.6208 19.2609 16.8461 19.5379 16.8461C19.8149 16.8461 20.0402 16.6208 20.0402 16.3438C20.0402 16.0667 19.8149 15.8415 19.5379 15.8415Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M22.0109 14.0808C21.9273 14.0808 21.8437 14.049 21.78 13.985L20.6551 12.8571C20.528 12.7295 20.5283 12.5231 20.6557 12.396C20.7833 12.2689 20.9897 12.2692 21.1168 12.3966L22.2417 13.5244C22.3688 13.652 22.3685 13.8584 22.2411 13.9855C22.1775 14.049 22.0942 14.0808 22.0109 14.0808Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M16.949 15.4501C16.8655 15.4501 16.782 15.4183 16.7183 15.3546L15.5905 14.2267C15.4632 14.0993 15.4632 13.8929 15.5905 13.7656C15.7177 13.6383 15.9243 13.6383 16.0516 13.7656L17.1794 14.8935C17.3067 15.0209 17.3067 15.2273 17.1794 15.3546C17.1159 15.4183 17.0324 15.4501 16.949 15.4501Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M18.1564 19.271C18.0728 19.271 17.9892 19.2391 17.9255 19.1751L16.8006 18.0473C16.6735 17.9197 16.6738 17.7133 16.8012 17.5862C16.9288 17.459 17.1352 17.4593 17.2623 17.5867L18.3871 18.7146C18.5143 18.8422 18.514 19.0486 18.3866 19.1757C18.3229 19.2393 18.2397 19.271 18.1564 19.271Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M22.8056 16.7218C22.7222 16.7218 22.6387 16.69 22.575 16.6263L21.4502 15.5014C21.3229 15.374 21.3229 15.1676 21.4502 15.0403C21.5775 14.9131 21.784 14.9131 21.9113 15.0403L23.0361 16.1652C23.1634 16.2926 23.1634 16.499 23.0361 16.6263C22.9726 16.69 22.889 16.7218 22.8056 16.7218Z" fill="white"/> </g> <defs> <clipPath id="clip0_222_4492"> <rect width="16.1126" height="19" fill="white" transform="translate(11 9)"/> </clipPath> </defs> </svg> ',

			'fried'             => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>fried</defs> <rect width="38" height="38" rx="19" fill="#FFBA34"/> <path d="M29.6797 13.6867C29.7037 13.7806 29.6901 13.8801 29.6416 13.964C29.5931 14.0479 29.5136 14.1095 29.4203 14.1355L23.7413 15.6685C23.72 15.5487 23.6958 15.4288 23.6636 15.3094L23.5676 14.9558L29.2272 13.428C29.3215 13.4035 29.4216 13.4169 29.5061 13.4652C29.5907 13.5135 29.653 13.593 29.6797 13.6867Z" fill="white"/> <path d="M20.2397 16.4269C20.2009 16.4427 20.1606 16.4584 20.121 16.4738C19.9474 16.5423 19.7704 16.6105 19.5865 16.6779L19.4857 16.7145C19.2478 16.8005 19.0027 16.8853 18.7504 16.9688L18.6848 16.99C18.4587 17.0633 18.2268 17.1366 17.9916 17.2099C17.9399 17.2253 17.8894 17.241 17.837 17.2564C17.575 17.3345 17.3097 17.4107 17.0405 17.4832C16.7712 17.5557 16.5037 17.6239 16.2388 17.6884L16.0798 17.725C15.8409 17.7822 15.6035 17.8349 15.3708 17.8862L15.306 17.9002C15.0456 17.9554 14.7903 18.0058 14.5402 18.0515L14.4391 18.0691C14.2442 18.1039 14.0551 18.1346 13.869 18.1632L13.7444 18.1823C14.7902 17.77 15.8587 17.418 16.9448 17.1282C18.0289 16.8318 19.129 16.5977 20.2397 16.4269Z" fill="white"/> <path d="M22.1086 15.5285C20.2946 15.6121 18.497 15.9111 16.7535 16.4192C14.9919 16.8614 13.2881 17.5087 11.6772 18.3479C11.3813 18.367 11.0847 18.3219 10.8078 18.2156C11.032 17.7946 13.0252 16.6676 16.5601 15.7121C20.095 14.7566 22.3856 14.7258 22.7916 14.9768C22.6059 15.2073 22.3731 15.3954 22.1086 15.5285Z" fill="white"/> <path d="M12.343 19.0649H12.3624C12.4291 19.0612 12.4928 19.0587 12.5625 19.0535C12.6779 19.0455 12.7999 19.0334 12.9211 19.0217L12.9402 19.0198C13.5477 18.9601 14.2226 18.8586 14.9403 18.7201C14.955 18.7172 14.9696 18.7139 14.9846 18.7113C15.1957 18.6701 15.4096 18.6257 15.6265 18.5783L15.757 18.5501C15.949 18.5071 16.1427 18.4621 16.3381 18.4149C16.3938 18.4017 16.448 18.3889 16.5044 18.375C16.745 18.3156 16.9873 18.2532 17.2313 18.1878C17.3291 18.1614 17.5376 18.1039 17.8175 18.0218C18.5041 17.8229 19.1622 17.6081 19.7748 17.385C19.8691 17.3508 19.9638 17.316 20.0591 17.2806L20.2126 17.2234C21.2144 16.8848 22.1595 16.3973 23.0162 15.7773C23.0608 16.0229 23.0773 16.2728 23.0653 16.5222C23.0609 16.5262 23.055 16.5273 23.051 16.5313C22.6245 16.9472 21.5792 17.4869 20.184 18.0126C20.1037 18.0428 20.0366 18.1003 19.9942 18.1749C19.9519 18.2496 19.9372 18.3367 19.9525 18.4211C19.9679 18.5055 20.0124 18.5819 20.0782 18.6369C20.1441 18.6919 20.2272 18.722 20.313 18.722C20.3572 18.7218 20.4009 18.7139 20.4423 18.6985C21.2678 18.4013 22.0654 18.0321 22.8261 17.595C22.6369 18.0677 22.3405 18.4901 21.9603 18.8288C21.5801 19.1675 21.1264 19.4132 20.6351 19.5467L14.9744 21.0745C14.1781 21.289 13.3303 21.1972 12.5983 20.8174C11.8663 20.4376 11.3031 19.7973 11.02 19.0228C11.2843 19.0647 11.5517 19.0846 11.8194 19.0821C11.981 19.0832 12.158 19.0763 12.343 19.0649ZM17.8721 19.1525C17.8944 19.2284 17.9406 19.2952 18.004 19.3427C18.0673 19.3902 18.1443 19.4158 18.2235 19.4159C18.2585 19.4157 18.2932 19.4107 18.3268 19.4009C18.5679 19.3298 18.8042 19.2575 19.0358 19.184C19.1284 19.1545 19.2055 19.0895 19.2502 19.0032C19.2948 18.9169 19.3034 18.8164 19.2739 18.7238C19.2445 18.6312 19.1795 18.5541 19.0932 18.5094C19.0069 18.4648 18.9064 18.4562 18.8137 18.4856C18.5878 18.5575 18.3566 18.6282 18.1202 18.6978C18.0739 18.7114 18.0308 18.7339 17.9933 18.7642C17.9558 18.7945 17.9246 18.8319 17.9016 18.8742C17.8785 18.9165 17.8641 18.963 17.859 19.011C17.854 19.0589 17.8584 19.1074 17.8721 19.1536V19.1525ZM15.7379 19.6793C15.7561 19.7605 15.8013 19.8331 15.8661 19.8852C15.931 19.9372 16.0116 19.9657 16.0948 19.9658C16.1218 19.9658 16.1487 19.9629 16.175 19.9571C16.412 19.9043 16.6526 19.8476 16.8968 19.7871C16.9897 19.7624 17.0691 19.7022 17.118 19.6194C17.1669 19.5367 17.1814 19.4381 17.1583 19.3448C17.1352 19.2516 17.0763 19.1711 16.9945 19.1208C16.9126 19.0705 16.8142 19.0544 16.7206 19.0759C16.4821 19.1349 16.2465 19.1902 16.0156 19.2419C15.9686 19.2523 15.924 19.272 15.8846 19.2998C15.8451 19.3275 15.8115 19.3627 15.7857 19.4035C15.7599 19.4442 15.7424 19.4897 15.7342 19.5372C15.726 19.5847 15.7273 19.6334 15.7379 19.6804V19.6793Z" fill="white"/> <path d="M14.505 8.54958C14.505 8.40382 14.5629 8.26403 14.666 8.16097C14.7691 8.0579 14.9088 8 15.0546 8H16.1538C16.2995 8 16.4393 8.0579 16.5424 8.16097C16.6454 8.26403 16.7033 8.40382 16.7033 8.54958C16.7033 8.69534 16.6454 8.83513 16.5424 8.93819C16.4393 9.04126 16.2995 9.09916 16.1538 9.09916H16.1244V10.0151H15.048V9.09916C14.9034 9.09743 14.7653 9.03876 14.6636 8.93588C14.562 8.833 14.505 8.6942 14.505 8.54958Z" fill="white"/> <path d="M12.019 13.0221L12.8962 11.9452L11.8197 11.0659L10.9404 12.1427L12.019 13.0221Z" fill="white"/> <path d="M18.9017 11.2975C19.0466 11.2975 19.1883 11.2545 19.3088 11.174C19.4293 11.0934 19.5232 10.979 19.5787 10.8451C19.6341 10.7112 19.6486 10.5639 19.6204 10.4217C19.5921 10.2796 19.5223 10.149 19.4198 10.0465C19.3173 9.94405 19.1868 9.87426 19.0446 9.84599C18.9025 9.81772 18.7551 9.83223 18.6212 9.88769C18.4874 9.94315 18.3729 10.0371 18.2924 10.1576C18.2119 10.2781 18.1689 10.4198 18.1689 10.5647C18.1689 10.759 18.2461 10.9454 18.3835 11.0828C18.5209 11.2203 18.7073 11.2975 18.9017 11.2975Z" fill="white"/> <path d="M18.5352 12.763H19.268V13.4958H18.5352V12.763Z" fill="white"/> <path d="M15.2378 12.0303H15.9706V12.763H15.2378V12.0303Z" fill="white"/> <path d="M15.2378 13.4958H15.9706V14.595H15.2378V13.4958Z" fill="white"/> <path d="M11.2075 13.8622H11.9403V14.595H11.2075V13.8622Z" fill="white"/> <path d="M11.2075 15.3277H11.9403V16.0605H11.2075V15.3277Z" fill="white"/> <path d="M22.5128 23.2102L23.3274 24.6756L22.6869 25.0316L21.8723 23.5662L22.5128 23.2102Z" fill="white"/> <path d="M23.9901 27.9483L23.6394 28.9998L22.9433 28.7683L23.2939 27.7168C23.3981 27.4035 23.436 27.072 23.4054 26.7433C23.3749 26.4146 23.2764 26.0958 23.1162 25.8071L23.7567 25.451C23.9662 25.8285 24.0951 26.2453 24.1353 26.6752C24.1754 27.105 24.126 27.5385 23.9901 27.9483Z" fill="white"/> <path d="M21.7056 27.7171C21.8094 27.4039 21.8472 27.0726 21.8167 26.7441C21.7861 26.4155 21.6878 26.0969 21.5279 25.8082L20.9688 24.8025V25.4034C20.9665 26.0768 20.7482 26.7318 20.3459 27.2719C20.158 27.5209 20.0234 27.8059 19.9506 28.1091C19.9703 28.2443 19.9826 28.3804 19.9873 28.5169H21.4378L21.7056 27.7171ZM19.2578 28.5177C19.2273 27.9345 19.0246 27.3735 18.6752 26.9056C18.2008 26.2707 17.9439 25.4998 17.9424 24.7072V23.7026L17.1364 25.1557C16.7947 25.773 16.6866 26.4928 16.8319 27.1833C16.8495 27.2677 16.8368 27.3555 16.796 27.4314C16.7551 27.5073 16.6889 27.5664 16.6088 27.5983C16.5288 27.6302 16.44 27.6329 16.3582 27.6058C16.2764 27.5788 16.2067 27.5237 16.1614 27.4504L16.1248 27.3866C16.1028 27.3469 16.0783 27.3086 16.0515 27.2719C15.6485 26.7322 15.4301 26.077 15.4287 25.4034V24.8021L14.8677 25.8064C14.7077 26.0952 14.6094 26.4141 14.5789 26.7428C14.5483 27.0715 14.5863 27.403 14.6904 27.7164L14.9571 28.5177H19.2578Z" fill="white"/> <path d="M13.9371 23.2102L14.5777 23.5661L14.086 24.451L13.4455 24.0951L13.9371 23.2102Z" fill="white"/> <path d="M12.6921 25.451L13.079 24.7549L13.7195 25.1106L13.3325 25.8068C13.1231 26.1834 13.0195 26.6097 13.0328 27.0404L12.3001 27.0639C12.2824 26.5007 12.4179 25.9433 12.6921 25.451Z" fill="white"/> <path d="M13.261 28.0355L13.5055 28.7682L12.8107 29L12.5663 28.2674L13.261 28.0355Z" fill="white"/> <path d="M21.7876 21.759L22.1539 22.4918L21.4985 22.8195L21.1321 22.0867L21.7876 21.759Z" fill="white"/> <path d="M9.47635 13.022C9.21052 12.7551 9.04505 12.4046 9.00793 12.0297C8.9708 11.6548 9.06432 11.2786 9.27264 10.9648L9.43055 10.7281L10.0402 11.134L9.88231 11.3715C9.76764 11.5442 9.71617 11.7513 9.7366 11.9577C9.75703 12.164 9.84811 12.357 9.99442 12.504L9.47635 13.022Z" fill="white"/> </svg> ',

			'grilled'           => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>grilled</defs> <rect width="38" height="38" rx="19" fill="#FF7B5F"/> <path d="M28.1034 9.51724H9.89655C9.66897 9.51724 9.51724 9.66897 9.51724 9.89655V21.2759C9.51724 21.5034 9.66897 21.6552 9.89655 21.6552H28.1034C28.331 21.6552 28.4828 21.5034 28.4828 21.2759V9.89655C28.4828 9.66897 28.331 9.51724 28.1034 9.51724ZM23.7414 17.8621H19.4931C18.9241 17.8621 18.431 18.0138 18.0138 18.3172C17.4069 18.7724 16.7241 19 15.9655 19C14.9793 19 14.069 18.5828 13.4241 17.8621C12.8552 17.2172 12.5517 16.4207 12.5517 15.5483C12.5517 15.4345 12.5517 15.3207 12.5897 15.169C12.7414 13.6517 13.9931 12.4 15.5103 12.1724C15.6241 12.1724 15.8138 12.1724 15.9655 12.1724C17.0276 12.1724 18.0138 12.6655 18.6586 13.5C18.6586 13.5379 18.6966 13.5379 18.6966 13.5759C19.1517 14.1069 19.8345 14.4483 20.5552 14.4483H23.7034C24.6517 14.4483 25.4103 15.2069 25.4103 16.1552C25.4103 17.1034 24.6897 17.8621 23.7414 17.8621ZM23.1724 13.6897H21.6552V10.2759H23.1724V13.6897ZM20.8966 13.6897H20.5931C20.1379 13.6897 19.6828 13.5 19.3793 13.1586V10.2759H20.8966V13.6897ZM18.6207 12.3621C18.1655 11.9828 17.6724 11.7172 17.1034 11.5655V10.2759H18.6207V12.3621ZM16.3448 11.4517C16.0414 11.4138 15.7 11.4138 15.3586 11.4517C15.169 11.4897 14.9793 11.5276 14.8276 11.5655V10.2759H16.3448V11.4517ZM14.069 11.869C13.4621 12.1724 12.931 12.6276 12.5517 13.1966V10.2759H14.069V11.7931C14.069 11.831 14.069 11.831 14.069 11.869ZM12.5517 17.9759C12.6276 18.1276 12.7414 18.2414 12.8552 18.3552C13.1966 18.7345 13.6138 19.0759 14.069 19.3034C14.069 19.3414 14.069 19.3414 14.069 19.3793V20.8966H12.5517V17.9759ZM14.8276 19.6069C15.2069 19.7207 15.5862 19.7586 15.9655 19.7586C16.0793 19.7586 16.231 19.7586 16.3448 19.7586V20.8966H14.8276V19.6069ZM17.1034 19.6069C17.5966 19.4552 18.0517 19.2655 18.469 18.9241C18.5069 18.8862 18.5828 18.8483 18.6207 18.8103V20.8966H17.1034V19.6069ZM19.3793 18.6207C19.4172 18.6207 19.4552 18.6207 19.4931 18.6207H20.8966V20.8966H19.3793V18.6207ZM21.6552 18.6207H23.1724V20.8966H21.6552V18.6207ZM23.931 18.6207C24.5 18.5828 25.031 18.3172 25.4483 17.9379V20.8966H23.931V18.6207ZM25.4483 14.3724C25.031 13.9931 24.5 13.7276 23.931 13.6897V10.2759H25.4483V14.3724ZM10.2759 10.2759H11.7931V20.8966H10.2759V10.2759ZM27.7241 20.8966H26.2069V10.2759H27.7241V20.8966Z" fill="white"/> <path d="M28.1034 8H9.89655C8.83448 8 8 8.83448 8 9.89655V21.2759C8 22.3 8.83448 23.1724 9.89655 23.1724H17.1034C17.2931 23.1724 17.4828 23.3241 17.4828 23.5517V24.2345C17.2552 24.4241 17.1034 24.7276 17.1034 25.069V28.8621C17.1034 29.5069 17.5966 30 18.2414 30H19.7586C20.4034 30 20.8966 29.5069 20.8966 28.8621V25.069C20.8966 24.7276 20.7448 24.4241 20.5172 24.2345V23.5517C20.5172 23.3621 20.669 23.1724 20.8966 23.1724H28.1034C29.1276 23.1724 30 22.3379 30 21.2759V9.89655C30 8.83448 29.1655 8 28.1034 8ZM20.1379 28.8621C20.1379 29.0517 19.9862 29.2414 19.7586 29.2414H18.2414C18.0517 29.2414 17.8621 29.0897 17.8621 28.8621V25.069C17.8621 24.8793 18.0138 24.6897 18.2414 24.6897H19.7586C19.9483 24.6897 20.1379 24.8414 20.1379 25.069V28.8621ZM29.2414 21.2759C29.2414 21.8828 28.7483 22.4138 28.1034 22.4138H20.8966C20.2517 22.4138 19.7586 22.9069 19.7586 23.5517V23.931H18.2414V23.5517C18.2414 22.9069 17.7483 22.4138 17.1034 22.4138H9.89655C9.28966 22.4138 8.75862 21.9207 8.75862 21.2759V9.89655C8.75862 9.28966 9.25172 8.75862 9.89655 8.75862H28.1034C28.7103 8.75862 29.2414 9.25172 29.2414 9.89655V21.2759Z" fill="white"/> <path d="M15.9655 13.6897C14.9034 13.6897 14.069 14.5241 14.069 15.5862C14.069 16.6483 14.9034 17.4828 15.9655 17.4828C17.0276 17.4828 17.8621 16.6483 17.8621 15.5862C17.8621 14.5241 17.0276 13.6897 15.9655 13.6897ZM15.9655 16.7241C15.3207 16.7241 14.8276 16.231 14.8276 15.5862C14.8276 14.9414 15.3207 14.4483 15.9655 14.4483C16.6103 14.4483 17.1034 14.9414 17.1034 15.5862C17.1034 16.231 16.6103 16.7241 15.9655 16.7241Z" fill="white"/> </svg> ',

			'pork'              => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>pork</defs> <rect width="38" height="38" rx="19" fill="#FF3C5F"/> <path d="M27.398 13.3793C28.7197 11.5557 28.6183 8.51774 28.6132 8.38742L28.5982 8L28.2205 8.08727C28.096 8.11606 25.2536 8.78899 23.8104 10.4665C22.4452 9.79621 20.9112 9.41909 19.2905 9.41909C17.6988 9.41909 16.1903 9.78247 14.8436 10.4305C13.391 8.77938 10.5886 8.11585 10.465 8.08727L10.0872 8L10.0722 8.38742C10.0672 8.5161 9.96844 11.4795 11.2383 13.3097C9.8381 15.0678 9.00003 17.2925 9.00003 19.7095C9.00003 25.3837 13.6163 30 19.2905 30C24.9647 30 29.581 25.3837 29.581 19.7095C29.581 17.3244 28.7651 15.1263 27.398 13.3793ZM27.9716 8.82C27.9524 9.65538 27.8121 11.5889 26.9705 12.8684C26.8512 12.7346 26.7286 12.6038 26.6025 12.4763C27.1054 11.7056 27.0667 10.5297 27.0646 10.4777L27.0496 10.0903L26.6719 10.1776C26.6225 10.189 25.5351 10.4468 24.9135 11.0955C24.7437 10.9844 24.5708 10.8776 24.3943 10.7764C25.4195 9.67394 27.1878 9.05292 27.9716 8.82ZM26.1259 12.0244C25.9077 11.8302 25.6817 11.6446 25.4475 11.4692C25.725 11.2136 26.1153 11.0364 26.4084 10.9294C26.3815 11.2485 26.3099 11.6857 26.1259 12.0244ZM10.7138 8.81969C11.4882 9.04932 13.2238 9.65759 14.2551 10.7381C14.0771 10.8384 13.9022 10.9434 13.7311 11.0537C13.1038 10.4356 12.0619 10.1887 12.0136 10.1776L11.6358 10.0903L11.6208 10.4777C11.6189 10.5283 11.5821 11.6418 12.0426 12.4119C11.9154 12.5382 11.7914 12.6676 11.6709 12.8004C10.8682 11.5211 10.7327 9.64018 10.7138 8.81969ZM12.5267 11.9609C12.3667 11.6321 12.3022 11.2281 12.2769 10.9286C12.5678 11.036 12.9249 11.2035 13.1899 11.4279C12.9615 11.5966 12.7402 11.7743 12.5267 11.9609ZM19.2905 29.3584C13.9701 29.3584 9.64162 25.03 9.64162 19.7095C9.64162 14.3891 13.9701 10.0607 19.2905 10.0607C24.6109 10.0607 28.9394 14.3891 28.9394 19.7095C28.9394 25.03 24.6109 29.3584 19.2905 29.3584Z" fill="white"/> <path d="M15.8465 17.4786C15.8465 16.8019 15.296 16.2515 14.6194 16.2515C13.9427 16.2515 13.3923 16.8019 13.3923 17.4786C13.3923 18.1552 13.9427 18.7057 14.6194 18.7057C15.296 18.7057 15.8465 18.1552 15.8465 17.4786Z" fill="white"/> <path d="M23.8919 16.2515C23.2152 16.2515 22.6648 16.8019 22.6648 17.4786C22.6648 18.1552 23.2152 18.7057 23.8919 18.7057C24.5685 18.7057 25.119 18.1552 25.119 17.4786C25.119 16.8019 24.5685 16.2515 23.8919 16.2515Z" fill="white"/> <path d="M22.8124 20.9923C21.8486 20.3329 20.5731 19.9697 19.2208 19.9697C17.8685 19.9697 16.593 20.3329 15.6292 20.9923C14.6282 21.6772 14.0769 22.5999 14.0769 23.5905C14.0769 24.5811 14.6282 25.5038 15.6292 26.1887C16.593 26.8481 17.8685 27.2113 19.2208 27.2113C20.5731 27.2113 21.8486 26.8481 22.8124 26.1887C23.8134 25.5038 24.3646 24.5811 24.3646 23.5905C24.3646 22.5999 23.8134 21.6772 22.8124 20.9923ZM19.2208 26.5697C16.7383 26.5697 14.7185 25.2332 14.7185 23.5905C14.7185 21.9478 16.7383 20.6113 19.2208 20.6113C21.7034 20.6113 23.7231 21.9478 23.7231 23.5905C23.7231 25.2332 21.7034 26.5697 19.2208 26.5697Z" fill="white"/> <path d="M17.5378 22.5767C16.9788 22.5767 16.524 23.0315 16.524 23.5905C16.524 24.1495 16.9788 24.6043 17.5378 24.6043C18.0968 24.6043 18.5516 24.1495 18.5516 23.5905C18.5516 23.0315 18.0968 22.5767 17.5378 22.5767Z" fill="white"/> <path d="M21.1018 22.5767C20.5428 22.5767 20.088 23.0315 20.088 23.5905C20.088 24.1495 20.5428 24.6043 21.1018 24.6043C21.6608 24.6043 22.1156 24.1495 22.1156 23.5905C22.1156 23.0315 21.6608 22.5767 21.1018 22.5767Z" fill="white"/> </svg> ',

			'beef'              => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>beef</defs> <rect width="38" height="38" rx="19" fill="#E49B4C"/> <path d="M15.5636 17.4046L14.1071 15.948C13.685 15.5256 13.6196 14.8852 13.8995 14.3883L16.24 16.7289L16.7292 16.2397L14.3883 13.8988C14.5821 13.7891 14.799 13.7251 15.0277 13.7251C15.3758 13.7251 15.7024 13.8607 15.9483 14.1067L17.4049 15.5633L17.8941 15.0741L16.4376 13.6175C15.7446 12.9235 14.6644 12.8532 13.887 13.3975L12.5557 12.0661C12.7117 11.7129 12.6477 11.2842 12.3592 10.995C12.3588 10.9946 12.3585 10.9946 12.3581 10.9943L10.4646 9.10146C10.3293 8.96618 10.1107 8.96618 9.9754 9.10146C9.84012 9.23673 9.84012 9.45539 9.9754 9.59067L11.8686 11.4835C11.8689 11.4839 11.8689 11.4842 11.8693 11.4845C11.9755 11.5908 11.9755 11.7634 11.87 11.8689C11.7669 11.9717 11.5877 11.972 11.4852 11.8696C11.4852 11.8693 11.4849 11.8693 11.4849 11.8693L9.59067 9.97505C9.45539 9.83977 9.23673 9.83977 9.10146 9.97505C8.96618 10.1103 8.96618 10.329 9.10146 10.4643L10.9957 12.3585C10.9957 12.3585 10.996 12.3585 10.996 12.3588C11.1777 12.5408 11.4195 12.6408 11.6769 12.6408C11.8139 12.6408 11.9457 12.6104 12.0679 12.5564L13.3982 13.8867C12.8546 14.6641 12.9249 15.7435 13.6182 16.4372L15.0748 17.8938L15.5636 17.4046Z" fill="white"/> <path d="M24.6542 11.6669C24.395 11.6299 24.1266 11.6108 23.855 11.6108C22.2354 11.6108 20.5865 12.2841 19.4438 13.4116C18.3148 14.5243 17.7689 16.0155 17.9426 17.5066C17.9913 18.6459 17.8931 19.1009 17.8021 19.2825C17.7671 19.3517 17.7582 19.3704 17.5063 19.4406C15.0384 19.9087 13.7113 22.5517 13.8964 24.7448C13.9898 25.8426 14.4503 26.8601 15.1934 27.6105C16.0937 28.5194 17.3727 29 18.8919 29C24.9649 29 29.9148 23.9525 29.9269 17.7485V17.7239C29.9168 14.5008 27.8472 12.1243 24.6542 11.6669ZM29.2349 17.745V17.7467C29.2238 23.5702 24.5839 28.3077 18.8919 28.3077C15.94 28.3077 14.7274 26.3567 14.5859 24.6863C14.4247 22.7779 15.5543 20.4973 17.662 20.1149C18.2878 19.943 18.7359 19.7821 18.6331 17.4502C18.4754 16.1639 18.9469 14.873 19.9299 13.904C20.9463 12.9013 22.4136 12.3024 23.8553 12.3024C24.0923 12.3024 24.3276 12.3187 24.5559 12.3516C27.3891 12.7577 29.2259 14.8665 29.2349 17.7232V17.745Z" fill="white"/> <path d="M24.4576 13.0366C23.0464 12.8352 21.4584 13.3681 20.4156 14.397C19.5686 15.2322 19.1815 16.3061 19.323 17.3914C19.4479 20.1371 18.7937 20.5229 17.8138 20.791C16.0708 21.0878 15.1394 23.0142 15.2758 24.6282C15.3982 26.0727 16.4123 27.6161 18.8923 27.6161C24.2072 27.6161 28.5326 23.19 28.5433 17.7457C28.5433 17.7429 28.5433 17.7398 28.5433 17.7374C28.5433 17.7336 28.5433 17.7301 28.5433 17.7263C28.535 15.1927 26.9712 13.3971 24.4576 13.0366ZM23.8633 13.687C24.0311 13.687 24.1975 13.6985 24.3597 13.722C25.1465 13.8348 25.8232 14.1136 26.3671 14.5264L24.9794 15.9214C24.6704 15.7044 24.295 15.5761 23.8899 15.5761C23.5802 15.5761 23.2886 15.6522 23.0301 15.784L21.5964 14.3503C22.2759 13.9296 23.0897 13.687 23.8633 13.687ZM22.1147 19.7825L22.8329 19.0604L22.8319 19.0597C23.1346 19.2628 23.4986 19.3818 23.8896 19.3818C24.3777 19.3818 24.822 19.1954 25.1593 18.8919L25.9952 19.7278L24.0511 21.6719L22.1147 19.7825ZM23.5619 22.1611L21.2193 24.5037L19.3261 22.5866L21.6269 20.2727L23.5619 22.1611ZM23.8896 16.268C24.5573 16.268 25.1005 16.8109 25.1005 17.4786C25.1005 18.1467 24.5573 18.6899 23.8896 18.6899C23.2218 18.6899 22.6786 18.1467 22.6786 17.4786C22.6786 16.8109 23.2222 16.268 23.8896 16.268ZM20.9013 14.8893C20.9439 14.8471 20.992 14.8097 21.037 14.7693L22.4762 16.2085C22.4769 16.2078 22.4776 16.2075 22.478 16.2068C22.1739 16.5441 21.9867 16.9894 21.9867 17.4783C21.9867 17.8876 22.1178 18.2657 22.3385 18.5767L21.6199 19.2995L20.0253 17.7436C20.0226 17.6097 20.0184 17.4741 20.0122 17.3312C19.8634 16.1847 20.4135 15.3702 20.9013 14.8893ZM20.0066 18.6923L21.1321 19.7904L18.8404 22.0946L18.1595 21.4055C19.3534 21.0228 19.8963 20.2717 20.0066 18.6923ZM16.02 24.9296C15.9961 24.8133 15.9757 24.6943 15.9653 24.5698C15.8701 23.4433 16.4182 22.1403 17.4236 21.6445L18.3522 22.5852L16.02 24.9296ZM18.802 26.921C17.6111 26.8989 16.7309 26.4398 16.2843 25.6451L18.8383 23.0772L20.7301 24.9929L18.802 26.921ZM19.8278 26.8736L21.2162 25.4852L22.0517 26.3315C21.348 26.6048 20.6024 26.7909 19.8278 26.8736ZM22.7312 26.035L21.7054 24.996L24.0574 22.6441L25.4409 23.9937C24.6746 24.8372 23.7564 25.5337 22.7312 26.035ZM25.8876 23.463L24.5466 22.1552L26.4847 20.217L27.2642 20.9965C26.93 21.8947 26.4629 22.7247 25.8876 23.463ZM27.5036 20.2575L26.9739 19.7278L27.7725 18.9293C27.714 19.3836 27.6237 19.8268 27.5036 20.2575ZM27.8475 17.8754L26.4847 19.2383L25.5848 18.3384C25.7167 18.0796 25.7928 17.7879 25.7928 17.4783C25.7928 17.0839 25.672 16.7175 25.4658 16.4133L26.8764 14.9952C27.5009 15.6934 27.8496 16.6327 27.8513 17.7371C27.851 17.7838 27.8486 17.8294 27.8475 17.8754Z" fill="white"/> </svg> ',

			'meat'              => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>meat</defs> <rect width="38" height="38" rx="19" fill="#F15701"/> <path d="M23.2514 15.615C23.8373 16.2896 24.6548 16.72 25.5425 16.8214C25.6911 16.827 25.8393 16.8027 25.9783 16.7498C26.1173 16.6969 26.2442 16.6166 26.3514 16.5135C26.9893 15.8761 26.6111 14.5722 25.4529 13.414C24.2947 12.2558 22.9913 11.878 22.3534 12.5155C22.156 12.7124 21.8727 13.1721 22.1885 14.0359C22.4273 14.6314 22.7896 15.1696 23.2514 15.615ZM22.9984 13.1605C23.0434 13.1278 23.099 13.1129 23.1543 13.1186C23.7985 13.2291 24.3836 13.5619 24.8079 14.059C25.6872 14.9384 25.8387 15.7362 25.7064 15.8685C25.5745 15.9995 24.7767 15.8498 23.8965 14.97C23.5295 14.6174 23.2396 14.1925 23.0452 13.7222C22.9267 13.3975 22.9534 13.2055 22.9984 13.1605Z" fill="white"/> <path d="M18.6328 14.1575C18.5303 14.0933 18.4065 14.0724 18.2886 14.0993C18.1708 14.1262 18.0683 14.1987 18.0038 14.3009C17.798 14.6288 17.6028 14.9642 17.4238 15.2979C17.3953 15.3506 17.3776 15.4085 17.3715 15.4681C17.3655 15.5278 17.3712 15.588 17.3885 15.6454C17.4058 15.7029 17.4342 15.7563 17.4721 15.8027C17.5101 15.8491 17.5568 15.8876 17.6096 15.916C17.6624 15.9443 17.7203 15.962 17.78 15.9679C17.8396 15.9739 17.8999 15.968 17.9573 15.9506C18.0146 15.9333 18.068 15.9048 18.1144 15.8667C18.1607 15.8287 18.1991 15.7819 18.2274 15.7291C18.3967 15.4137 18.5815 15.0965 18.7762 14.7865C18.8405 14.684 18.8615 14.5602 18.8346 14.4423C18.8077 14.3243 18.7351 14.2219 18.6328 14.1575Z" fill="white"/> <path d="M17.4078 16.3353C17.2961 16.2887 17.1705 16.2883 17.0586 16.3342C16.9466 16.3802 16.8575 16.4687 16.8108 16.5803C16.1529 18.1524 15.8949 19.6189 16.0847 20.7098C16.1032 20.8155 16.1584 20.9113 16.2405 20.9804C16.3226 21.0495 16.4264 21.0874 16.5337 21.0876C16.5602 21.0875 16.5865 21.0853 16.6126 21.0809C16.6716 21.0706 16.728 21.0488 16.7786 21.0167C16.8292 20.9846 16.873 20.9428 16.9075 20.8938C16.9419 20.8448 16.9664 20.7895 16.9795 20.731C16.9926 20.6725 16.994 20.612 16.9837 20.553C16.8233 19.6318 17.0607 18.3462 17.6528 16.9323C17.6993 16.8206 17.6997 16.695 17.6538 16.5831C17.6078 16.4712 17.5193 16.3821 17.4078 16.3353Z" fill="white"/> <path d="M27.7533 13.1119C27.4548 12.7464 27.1378 12.3964 26.8036 12.0633C25.6665 10.8811 24.2927 9.95209 22.7721 9.33706C21.3371 8.81199 20.1629 8.90292 19.4592 9.59205C19.4537 9.59701 19.4466 9.59865 19.4413 9.60391L19.4409 9.60437L19.4404 9.60482C15.517 13.533 13.5001 18.3931 14.4259 21.5758L12.3553 23.6464C11.9604 23.4586 11.5158 23.4022 11.0866 23.4856C10.525 23.5724 10.003 23.8278 9.5898 24.2179C9.40296 24.4042 9.2547 24.6255 9.15349 24.8691C9.05228 25.1128 9.00012 25.374 9 25.6378C8.99988 25.9016 9.05179 26.1629 9.15277 26.4066C9.25374 26.6504 9.4018 26.8718 9.58847 27.0582C9.79312 27.2605 10.0386 27.4168 10.3084 27.5167C10.5783 27.6165 10.8664 27.6577 11.1534 27.6373C11.1237 28.0458 11.2196 28.4536 11.4283 28.806C11.637 29.1584 11.9485 29.4385 12.321 29.6089C12.6935 29.7792 13.1091 29.8315 13.5122 29.7588C13.9152 29.6861 14.2864 29.4919 14.576 29.2022C15.2611 28.5175 15.5823 27.2774 15.1484 26.4354L17.1699 24.4099C17.7305 24.5827 18.3143 24.6684 18.901 24.6638C21.964 24.6638 25.9745 22.6992 29.2419 19.4394C29.248 19.4335 29.2557 19.4298 29.2616 19.4238C30.4493 18.2357 29.8292 15.64 27.7533 13.1119ZM22.4585 10.1941C23.8548 10.7631 25.1156 11.6199 26.1585 12.7088C26.4713 13.0202 26.7677 13.3474 27.0468 13.6893L27.0481 13.6911C28.7547 15.7696 29.4293 17.9509 28.6217 18.7724C28.6196 18.7744 28.6169 18.7749 28.6148 18.7769L28.6113 18.7801C28.1827 19.2068 27.3221 19.2269 26.2481 18.8349C26.1028 18.7818 25.9559 18.7214 25.808 18.6555C25.7684 18.6377 25.7279 18.6171 25.688 18.5983C25.5986 18.5567 25.5089 18.5129 25.4189 18.467C25.3309 18.4217 25.2422 18.3737 25.153 18.3239C25.1147 18.3026 25.0764 18.281 25.0381 18.259C24.9294 18.1963 24.8201 18.1316 24.7102 18.0625L24.6862 18.0471C24.2831 17.7914 23.8961 17.5113 23.5272 17.2085C23.185 16.9296 22.8574 16.6333 22.5458 16.3206C21.9683 15.7473 21.4476 15.1194 20.9911 14.4457C20.0932 13.1059 19.693 11.8621 19.773 11.0269C19.7849 10.7393 19.8963 10.4648 20.0881 10.2503C20.2243 10.1307 20.3828 10.0393 20.5544 9.9813C20.726 9.92327 20.9075 9.89977 21.0882 9.91215C21.5578 9.92664 22.0213 10.022 22.4585 10.1941ZM14.2316 26.0626C14.1462 26.1481 14.0983 26.2641 14.0983 26.385C14.0984 26.5058 14.1465 26.6217 14.232 26.7072C14.6236 27.0987 14.3906 28.0975 13.9309 28.5572C13.7223 28.757 13.4444 28.8685 13.1555 28.8682C12.8666 28.868 12.589 28.7561 12.3807 28.5558C12.2033 28.3785 12.0922 28.1457 12.0659 27.8963C12.0396 27.6468 12.0998 27.396 12.2364 27.1856C12.2935 27.0979 12.3187 26.9933 12.3079 26.8892C12.297 26.7851 12.2507 26.688 12.1767 26.614C12.1028 26.54 12.0056 26.4937 11.9015 26.4828C11.7974 26.472 11.6928 26.4972 11.6051 26.5544C11.4398 26.6625 11.2482 26.7238 11.0509 26.7319C10.8535 26.7399 10.6576 26.6944 10.484 26.6C10.3104 26.5057 10.1656 26.3661 10.0649 26.1961C9.9643 26.0261 9.91157 25.832 9.91235 25.6345C9.91202 25.4912 9.94025 25.3493 9.9954 25.217C10.0505 25.0848 10.1315 24.9649 10.2335 24.8643C10.5107 24.6094 10.8577 24.4431 11.23 24.3867C11.3763 24.3485 11.5294 24.3446 11.6774 24.3752C11.8254 24.4058 11.9644 24.4701 12.0836 24.5631C12.1259 24.6055 12.1762 24.6391 12.2315 24.662C12.2869 24.685 12.3462 24.6968 12.4061 24.6968C12.466 24.6968 12.5253 24.685 12.5806 24.662C12.636 24.6391 12.6862 24.6055 12.7286 24.5631L14.7933 22.4984C15.1382 23.1304 15.6489 23.6564 16.2705 24.0197L14.2316 26.0626ZM17.2122 23.4649L17.2109 23.4642C16.7819 23.3017 16.3942 23.0464 16.0755 22.7166C15.7568 22.3867 15.5149 21.9905 15.3673 21.5562L15.3671 21.5549C14.477 19.0727 15.9275 15.0649 18.864 11.5804C18.8662 11.6051 18.8736 11.632 18.8763 11.6569C18.8926 11.8061 18.9172 11.9591 18.9502 12.1157C18.9571 12.1485 18.9617 12.1801 18.9693 12.2132C19.0165 12.4173 19.0742 12.6189 19.1422 12.8171C19.1474 12.8322 19.1542 12.8479 19.1595 12.8631C19.2245 13.0486 19.3006 13.2373 19.3852 13.4284C19.4071 13.4781 19.4304 13.5278 19.4537 13.5778C19.5323 13.7465 19.6182 13.9166 19.7114 14.0881C19.7362 14.1338 19.7591 14.1789 19.785 14.2247C19.9055 14.4385 20.0347 14.6535 20.1771 14.8694C20.1916 14.8914 20.2083 14.9134 20.223 14.9354C20.3477 15.1217 20.4804 15.3081 20.6212 15.4945C20.6765 15.5682 20.7346 15.6413 20.7924 15.7148C20.8998 15.8512 21.0112 15.9871 21.1268 16.1225C21.1973 16.2054 21.267 16.2881 21.3406 16.3706C21.4646 16.5094 21.5949 16.6466 21.7273 16.7836C21.7864 16.8449 21.8396 16.9075 21.9003 16.9683C22.411 17.4784 22.9606 17.948 23.544 18.3729C23.5733 18.3942 23.6024 18.4145 23.6316 18.4354C23.8018 18.557 23.9724 18.6723 24.1434 18.7816C24.1779 18.8036 24.2124 18.826 24.2469 18.8476C24.4306 18.9622 24.6142 19.0695 24.7975 19.1694C24.8077 19.1749 24.8181 19.1812 24.8283 19.1867C25.1847 19.3811 25.5545 19.5498 25.9349 19.6915C26.315 19.832 26.7095 19.9301 27.1111 19.9841L27.1206 19.9857C27.1701 19.9921 27.2237 20.0067 27.2722 20.0113C23.7325 22.9992 19.6886 24.4322 17.2122 23.4649Z" fill="white"/> </svg> ',

			'poultry'           => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>poultry</defs> <rect width="38" height="38" rx="19" fill="#FF7B5F"/> <path d="M28.0304 17.4497C27.8884 16.5632 26.9407 16.3829 26.4801 16.4096H19.9453C18.3143 16.4096 18.0973 15.4589 18.0794 15.3561C18.0748 15.3253 18.0665 15.2953 18.0553 15.2657C18.0186 15.172 17.277 13.3108 15.9608 12.0383C16.1611 11.1093 16.0878 10.3515 15.736 9.7885C15.3508 9.17266 14.7841 9.03192 14.7208 9.01776C14.7062 9.01443 14.6917 9.01193 14.6771 9.01026C14.1212 8.95114 13.729 9.15642 13.4879 9.39667C13.3142 9.34296 13.0815 9.30299 12.7717 9.30007C12.1733 9.30798 11.8369 9.70314 11.6928 10.0862C11.4634 10.1045 11.2968 10.1837 11.1856 10.2615C10.86 10.4893 10.793 10.8332 10.7905 10.8628C10.6547 11.4145 10.8067 11.7564 10.9579 11.9454C11.0228 12.027 11.0974 12.0887 11.1736 12.1403L10.1184 13.2208C10.0014 13.3403 9.96768 13.519 10.0326 13.6734C10.0976 13.8275 10.2487 13.9279 10.4161 13.9279H11.1744C11.2356 14.398 11.4409 15.2324 12.1138 15.6301C11.3601 19.7632 13.8393 23.6685 13.9742 23.8759C15.1314 25.4615 16.5887 26.2922 17.9691 26.7128L17.5943 27.7825C16.7765 27.8299 16.1399 28.2014 16.0974 28.2268C15.9004 28.345 15.8363 28.6007 15.9546 28.798C16.0328 28.928 16.1707 29 16.3118 29C16.3847 29 16.4588 28.9808 16.5259 28.9405C16.5379 28.933 17.7659 28.2126 18.7685 28.923C18.9568 29.0562 19.2162 29.0121 19.3494 28.8243C19.4822 28.6369 19.4381 28.3771 19.2507 28.2438C18.9863 28.0565 18.7152 27.937 18.4483 27.8649L18.7802 26.918C19.481 27.0596 20.1397 27.1071 20.701 27.1071C20.9546 27.1071 21.1865 27.0971 21.3956 27.0829L21.3344 27.8216C20.6552 27.9145 20.2018 28.2172 20.1689 28.2397C19.9802 28.3692 19.9324 28.6261 20.0602 28.8155C20.1889 29.0058 20.4466 29.0558 20.6373 28.9284C20.679 28.9013 21.665 28.2613 22.8238 28.9425C22.89 28.9817 22.9629 29 23.0345 29C23.1773 29 23.3164 28.9263 23.3938 28.7947C23.51 28.5961 23.4438 28.3413 23.246 28.2247C22.8696 28.0036 22.5065 27.8866 22.1688 27.8333L22.2392 26.9872C22.3379 26.9705 22.4078 26.9568 22.4436 26.9493C27.2838 26.3759 27.3841 20.5968 27.3666 19.6349C27.579 19.2756 28.1549 18.2134 28.0324 17.4497H28.0304ZM11.4047 13.0951L11.6599 12.8336C11.7086 12.9243 11.7636 13.0122 11.8248 13.0951H11.4047ZM11.8235 11.2425C11.8002 11.2621 11.7794 11.2842 11.7611 11.3079C11.7215 11.3595 11.6932 11.4141 11.662 11.4674C11.6403 11.4553 11.6208 11.4412 11.6074 11.4237C11.5604 11.3633 11.5575 11.2309 11.6045 11.036C11.6153 10.9865 11.6407 10.8711 11.931 10.9298C12.0467 10.954 12.1683 10.9269 12.2637 10.8553C12.359 10.7841 12.4194 10.6758 12.4294 10.5572C12.4306 10.5401 12.4689 10.132 12.7596 10.132H12.7629C13.1464 10.1358 13.298 10.2095 13.3275 10.2265C13.4187 10.3136 13.5445 10.3548 13.6715 10.3381C13.8156 10.3182 13.9388 10.2253 13.9967 10.0916C13.9979 10.0883 14.1266 9.79849 14.5563 9.83514C14.6175 9.85512 14.8545 9.94839 15.0302 10.229C15.2117 10.5197 15.2754 10.9319 15.2196 11.4382C15.1892 11.4183 15.1601 11.3954 15.1293 11.3762C13.156 10.1512 11.8773 11.1967 11.8235 11.2421V11.2425ZM12.3315 14.7277C12.1479 14.4971 12.0596 14.1731 12.018 13.9279H12.6209C12.5085 14.1939 12.4123 14.4608 12.3315 14.7277ZM26.5934 19.3018C26.5472 19.3755 26.5247 19.4617 26.5301 19.5479C26.5334 19.6087 26.8295 25.606 22.3262 26.1248C22.3208 26.1252 22.315 26.1269 22.3091 26.1277L22.3691 25.404C22.3882 25.1746 22.2179 24.9735 21.9889 24.9548C21.7537 24.931 21.5584 25.1059 21.5396 25.3349L21.4634 26.2551C20.793 26.3189 19.7791 26.3338 18.6719 26.069C18.6611 26.064 18.6536 26.0553 18.642 26.0511C18.6062 26.0382 18.5699 26.0386 18.5341 26.0357C17.2167 25.6947 15.7855 24.9468 14.66 23.4045C14.6271 23.3537 11.4092 18.2879 13.5457 13.9016C13.6419 13.7043 13.5678 13.4665 13.3775 13.3574C13.1318 13.2171 12.4086 12.7087 12.3116 12.229C12.2849 12.097 12.3099 11.98 12.3919 11.8567C12.571 11.736 13.3963 11.2813 14.6904 12.0841C16.1819 13.0097 17.1367 15.2224 17.2654 15.5326C17.3874 16.1543 18.0815 17.2419 19.9461 17.2419L26.5055 17.2411C26.5126 17.2428 27.1505 17.2144 27.2088 17.5809C27.2779 18.0118 26.8682 18.8654 26.5934 19.3018Z" fill="white"/> <path d="M13.3059 11.7447C12.9686 11.7447 12.7038 12.0075 12.7021 12.3464C12.7034 12.682 12.9686 12.9448 13.3059 12.9448C13.6432 12.9448 13.908 12.682 13.9097 12.3464V12.3431C13.9084 12.007 13.6432 11.7447 13.3059 11.7447Z" fill="white"/> <path d="M24.1209 21.2368C24.113 21.2405 23.5667 21.4999 23.0274 21.4362C23.5288 21.0998 24.1342 20.531 24.4028 19.6433C24.4694 19.423 24.3449 19.1906 24.125 19.124C23.9043 19.057 23.6724 19.1823 23.6058 19.4017C23.2439 20.598 21.991 21.0477 21.9802 21.0519C21.844 21.0981 21.7416 21.2118 21.7091 21.3517C21.6766 21.492 21.7183 21.6386 21.8203 21.7406C22.34 22.2603 22.997 22.3252 23.5317 22.2582C22.5482 23.6897 20.5957 23.5761 20.3542 23.5536C18.9355 23.2346 17.9029 22.5713 17.2833 21.5811C16.2136 19.8719 16.7112 17.7191 16.7162 17.6975C16.7695 17.4738 16.6321 17.2494 16.4089 17.1953C16.1857 17.1424 15.9608 17.2781 15.9067 17.5017C15.8821 17.6029 15.3192 20.0055 16.5729 22.0163C17.3183 23.2121 18.5383 24.0045 20.1989 24.3726C20.2139 24.376 20.2293 24.3785 20.2451 24.3801C20.3742 24.3939 20.517 24.4018 20.6711 24.4018C21.9619 24.4018 24.0159 23.8646 24.6984 21.739C24.7492 21.5803 24.7005 21.4075 24.5747 21.298C24.4494 21.1885 24.2708 21.1652 24.1209 21.2368Z" fill="white"/> <path d="M20.091 20.8012C20.3213 20.8012 20.5074 20.6151 20.5074 20.3848C20.5074 20.1546 20.3213 19.9685 20.091 19.9685C19.2919 19.9685 18.7194 19.7607 18.3896 19.3509C17.8658 18.7005 18.0736 17.707 18.0773 17.6908C18.1273 17.4668 17.9866 17.2448 17.763 17.194C17.5381 17.1436 17.3158 17.2844 17.265 17.5084C17.2525 17.5646 16.9618 18.8983 17.7367 19.8677C18.2318 20.4873 19.0242 20.8017 20.0914 20.8017L20.091 20.8012Z" fill="white"/> </svg> ',

			'caffeine'          => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>caffeine</defs> <rect width="38" height="38" rx="19" fill="#8E4B3D"/> <path d="M28.795 12.0208C28.2493 11.7568 27.7037 11.7172 27.1757 11.9064C26.8897 12.0076 26.6389 12.1704 26.4233 12.3552C26.4365 12.0516 26.4409 11.748 26.4409 11.44C26.4409 11.198 26.2429 11 26.0009 11H10.4685C10.2265 11 10.0285 11.198 10.0285 11.44C10.0285 16.1921 11.4365 20.4999 13.751 23.0255H8.44001C8.19801 23.0255 8 23.2235 8 23.4655V25.56C8 25.802 8.19801 26 8.44001 26H28.0293C28.2713 26 28.4693 25.802 28.4693 25.56V23.4655C28.4693 23.2235 28.2713 23.0255 28.0293 23.0255H22.7184C23.1804 22.5239 23.6028 21.9519 23.99 21.3139C27.5321 20.6935 29.5518 17.8026 29.9346 15.3297C30.1766 13.7501 29.741 12.4828 28.795 12.0208ZM27.5893 25.1156H8.88002V23.9011H14.829H21.6404H27.5893V25.1156ZM14.9918 23.0211C12.5805 20.8959 11.0009 16.5793 10.9085 11.8756H25.5565C25.4641 16.5793 23.8844 20.8959 21.4732 23.0211H14.9918ZM29.0634 15.1933C28.7554 17.191 27.2285 19.4922 24.562 20.2754C25.4553 18.4626 26.0581 16.2713 26.3089 13.8997C26.4145 13.7017 26.8369 12.9581 27.4705 12.7336C27.7697 12.628 28.0777 12.65 28.4121 12.8128C28.9886 13.0945 29.2438 14.0273 29.0634 15.1933Z" fill="white"/> <path d="M18.2347 14.3397C16.3426 14.3397 14.7982 15.5101 14.7674 16.9622C14.7674 16.9798 14.763 16.993 14.763 17.0106C14.763 18.4846 16.3206 19.6814 18.2347 19.6814C20.1267 19.6814 21.6712 18.511 21.702 17.059C21.702 17.0414 21.7064 17.0282 21.7064 17.0106C21.7064 15.5409 20.1487 14.3397 18.2347 14.3397ZM18.2347 15.2197C19.5547 15.2197 20.646 15.9061 20.8044 16.7862C20.4304 16.9974 19.5415 17.3406 18.4767 16.641C17.5395 16.0249 16.6331 16.0029 15.9378 16.1833C16.3734 15.6113 17.2359 15.2197 18.2347 15.2197ZM18.2347 18.8014C16.9147 18.8014 15.8278 18.115 15.665 17.235C16.039 17.0238 16.9279 16.6806 17.9927 17.3802C18.6175 17.7894 19.2291 17.939 19.7703 17.939C20.0431 17.939 20.2983 17.8994 20.5272 17.8378C20.0959 18.4098 19.2291 18.8014 18.2347 18.8014Z" fill="white"/> </svg> ',

			'bread'             => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>bread</defs> <rect width="38" height="38" rx="19" fill="#F59C7B"/> <path d="M26.4461 11.0048C25.0244 10.5018 23.2527 10.1518 21.6122 10.0643C21.5685 10.0643 21.5466 10.0643 21.5028 10.0862C21.2404 10.0643 20.9779 10.0424 20.7154 10.0206C17.1939 9.84558 14.1536 10.808 14.088 12.1204C14.0661 12.4484 14.2192 12.7547 14.5473 13.039C11.2664 13.7389 8.79474 15.27 9.01347 16.5168C9.10096 16.9324 9.45092 17.2605 10.1071 17.4792C10.1727 17.5011 10.1946 17.5448 10.2165 17.5886L12.01 27.3219C12.0538 27.5407 12.1631 27.7157 12.3381 27.8469C12.4694 27.9344 12.6225 28 12.7975 28C12.8412 28 12.9068 28 12.9506 27.9781L22.0497 26.2939L23.9307 25.944C24.1494 25.9002 24.3244 25.7908 24.4557 25.6159C24.5869 25.4409 24.6306 25.2222 24.5869 25.0034L24.5213 24.6535L26.4023 24.741C26.4242 24.741 26.4242 24.741 26.4461 24.741C26.6429 24.741 26.8398 24.6753 26.9929 24.5222C27.146 24.3691 27.2554 24.1723 27.2554 23.9535L27.7585 14.067C27.7585 14.0233 27.8022 13.9577 27.8459 13.9358C28.4584 13.6296 28.7865 13.2796 28.8083 12.8422C28.8302 12.0329 27.5397 11.3985 26.4461 11.0048ZM20.6717 10.6767C24.2369 10.8517 26.468 12.1422 26.4461 12.7328C26.4461 12.9078 26.2274 13.1046 25.8555 13.2796C25.593 13.4108 25.4181 13.6733 25.3962 13.9577L24.8931 23.8442C24.8931 23.8879 24.8712 23.9317 24.8494 23.9535C24.8275 23.9754 24.7838 23.9973 24.74 23.9973L24.4119 23.9754L22.7933 15.2482C22.7933 15.2044 22.8152 15.1388 22.8589 15.0951C23.3839 14.6576 23.6026 14.2201 23.537 13.8046C23.2745 12.3391 18.8344 12.2078 15.9034 12.7547C15.8597 12.7547 15.8378 12.7765 15.794 12.7984C15.6628 12.8203 15.5316 12.8422 15.4003 12.864C15.3566 12.8203 15.3128 12.7765 15.2472 12.7547C14.8972 12.5359 14.7004 12.3172 14.7223 12.1641C14.7441 11.5517 17.0845 10.5018 20.6717 10.6767ZM12.8412 27.3219C12.7975 27.3219 12.7537 27.3219 12.7318 27.3001C12.71 27.2782 12.6881 27.2563 12.6662 27.2126L10.8727 17.4792C10.807 17.173 10.5883 16.9542 10.304 16.8449C9.91025 16.7136 9.69152 16.5605 9.64778 16.3855C9.53841 15.795 11.5726 14.2201 15.0941 13.5858C16.0784 13.4108 16.9752 13.3234 17.7626 13.3234C19.7968 13.3234 21.1091 13.8264 21.1747 14.242C21.1966 14.3951 21.0435 14.6576 20.7373 14.8982C20.4967 15.0951 20.3873 15.3794 20.4529 15.6637L22.2465 25.3971C22.2465 25.4409 22.2465 25.4846 22.2246 25.5065C22.2028 25.5284 22.1809 25.5502 22.1371 25.5721L12.8412 27.3219ZM23.8213 25.2878L22.9246 25.4628C22.9246 25.419 22.9246 25.3534 22.9027 25.3096L21.1091 15.5763C21.1091 15.5325 21.131 15.4669 21.1747 15.4231C21.6997 14.9857 21.9184 14.5482 21.8528 14.1326C21.7872 13.7827 21.5028 13.4765 21.0435 13.2359C22.1371 13.389 22.8371 13.6514 22.8808 13.9358C22.9027 14.0889 22.7496 14.3514 22.4434 14.592C22.2028 14.7888 22.0934 15.0732 22.159 15.3575L23.9526 25.0909C23.9745 25.2003 23.9088 25.2878 23.8213 25.2878ZM27.5616 13.3671C27.2991 13.4983 27.1241 13.7608 27.1023 14.0452L26.5992 23.9317C26.5992 23.9754 26.5773 24.0192 26.5554 24.041C26.5336 24.0629 26.5117 24.0848 26.4461 24.0848L25.5274 24.041C25.5493 23.9973 25.5493 23.9317 25.5493 23.8879L26.0524 14.0014C26.0524 13.9577 26.0961 13.892 26.1399 13.8702C26.7523 13.564 27.0804 13.214 27.1023 12.7765C27.1241 12.4266 26.9054 12.0547 26.5336 11.7266C27.7147 12.186 28.174 12.6453 28.1522 12.8203C28.1303 12.9953 27.9116 13.1921 27.5616 13.3671Z" fill="white"/> <path d="M19.4687 22.4662C19.6499 22.4662 19.7968 22.3193 19.7968 22.1381C19.7968 21.9569 19.6499 21.81 19.4687 21.81C19.2875 21.81 19.1406 21.9569 19.1406 22.1381C19.1406 22.3193 19.2875 22.4662 19.4687 22.4662Z" fill="white"/> <path d="M20.4311 24.4129C20.6123 24.4129 20.7592 24.266 20.7592 24.0848C20.7592 23.9036 20.6123 23.7567 20.4311 23.7567C20.2499 23.7567 20.103 23.9036 20.103 24.0848C20.103 24.266 20.2499 24.4129 20.4311 24.4129Z" fill="white"/> <path d="M18.55 24.741C18.7312 24.741 18.8781 24.5941 18.8781 24.4129C18.8781 24.2317 18.7312 24.0848 18.55 24.0848C18.3688 24.0848 18.2219 24.2317 18.2219 24.4129C18.2219 24.5941 18.3688 24.741 18.55 24.741Z" fill="white"/> <path d="M14.7609 17.8196C14.7673 17.6385 14.6257 17.4865 14.4446 17.4801C14.2635 17.4737 14.1115 17.6153 14.1051 17.7964C14.0988 17.9775 14.2404 18.1295 14.4215 18.1359C14.6026 18.1423 14.7545 18.0006 14.7609 17.8196Z" fill="white"/> <path d="M12.7692 18.7029C12.7756 18.5218 12.634 18.3698 12.4529 18.3634C12.2718 18.357 12.1198 18.4986 12.1134 18.6797C12.107 18.8608 12.2486 19.0128 12.4297 19.0192C12.6108 19.0256 12.7628 18.884 12.7692 18.7029Z" fill="white"/> <path d="M12.5042 16.8048C12.5106 16.6237 12.3689 16.4717 12.1878 16.4653C12.0068 16.4589 11.8548 16.6005 11.8484 16.7816C11.842 16.9627 11.9836 17.1147 12.1647 17.1211C12.3458 17.1275 12.4978 16.9859 12.5042 16.8048Z" fill="white"/> </svg> ',

			'soup'              => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>soup</defs> <rect width="38" height="38" rx="19" fill="#8EBB5C"/> <path d="M10.1073 19.8008H27.8232M23.7485 18.4056L27.295 14.4286L26.4336 13.5194L21.3539 18.4056M15.1045 24.848C15.4182 25.0445 15.7576 25.2129 16.1217 25.3369M12.3636 21.6956C12.3636 21.6956 12.9843 23.0172 14.1605 24.1157M14.0287 10C14.0287 10 12.0292 12.5946 14.0287 14.1212C16.0282 15.6477 15.1418 16.2477 14.3639 17.3265M16.0729 10C16.0729 10 14.0734 12.5946 16.0729 14.1212C18.0724 15.6477 17.186 16.2477 16.4082 17.3265M18.1171 10C18.1171 10 16.1176 12.5946 18.1171 14.1212C20.1167 15.6477 19.2302 16.2477 18.4524 17.3265M24.8286 28C24.8286 26.9475 22.3145 26.6053 19.2129 26.6053C16.1112 26.6053 13.5971 26.9475 13.5971 28H24.8286ZM23.2336 26.415C26.1139 24.9035 28.0787 21.8842 28.0787 18.4056H10C10 21.9805 12.093 24.9883 15.1047 26.4548L23.2336 26.415Z" stroke="white" stroke-width="0.8" stroke-miterlimit="10" stroke-linecap="round"/> </svg> ',

			'snacks'            => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>snacks</defs> <rect width="38" height="38" rx="19" fill="#FFAB48"/> <path d="M13.4619 19.6875C13.4032 19.6875 13.3458 19.7049 13.297 19.7375C13.2482 19.7701 13.2101 19.8165 13.1876 19.8707C13.1652 19.925 13.1593 19.9847 13.1707 20.0423C13.1822 20.0998 13.2105 20.1527 13.252 20.1943C13.2935 20.2358 13.3464 20.264 13.404 20.2755C13.4616 20.287 13.5213 20.2811 13.5755 20.2586C13.6298 20.2361 13.6761 20.1981 13.7088 20.1493C13.7414 20.1004 13.7588 20.0431 13.7588 19.9843C13.7588 19.9056 13.7275 19.8301 13.6718 19.7744C13.6162 19.7187 13.5407 19.6875 13.4619 19.6875ZM15.2432 20.2812C15.1844 20.2812 15.127 20.2986 15.0782 20.3312C15.0294 20.3639 14.9914 20.4102 14.9689 20.4645C14.9464 20.5187 14.9405 20.5784 14.952 20.636C14.9634 20.6936 14.9917 20.7465 15.0332 20.788C15.0748 20.8295 15.1277 20.8578 15.1852 20.8693C15.2428 20.8807 15.3025 20.8748 15.3568 20.8524C15.411 20.8299 15.4574 20.7918 15.49 20.743C15.5226 20.6942 15.54 20.6368 15.54 20.5781C15.54 20.4993 15.5088 20.4238 15.4531 20.3682C15.3974 20.3125 15.3219 20.2812 15.2432 20.2812ZM11.6807 20.2812C11.622 20.2812 11.5646 20.2986 11.5157 20.3312C11.4669 20.3639 11.4289 20.4102 11.4064 20.4645C11.3839 20.5187 11.3781 20.5784 11.3895 20.636C11.401 20.6936 11.4292 20.7465 11.4708 20.788C11.5123 20.8295 11.5652 20.8578 11.6228 20.8693C11.6803 20.8807 11.74 20.8748 11.7943 20.8524C11.8485 20.8299 11.8949 20.7918 11.9275 20.743C11.9601 20.6942 11.9775 20.6368 11.9775 20.5781C11.9775 20.4993 11.9463 20.4238 11.8906 20.3682C11.8349 20.3125 11.7594 20.2812 11.6807 20.2812ZM13.4619 20.875C13.4032 20.875 13.3458 20.8924 13.297 20.925C13.2482 20.9576 13.2101 21.004 13.1876 21.0582C13.1652 21.1125 13.1593 21.1722 13.1707 21.2297C13.1822 21.2873 13.2105 21.3402 13.252 21.3818C13.2935 21.4233 13.3464 21.4515 13.404 21.463C13.4616 21.4745 13.5213 21.4686 13.5755 21.4461C13.6298 21.4236 13.6761 21.3856 13.7088 21.3368C13.7414 21.2879 13.7588 21.2305 13.7588 21.1718C13.7588 21.0931 13.7275 21.0176 13.6718 20.9619C13.6162 20.9062 13.5407 20.875 13.4619 20.875ZM19.3994 20.875C19.3407 20.875 19.2833 20.8924 19.2345 20.925C19.1856 20.9576 19.1476 21.004 19.1251 21.0582C19.1027 21.1125 19.0968 21.1722 19.1082 21.2297C19.1197 21.2873 19.148 21.3402 19.1895 21.3818C19.231 21.4233 19.2839 21.4515 19.3415 21.463C19.3991 21.4745 19.4588 21.4686 19.513 21.4461C19.5673 21.4236 19.6136 21.3856 19.6462 21.3368C19.6789 21.2879 19.6963 21.2305 19.6963 21.1718C19.6963 21.0931 19.665 21.0176 19.6093 20.9619C19.5536 20.9062 19.4781 20.875 19.3994 20.875ZM26.2275 14.9375V13.75H23.2588V11.0069L24.85 9.42156L24.4463 9L22.665 10.7812V13.75H19.6963V14.9375H17.915V16.7187H18.5385L18.7285 18.5475C18.558 18.5164 18.3852 18.5005 18.2119 18.5H12.8682C12.0026 18.5015 11.1728 18.8461 10.5608 19.4582C9.94869 20.0703 9.60413 20.9 9.60256 21.7656V22.1159C9.42791 22.1755 9.27629 22.2883 9.16894 22.4385C9.06158 22.5886 9.00386 22.7685 9.00386 22.9531C9.00386 23.1376 9.06158 23.3176 9.16894 23.4677C9.27629 23.6178 9.42791 23.7306 9.60256 23.7903V24.7878C9.42718 24.8458 9.27456 24.9577 9.16639 25.1074C9.05822 25.2572 9 25.4372 9 25.622C9 25.8067 9.05822 25.9867 9.16639 26.1365C9.27456 26.2863 9.42718 26.3981 9.60256 26.4562C9.66041 26.8863 9.87333 27.2806 10.2013 27.5648C10.5293 27.8491 10.9498 28.0038 11.3838 27.9999H26.2275L27.415 16.7187H28.0087V14.9375H26.2275ZM20.29 14.3437H25.6337V14.9375H20.29V14.3437ZM10.1963 21.7656C10.1979 21.0574 10.4799 20.3787 10.9806 19.878C11.4813 19.3773 12.16 19.0953 12.8682 19.0937H18.2119C18.92 19.0953 19.5987 19.3773 20.0995 19.878C20.6002 20.3787 20.8822 21.0574 20.8838 21.7656V22.0625H10.1963V21.7656ZM9.89943 22.6562H21.1806C21.2594 22.6562 21.3349 22.6875 21.3906 22.7432C21.4462 22.7988 21.4775 22.8743 21.4775 22.9531C21.4775 23.0318 21.4462 23.1073 21.3906 23.163C21.3349 23.2187 21.2594 23.2499 21.1806 23.2499H9.89943C9.8207 23.2499 9.74519 23.2187 9.68951 23.163C9.63384 23.1073 9.60256 23.0318 9.60256 22.9531C9.60256 22.8743 9.63384 22.7988 9.68951 22.7432C9.74519 22.6875 9.8207 22.6562 9.89943 22.6562ZM10.1963 24.7343V23.8437H21.0975V24.7343H10.1963ZM13.8004 25.3281L13.165 26.1118L12.5297 25.3281H13.8004ZM14.5663 25.3281H21.1866C21.2653 25.3281 21.3408 25.3593 21.3965 25.415C21.4522 25.4707 21.4835 25.5462 21.4835 25.6249C21.4835 25.7037 21.4522 25.7792 21.3965 25.8349C21.3408 25.8905 21.2653 25.9218 21.1866 25.9218H14.0972L14.5663 25.3281ZM9.89349 25.9218C9.81476 25.9218 9.73925 25.8905 9.68357 25.8349C9.6279 25.7792 9.59662 25.7037 9.59662 25.6249C9.59662 25.5462 9.6279 25.4707 9.68357 25.415C9.73925 25.3593 9.81476 25.3281 9.89349 25.3281H11.7638L12.2329 25.9218H9.89349ZM19.6963 27.4062H11.3838C11.1117 27.4138 10.8453 27.3277 10.6291 27.1623C10.413 26.9969 10.2602 26.7623 10.1963 26.4977H12.7138L13.165 27.0499L13.6163 26.4977H20.8481C20.7856 26.7563 20.6381 26.9864 20.4293 27.1511C20.2204 27.3158 19.9623 27.4057 19.6963 27.4062ZM25.6812 27.4062H20.9966C21.1551 27.234 21.2782 27.0322 21.3588 26.8124H25.7287L25.6812 27.4062ZM25.8059 26.2187H21.816C21.9755 26.0618 22.0671 25.8486 22.0713 25.6249C22.0709 25.4833 22.0362 25.344 21.97 25.2188C21.9038 25.0936 21.8081 24.9865 21.6913 24.9065V23.6774C21.7548 23.6355 21.811 23.5834 21.8575 23.5231C22.2015 23.7414 22.602 23.8538 23.0093 23.8463C23.4166 23.8389 23.8128 23.7119 24.1485 23.4811C24.4842 23.2503 24.7447 22.9259 24.8975 22.5483C25.0504 22.1706 25.0889 21.7564 25.0082 21.3571C24.9275 20.9577 24.7312 20.5909 24.4438 20.3023C24.1563 20.0136 23.7904 19.8158 23.3914 19.7334C22.9924 19.651 22.578 19.6877 22.1997 19.839C21.8215 19.9902 21.496 20.2493 21.2638 20.584C21.0958 20.1549 20.8393 19.7659 20.5109 19.4426C20.1825 19.1193 19.7896 18.8689 19.3578 18.7078L19.1381 16.7187H26.7856L25.8059 26.2187ZM21.4775 22.1159V21.7656C21.4762 21.428 21.589 21.0999 21.7977 20.8346C22.0064 20.5693 22.2987 20.3823 22.6271 20.3041C22.9555 20.2259 23.3007 20.261 23.6065 20.4038C23.9124 20.5466 24.1611 20.7886 24.312 21.0905C24.463 21.3924 24.5074 21.7365 24.4381 22.0669C24.3688 22.3973 24.1898 22.6945 23.9302 22.9103C23.6706 23.1261 23.3457 23.2477 23.0082 23.2555C22.6707 23.2633 22.3405 23.1567 22.0713 22.9531C22.0699 22.7695 22.0122 22.5908 21.906 22.4411C21.7999 22.2914 21.6503 22.1779 21.4775 22.1159ZM27.415 16.125H18.5088V15.5312H27.415V16.125ZM17.6182 20.875C17.5594 20.875 17.502 20.8924 17.4532 20.925C17.4044 20.9576 17.3663 21.004 17.3439 21.0582C17.3214 21.1125 17.3155 21.1722 17.327 21.2297C17.3384 21.2873 17.3667 21.3402 17.4082 21.3818C17.4498 21.4233 17.5026 21.4515 17.5602 21.463C17.6178 21.4745 17.6775 21.4686 17.7318 21.4461C17.786 21.4236 17.8324 21.3856 17.865 21.3368C17.8976 21.2879 17.915 21.2305 17.915 21.1718C17.915 21.0931 17.8837 21.0176 17.8281 20.9619C17.7724 20.9062 17.6969 20.875 17.6182 20.875ZM18.2119 19.6875C18.1532 19.6875 18.0958 19.7049 18.047 19.7375C17.9981 19.7701 17.9601 19.8165 17.9376 19.8707C17.9152 19.925 17.9093 19.9847 17.9207 20.0423C17.9322 20.0998 17.9605 20.1527 18.002 20.1943C18.0435 20.2358 18.0964 20.264 18.154 20.2755C18.2116 20.287 18.2713 20.2811 18.3255 20.2586C18.3798 20.2361 18.4261 20.1981 18.4587 20.1493C18.4914 20.1004 18.5088 20.0431 18.5088 19.9843C18.5088 19.9056 18.4775 19.8301 18.4218 19.7744C18.3661 19.7187 18.2906 19.6875 18.2119 19.6875Z" fill="white"/> </svg> ',

			'chocolate'         => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>chocolate</defs> <rect width="38" height="38" rx="19" fill="#C27363"/> <path d="M28 13.8324C27.9958 13.7534 27.9692 13.6773 27.9232 13.6129C27.8773 13.5486 27.8139 13.4987 27.7405 13.4691L26.3653 12.6907V11.1598C26.3653 11.0221 26.3106 10.8901 26.2133 10.7928C26.116 10.6955 25.984 10.6408 25.8463 10.6408H24.3154L23.537 9.26556C23.5074 9.1922 23.4575 9.1288 23.3932 9.08284C23.3288 9.03687 23.2527 9.01024 23.1737 9.00608C23.0951 8.99425 23.015 8.99973 22.9388 9.02214C22.8626 9.04455 22.7922 9.08335 22.7326 9.13582L22.2396 9.62883C21.5607 10.3117 21.1784 11.2348 21.1757 12.1977C21.1789 12.3976 21.1963 12.597 21.2276 12.7945C20.5833 12.5694 19.8887 12.5303 19.2233 12.6818C18.5578 12.8332 17.9485 13.169 17.4651 13.6508L13.6767 17.4651C13.1879 17.944 12.8454 18.552 12.6892 19.2182C12.533 19.8844 12.5695 20.5813 12.7945 21.2276L12.1977 21.1757C11.2348 21.1784 10.3117 21.5607 9.62883 22.2395L9.13582 22.7326C9.08335 22.7922 9.04455 22.8626 9.02214 22.9388C8.99973 23.015 8.99425 23.0951 9.00608 23.1737C9.01024 23.2527 9.03687 23.3288 9.08284 23.3932C9.1288 23.4575 9.1922 23.5074 9.26556 23.5369L10.6408 24.3154V25.8463C10.6408 25.984 10.6955 26.116 10.7928 26.2133C10.8901 26.3106 11.0221 26.3653 11.1598 26.3653H12.6907L13.4691 27.7405C13.4987 27.8139 13.5486 27.8773 13.6129 27.9232C13.6773 27.9692 13.7534 27.9958 13.8324 28H13.9102C14.0471 27.9981 14.1777 27.9421 14.2735 27.8443L14.7665 27.3513C15.1693 26.9438 15.4707 26.4473 15.6464 25.902C15.8222 25.3566 15.8675 24.7776 15.7785 24.2116C16.4239 24.4431 17.1225 24.4829 17.7901 24.3264C18.4577 24.1699 19.0657 23.8236 19.541 23.3294L23.3553 19.541C24.0295 18.8549 24.4109 17.9339 24.4192 16.9721C24.4178 16.5654 24.3476 16.1618 24.2116 15.7785L24.8084 15.8304C25.2855 15.8304 25.7579 15.7365 26.1986 15.554C26.6394 15.3714 27.0399 15.1039 27.3773 14.7665L27.8703 14.2735C27.9227 14.2139 27.9615 14.1435 27.9839 14.0673C28.0063 13.9911 28.0118 13.9109 28 13.8324ZM14.04 26.6507L13.4432 25.5868C13.4004 25.5074 13.3367 25.4411 13.2589 25.3954C13.1811 25.3496 13.0923 25.3261 13.0021 25.3274H11.6787V24.004C11.68 23.9138 11.6565 23.8249 11.6107 23.7471C11.5649 23.6694 11.4987 23.6056 11.4192 23.5629L10.3554 22.9661C10.846 22.4812 11.5079 22.2093 12.1977 22.2093C12.8874 22.2093 13.5494 22.4812 14.04 22.9661C14.5249 23.4567 14.7968 24.1186 14.7968 24.8084C14.7968 25.4982 14.5249 26.1601 14.04 26.6507ZM22.6028 15.1298C22.848 15.3701 23.0431 15.6567 23.1767 15.9729C23.3103 16.2892 23.3798 16.6288 23.3813 16.9721C23.3793 17.3115 23.3094 17.647 23.1757 17.959C23.042 18.2709 22.8472 18.553 22.6028 18.7885L18.7885 22.6028C18.553 22.8472 18.2709 23.042 17.959 23.1757C17.6471 23.3094 17.3115 23.3793 16.9721 23.3813C16.6288 23.3798 16.2892 23.3103 15.9729 23.1767C15.6567 23.0431 15.3701 22.848 15.1298 22.6028L14.4033 21.8763C13.92 21.3901 13.6487 20.7325 13.6487 20.0469C13.6487 19.3614 13.92 18.7038 14.4033 18.2176L18.2176 14.4033C18.4535 14.1612 18.7361 13.9696 19.0484 13.8402C19.3607 13.7109 19.696 13.6464 20.034 13.6508C20.3759 13.6459 20.7152 13.71 21.0318 13.8393C21.3483 13.9686 21.6356 14.1604 21.8763 14.4033L22.2396 14.7665L22.6028 15.1298ZM22.9661 14.04C22.4812 13.5494 22.2093 12.8874 22.2093 12.1977C22.2093 11.5079 22.4812 10.846 22.9661 10.3554L23.5629 11.4192C23.6056 11.4987 23.6694 11.5649 23.7472 11.6107C23.8249 11.6565 23.9138 11.68 24.004 11.6787H25.3274V13.0021C25.3261 13.0923 25.3496 13.1811 25.3954 13.2589C25.4411 13.3367 25.5074 13.4004 25.5868 13.4432L26.6507 14.04C26.1601 14.5249 25.4982 14.7968 24.8084 14.7968C24.1186 14.7968 23.4567 14.5249 22.9661 14.04Z" fill="white"/> </svg> ',

			'strawberry'        => '<svg width="38" height="38" viewBox="0 0 38 38" fill="none" xmlns="http://www.w3.org/2000/svg"><defs>strawberry</defs> <rect width="38" height="38" rx="19" fill="#FF3B5F"/> <path d="M16.8687 13.1847C16.9062 13.2071 16.9471 13.2227 16.9891 13.231C17.0553 13.244 17.1247 13.2392 17.1894 13.2158C17.2542 13.1925 17.3104 13.1514 17.3519 13.0996C17.3914 13.051 17.4188 12.9919 17.4297 12.9286C17.4757 12.2608 17.3023 11.6245 17.0109 11.0748C16.6747 10.4413 16.1844 9.92494 15.6928 9.607C15.5551 9.58976 15.4183 9.60936 15.2948 9.66015C15.1566 9.71684 15.034 9.81439 14.9459 9.9436C14.8573 10.0735 14.8113 10.2233 14.8087 10.3723C14.8065 10.502 14.8368 10.6333 14.8996 10.7524C15.4075 11.0753 15.8265 11.4412 16.1555 11.8496C16.4813 12.2538 16.7191 12.699 16.8687 13.1847ZM21.0867 23.8454C21.3038 23.7665 21.535 23.8116 21.733 23.9484C21.9012 24.0641 22.0474 24.2512 22.131 24.4808C22.2144 24.7099 22.2224 24.9471 22.1681 25.1441C22.0965 25.372 21.9713 25.5539 21.7386 25.6385C21.5216 25.7174 21.2903 25.6723 21.0924 25.5355C20.9093 25.4096 20.7638 25.2048 20.692 24.9957C20.6112 24.7697 20.6039 24.5356 20.6573 24.341C20.721 24.1085 20.8696 23.9243 21.0867 23.8454ZM21.3933 24.4406C21.3527 24.4125 21.3146 24.3995 21.2908 24.408C21.2669 24.4165 21.2464 24.4508 21.2336 24.4978C21.2121 24.5765 21.2166 24.6776 21.2513 24.7822L21.2636 24.8162C21.3038 24.9178 21.365 24.9972 21.432 25.0432C21.4727 25.0713 21.5107 25.0843 21.5346 25.0758C21.5568 25.0678 21.5761 25.0375 21.5891 24.9953L21.5917 24.9849C21.6146 24.9022 21.6083 24.7952 21.5683 24.6849C21.5284 24.575 21.4642 24.4893 21.3933 24.4406ZM22.0374 20.3976C22.2545 20.3187 22.4858 20.3639 22.6837 20.5004C22.8519 20.6166 22.9981 20.8034 23.0815 21.033C23.1647 21.2615 23.1727 21.4984 23.1188 21.6952C23.0551 21.9276 22.9065 22.1118 22.6894 22.1907C22.4726 22.2696 22.2413 22.2247 22.0429 22.0877C21.8747 21.9718 21.7285 21.7847 21.6451 21.5551C21.5686 21.3448 21.5464 21.0916 21.613 20.8757C21.6794 20.6518 21.8258 20.4746 22.0374 20.3976ZM22.3441 20.9927C22.3034 20.9645 22.2656 20.9516 22.2415 20.9603C22.2193 20.9683 22.2 20.9983 22.187 21.0406C22.1598 21.1493 22.1695 21.2456 22.2078 21.351C22.2477 21.4611 22.3119 21.5468 22.3826 21.5955C22.4234 21.6236 22.4612 21.6368 22.4853 21.6281C22.5092 21.6196 22.5297 21.5853 22.5425 21.5383C22.5651 21.4554 22.559 21.3477 22.5188 21.2371C22.4789 21.1268 22.4147 21.0416 22.3441 20.9927ZM18.1423 21.8151C18.3592 21.7363 18.5904 21.7811 18.7888 21.9181C18.957 22.0341 19.1032 22.2214 19.1866 22.4508C19.27 22.6797 19.278 22.9171 19.2237 23.1141C19.1547 23.3411 19.0243 23.5246 18.7943 23.6082C18.5772 23.6871 18.3459 23.642 18.148 23.5055C17.98 23.3893 17.8338 23.2022 17.7502 22.9728C17.6671 22.7444 17.659 22.5075 17.7129 22.3107C17.7767 22.0783 17.9252 21.894 18.1423 21.8151ZM18.4492 22.4104C18.4083 22.3823 18.3705 22.3691 18.3464 22.3778C18.3228 22.3863 18.3022 22.4203 18.2892 22.4676C18.2666 22.5505 18.2727 22.6582 18.3129 22.7687C18.3528 22.8788 18.417 22.9643 18.4877 23.0132C18.5283 23.0413 18.5661 23.0543 18.5902 23.0456C18.6216 23.0342 18.6398 22.9846 18.6473 22.9549C18.67 22.8724 18.6641 22.765 18.6239 22.6549C18.584 22.5448 18.5198 22.4591 18.4492 22.4104ZM22.884 16.6641C23.1004 16.5852 23.3324 16.6303 23.5303 16.7671C23.6985 16.883 23.8445 17.0701 23.9281 17.2995C24.0115 17.5286 24.0195 17.7658 23.9652 17.9628C23.8936 18.1907 23.7684 18.3726 23.5357 18.4572C23.3196 18.536 23.0874 18.4909 22.8895 18.3542C22.7213 18.2382 22.5753 18.0511 22.4917 17.8217C22.4085 17.5933 22.4005 17.3564 22.4544 17.1596C22.5181 16.927 22.6672 16.743 22.884 16.6641ZM23.1906 17.2593C23.15 17.2312 23.112 17.218 23.0881 17.2267C23.064 17.2355 23.0435 17.2695 23.0307 17.3165C23.008 17.3994 23.0142 17.5071 23.0543 17.6177C23.0943 17.7275 23.1585 17.8132 23.2291 17.8619C23.2698 17.89 23.3078 17.9032 23.3317 17.8945C23.3539 17.8865 23.3732 17.8562 23.3862 17.8135L23.3888 17.8036C23.4117 17.7209 23.4056 17.6139 23.3654 17.5036C23.3255 17.3937 23.2613 17.308 23.1906 17.2593ZM15.3056 20.0601C15.2678 20.0736 15.2549 20.1062 15.2435 20.143C15.2156 20.2498 15.2267 20.3495 15.2643 20.4529C15.3042 20.5627 15.3685 20.6485 15.4393 20.6972C15.48 20.7253 15.518 20.7383 15.5419 20.7298C15.5657 20.7212 15.5863 20.687 15.599 20.64C15.6205 20.5616 15.616 20.4607 15.5813 20.3561C15.5442 20.254 15.4922 20.1576 15.4006 20.0943C15.364 20.0691 15.3297 20.0558 15.3056 20.0601ZM15.1025 19.4965C15.3172 19.4216 15.5449 19.4672 15.7403 19.6021C15.9087 19.7183 16.0542 19.9051 16.1378 20.1345C16.2212 20.3643 16.2295 20.6005 16.1754 20.7968C16.1116 21.0293 15.963 21.2135 15.7459 21.2924C15.5289 21.3713 15.2976 21.3262 15.0997 21.1894C14.9315 21.0737 14.7853 20.8866 14.7016 20.657C14.6256 20.4482 14.6024 20.1921 14.6695 19.9779C14.7333 19.761 14.8828 19.5697 15.1025 19.4965ZM19.2003 18.6426C19.1604 18.6568 19.1465 18.6953 19.1358 18.7342C19.1063 18.8417 19.129 18.9499 19.1658 19.0519C19.206 19.154 19.2672 19.2336 19.3342 19.2799C19.3709 19.3052 19.4051 19.3184 19.4292 19.3141C19.4694 19.3 19.4835 19.2617 19.4939 19.2225C19.5199 19.1271 19.5043 19.012 19.4701 18.9208C19.4301 18.811 19.3661 18.7255 19.2955 18.6768C19.2587 18.6513 19.2242 18.6386 19.2003 18.6426ZM19.6352 18.1846C19.8038 18.3005 19.9493 18.4876 20.0327 18.7168C20.1161 18.9456 20.1241 19.1826 20.0703 19.3793C20.0065 19.6118 19.8579 19.796 19.6408 19.8749C19.4101 19.9519 19.1932 19.9094 18.9946 19.7721C18.8261 19.6559 18.6806 19.4691 18.597 19.2397C18.5136 19.0099 18.5054 18.7737 18.5595 18.5774C18.6206 18.3537 18.7666 18.1617 18.9889 18.0818C19.2197 18.0048 19.4365 18.0473 19.6352 18.1846ZM22.8115 12.3116C23.1217 12.3752 23.4103 12.4597 23.6779 12.5656C24.7567 12.9919 25.5083 13.7556 25.9319 14.8556C26.3415 15.9195 26.4357 17.2971 26.2142 18.986C25.9524 20.979 25.2518 23.4169 24.1133 26.2977C23.9685 26.6646 23.7604 26.9802 23.4946 27.2386C23.2294 27.4965 22.9081 27.6959 22.5368 27.831C22.165 27.9663 21.7913 28.0202 21.4238 27.9933C21.0548 27.9661 20.6927 27.8577 20.3452 27.6697C17.6175 26.1929 15.5135 24.7758 14.0341 23.419C12.7777 22.267 11.9642 21.1514 11.5943 20.0733C11.2123 18.9598 11.2976 17.8926 11.8503 16.8717C11.9883 16.6168 12.1558 16.3655 12.3516 16.1179C11.9795 15.9814 11.5881 15.7891 11.1691 15.5369C11.0517 15.4802 10.9811 15.3505 11.0045 15.2156C11.0328 15.0526 11.1882 14.9435 11.3512 14.9718C12.0795 15.0992 12.6272 15.0519 13.0779 14.8873C13.5286 14.7229 13.8988 14.4335 14.2635 14.0816C14.8138 13.5496 15.2102 13.2347 15.5452 13.0498C15.7415 12.9414 15.9165 12.8764 16.0868 12.8393C15.9781 12.6234 15.8454 12.4186 15.6893 12.2247C15.3656 11.8234 14.5139 11.2685 14.4243 11.1256C14.2776 10.8932 14.2065 10.6272 14.211 10.3629C14.2158 10.0976 14.2963 9.83353 14.4513 9.60629C14.6062 9.37882 14.8228 9.20757 15.0675 9.10694C15.2806 9.01907 15.749 8.93876 15.9569 9.06655C16.5481 9.43244 17.1412 10.0438 17.5404 10.7961C17.7566 11.204 17.917 11.6542 17.9909 12.1309C18.1241 12.0761 18.2748 12.0331 18.4534 12.0022C18.8214 11.9386 19.313 11.9292 20.0365 11.98C20.5429 12.0154 21.0128 11.9989 21.4637 11.8354C21.9111 11.6734 22.3561 11.3625 22.825 10.8098C22.8996 10.7047 23.0371 10.6565 23.1654 10.6997C23.322 10.7526 23.4058 10.9229 23.3529 11.0795C23.1918 11.5534 23.0128 11.9611 22.8115 12.3116ZM16.3107 13.4074C16.1612 13.4289 16.0096 13.4773 15.8333 13.5747C15.5449 13.7339 15.1885 14.0197 14.6783 14.5129C14.2602 14.9163 13.8286 15.2506 13.282 15.4499C13.0964 15.518 12.8986 15.5695 12.6858 15.6025C12.769 15.6299 12.8507 15.6543 12.9315 15.6753L12.9629 15.6833C13.2031 15.7436 13.4337 15.7747 13.6574 15.7785C14.2987 15.7894 14.9043 15.574 15.5558 15.1726C15.6961 15.0859 15.8806 15.1296 15.9673 15.2699C16.0533 15.4093 16.011 15.5921 15.873 15.6795C15.4332 15.9833 15.3531 16.5523 15.3501 17.2126C15.826 16.8462 16.2406 16.4739 16.5852 16.1C17.0132 15.6354 17.3326 15.1686 17.5256 14.7073C17.5983 14.5318 17.7868 14.4817 17.9507 14.562C18.3941 14.7885 18.9348 14.9395 19.5551 15.0193C20.0575 15.084 20.6145 15.1022 21.2166 15.077C20.7877 14.5705 20.3582 14.185 19.822 14.2401C19.6578 14.2568 19.5107 14.1371 19.4939 13.9729C19.4776 13.813 19.5908 13.6694 19.7481 13.6465C20.5059 13.5347 21.1084 13.3108 21.5922 12.8903C21.8244 12.6886 22.0332 12.4387 22.2224 12.1314C22.0382 12.2434 21.8537 12.3308 21.6678 12.3981C21.1207 12.5963 20.5758 12.618 19.9968 12.5776C19.3227 12.5304 18.875 12.5374 18.554 12.5927C18.332 12.6312 18.1728 12.6938 18.0355 12.7791C18.0344 12.8448 18.0315 12.9107 18.0268 12.9768C18.0039 13.161 17.9352 13.3299 17.8182 13.4747C17.7089 13.6103 17.5629 13.7169 17.3935 13.7785C17.2239 13.8402 17.0442 13.8527 16.8743 13.8191C16.7038 13.7858 16.5413 13.7062 16.4078 13.5846C16.3485 13.5305 16.3315 13.4804 16.3107 13.4074ZM12.9785 16.2996C12.734 16.5816 12.5335 16.8672 12.3776 17.1552C11.9082 18.0221 11.8362 18.9301 12.1617 19.8787C12.499 20.8618 13.2586 21.8957 14.4395 22.9783C15.8825 24.3018 17.946 25.69 20.6287 27.1424C20.901 27.2898 21.1824 27.3749 21.4658 27.3957C21.7505 27.4164 22.0417 27.3742 22.3327 27.2683C22.6237 27.1623 22.8736 27.008 23.0775 26.8096C23.2804 26.6121 23.4413 26.3665 23.5553 26.0776C24.6778 23.2379 25.3669 20.8474 25.6213 18.9086C25.8298 17.3188 25.7474 16.0386 25.3739 15.0687C25.0144 14.1347 24.3755 13.4861 23.4578 13.1235C23.151 13.0023 22.8141 12.9121 22.4477 12.853C22.3034 13.0363 22.1489 13.1986 21.9834 13.3425C21.6699 13.615 21.321 13.8177 20.9294 13.9665C21.3298 14.2488 21.6727 14.6797 22.0315 15.1308C22.0774 15.1797 22.1081 15.2435 22.114 15.3151C22.1274 15.4799 22.0046 15.625 21.8397 15.6384C20.9802 15.7098 20.1867 15.7034 19.48 15.6122C18.9083 15.5385 18.3939 15.4088 17.9472 15.2199C17.7263 15.6524 17.4157 16.0823 17.026 16.5053C16.5448 17.0276 15.9401 17.5428 15.2329 18.0428C15.0399 18.1765 14.7673 18.0466 14.7619 17.805C14.7491 17.2782 14.7217 16.6879 14.8573 16.1758C14.4612 16.314 14.0634 16.3832 13.6479 16.3761C13.4292 16.3723 13.2067 16.3471 12.9785 16.2996Z" fill="white"/> </svg> ',
		);

		/**
		 * Social Icons – svg sources.
		 *
		 * @var array
		 */
		public static $recipe_category_icons = array();

		/**
		 * Store the code for all SVGs in an array.
		 *
		 * @var array
		 */
		public static $dashboard_svg_icons = array(
			'wishlist'     => '<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
			<g><path d="M9.12502 11.6667L13.25 7.54167L12.0625 6.375L9.12502 9.3125L7.93752 8.125L6.75002 9.3125L9.12502 11.6667ZM4.16669 17.5V4.16667C4.16669 3.70833 4.32988 3.31597 4.65627 2.98958C4.98266 2.66319 5.37502 2.5 5.83335 2.5H14.1667C14.625 2.5 15.0174 2.66319 15.3438 2.98958C15.6702 3.31597 15.8334 3.70833 15.8334 4.16667V17.5L10 15L4.16669 17.5Z" fill="#566267"/></g></svg>',

			'edit-profile' => '<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
			<g><path d="M11.6666 17.5V14.9375L16.2708 10.3541C16.3958 10.2291 16.5347 10.1389 16.6875 10.0833C16.8403 10.0278 16.993 9.99998 17.1458 9.99998C17.3125 9.99998 17.4722 10.0312 17.625 10.0937C17.7778 10.1562 17.9166 10.25 18.0416 10.375L18.8125 11.1458C18.9236 11.2708 19.0104 11.4097 19.0729 11.5625C19.1354 11.7153 19.1666 11.868 19.1666 12.0208C19.1666 12.1736 19.1389 12.3298 19.0833 12.4896C19.0278 12.6493 18.9375 12.7916 18.8125 12.9166L14.2291 17.5H11.6666ZM3.33331 16.6666V14.3333C3.33331 13.8611 3.45484 13.4271 3.6979 13.0312C3.94095 12.6354 4.26387 12.3333 4.66665 12.125C5.52776 11.6944 6.40276 11.3715 7.29165 11.1562C8.18054 10.941 9.08331 10.8333 9.99998 10.8333C10.5139 10.8333 11.0208 10.8646 11.5208 10.9271C12.0208 10.9896 12.5208 11.0903 13.0208 11.2291L9.99998 14.25V16.6666H3.33331ZM17.1458 12.8333L17.9166 12.0208L17.1458 11.25L16.3541 12.0416L17.1458 12.8333ZM9.99998 9.99998C9.08331 9.99998 8.29859 9.67359 7.64581 9.02081C6.99304 8.36804 6.66665 7.58331 6.66665 6.66665C6.66665 5.74998 6.99304 4.96526 7.64581 4.31248C8.29859 3.6597 9.08331 3.33331 9.99998 3.33331C10.9166 3.33331 11.7014 3.6597 12.3541 4.31248C13.0069 4.96526 13.3333 5.74998 13.3333 6.66665C13.3333 7.58331 13.0069 8.36804 12.3541 9.02081C11.7014 9.67359 10.9166 9.99998 9.99998 9.99998Z" fill="#566267"/></g></svg>',

			'browse'       => '<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
			<g><path d="M17.9824 15.211L17.2788 15.0854C17.2866 14.9625 17.2918 14.8291 17.2918 14.6826C17.2944 14.3451 17.2709 13.9528 17.2055 13.5368C17.1715 13.3302 17.127 13.1183 17.0669 12.9064C17.0067 12.6945 16.9361 12.4826 16.8497 12.276C16.7634 12.0693 16.664 11.8705 16.5515 11.6822C16.439 11.4938 16.3161 11.3185 16.1853 11.1564C16.0545 10.9968 15.9185 10.8503 15.7825 10.7221C15.6464 10.5939 15.5078 10.4788 15.377 10.3847C15.2462 10.2879 15.1206 10.2042 15.0082 10.1388C14.9794 10.1205 14.9532 10.1048 14.9271 10.0891C14.9009 10.0734 14.8747 10.0603 14.8486 10.0446C14.7989 10.0158 14.7544 9.99229 14.7099 9.97136C14.6681 9.95043 14.6315 9.93212 14.5975 9.91381C14.5634 9.89812 14.5373 9.88504 14.5137 9.87457C14.4719 9.85365 14.4457 9.84318 14.4431 9.84318C14.4457 9.8458 14.4693 9.85888 14.5085 9.88242C14.5294 9.8955 14.5556 9.9112 14.587 9.93212C14.6184 9.95043 14.6524 9.97398 14.6916 10.0001C14.7309 10.0263 14.7727 10.0551 14.8198 10.0891C14.8434 10.1048 14.8669 10.1231 14.8904 10.1414C14.914 10.1597 14.9401 10.178 14.9663 10.1989C15.0683 10.2774 15.1808 10.3716 15.2985 10.4815C15.4162 10.5887 15.5339 10.7143 15.649 10.8503C15.7668 10.9889 15.8792 11.1407 15.9839 11.3055C16.0885 11.4703 16.1853 11.6482 16.269 11.8339C16.3527 12.0196 16.4234 12.2132 16.4809 12.412C16.5385 12.6082 16.5829 12.8096 16.6143 13.011C16.6457 13.2098 16.664 13.4087 16.6745 13.5996C16.6928 13.9842 16.6719 14.3425 16.6352 14.646C16.6222 14.7585 16.6065 14.8657 16.5882 14.9625L1.9679 12.3283C2.66634 8.44891 6.81777 5.94811 11.2413 6.74334C15.6621 7.54119 18.6809 11.3316 17.9824 15.211Z" fill="#566267"/><path d="M11.461 4.39167C10.9064 4.29227 10.3754 4.66111 10.276 5.21568C10.1766 5.77025 10.5454 6.30128 11.1 6.40068C11.6546 6.50008 12.1856 6.13124 12.285 5.57667C12.3844 5.0221 12.0156 4.49107 11.461 4.39167ZM12.0051 5.10058C11.9633 5.33863 11.7357 5.49558 11.4976 5.45372C11.2596 5.41187 11.1026 5.18429 11.1445 4.94624C11.1863 4.70819 11.4139 4.55124 11.652 4.59309C11.89 4.63495 12.047 4.86253 12.0051 5.10058Z" fill="#566267"/><path d="M18.7123 16.3411H1.28775C1.12818 16.3411 1 16.2103 1 16.0533C1 15.8937 1.13079 15.7656 1.28775 15.7656H18.7123C18.8718 15.7656 19 15.8964 19 16.0533C19 16.2129 18.8718 16.3411 18.7123 16.3411Z" fill="#566267"/><path d="M1.94957 9.02442C2.09606 9.41419 2.09606 9.76472 1.9888 10.0185C1.89202 10.2722 1.68798 10.4318 1.69844 10.4318C1.68536 10.4318 1.76645 10.2748 1.73768 10.0185C1.71937 9.76472 1.5938 9.41157 1.44731 9.02442C1.30082 8.63465 1.30082 8.28412 1.40808 8.03038C1.50486 7.77663 1.7089 7.61707 1.69844 7.61707C1.71152 7.61707 1.63043 7.77402 1.6592 8.03038C1.67751 8.28412 1.80569 8.63465 1.94957 9.02442Z" fill="#566267"/><path d="M2.49888 7.61182C2.64537 8.00159 2.64537 8.35212 2.53812 8.60586C2.44133 8.85961 2.23729 9.01918 2.24776 9.01918C2.23468 9.01918 2.31577 8.86222 2.28699 8.60586C2.26868 8.35212 2.14312 7.99897 1.99663 7.61182C1.85014 7.22205 1.85014 6.87152 1.95739 6.61778C2.05418 6.36404 2.25822 6.20447 2.24776 6.20447C2.26084 6.20447 2.17974 6.36142 2.20852 6.61778C2.22683 6.87152 2.35501 7.22205 2.49888 7.61182Z" fill="#566267"/><path d="M5.71123 4.40735C5.56474 4.79712 5.56474 5.14765 5.67199 5.4014C5.76878 5.65514 5.97282 5.81471 5.96236 5.81471C5.97544 5.81471 5.89434 5.65775 5.92312 5.4014C5.94143 5.14765 6.06699 4.79451 6.21348 4.40735C6.35997 4.01758 6.35997 3.66705 6.25272 3.41331C6.15593 3.15957 5.95189 3 5.96236 3C5.94928 3 6.03037 3.15695 6.00159 3.41331C5.98328 3.66705 5.85772 4.01758 5.71123 4.40735Z" fill="#566267"/></g></svg>',
		);
	}
}
